// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @author Phil Bradley


#ifndef INCLUDED_core_conformation_Residue_HH
#define INCLUDED_core_conformation_Residue_HH


// Unit headers
#include <core/conformation/Residue.fwd.hh>

// Package headers
#include <core/conformation/Conformation.fwd.hh>
#include <core/conformation/Atom.hh>
#include <core/conformation/PseudoBond.fwd.hh>
#include <core/chemical/AtomType.fwd.hh>

// Project headers
#include <core/chemical/AA.hh>
#include <core/chemical/ResConnID.hh>
#include <core/chemical/ResidueType.hh> // also defines AtomIndices == vector1< Size >
#include <core/chemical/ResidueTypeSet.fwd.hh>
// AUTO-REMOVED #include <core/chemical/AtomTypeSet.hh>
#include <core/types.hh>

// Utility headers
// AUTO-REMOVED #include <utility/vector1.hh>
#include <utility/pointer/access_ptr.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/pointer/ReferenceCount.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.fwd.hh>

// C++ headers
#include <map>
#include <iosfwd>
#include <limits>

namespace core {
namespace conformation {

///@brief  Instance Residue class, used for placed residues and rotamers
/**
	 This class is designed to be lightweight. It holds a const-reference ("rsd_type_") to a ResidueType object for
	 access to information common to all instances of a single type, eg Alanine or Thymine. Residue stores any data
	 unique to a placed residue or rotamer, currently:

		 - a vector1 of Atoms which hold the positions (and also the atom-types for fast access during scoring);

		 - the sequence position and chain, both integers

		 - the backbone and sidechain torsion angles (of course backbone torsions are not unique to a rotamer, and
			 the chi angles are derivable from the coordinates, but storing them in the residue is convenient for scoring
			 purposes).

		 - the coordinates of an interaction center or centroid, used eg in the knowledge-based fullatom pair term
			 ("actcoord_"). Maybe this will also hold the centroid position for centroid-mode scoring??

**/

class Residue : public utility::pointer::ReferenceCount {

public:
	typedef chemical::AtomType AtomType;
	typedef chemical::ResidueType ResidueType;
	typedef chemical::AtomIndices AtomIndices;

public:

	///@brief constructors
	///Residue( Residue const & ); // user defined copy ctor to avoid #including PseudoBond.hh
	///Residue const & operator = ( Residue const & ) // user defined assignment operator for same reason
	Residue( ResidueType const & rsd_type_in, bool const dummy_arg );

	///@brief  Rotamer-style constructor; orients ideal coords onto backbone of current_rsd
	Residue(
		ResidueType const & rsd_type_in,
		Residue const & current_rsd,
		Conformation const & conformation,
		bool preserve_c_beta = false
	);

	Residue( Residue const & src );

	~Residue();

	///@brief make a copy of this residue( allocate actual memory for it )
	ResidueOP
	clone() const;

	///@brief create a rotamer of this residue
	// Temporary hack until Residue hierarchy is worked out
	ResidueOP
	create_rotamer() const
	{
		return clone();
	}

	///@brief create a copy of residue, same as clone()
	// Temporary hack until Residue hierarchy is worked out
	ResidueOP
	create_residue() const
	{
		return clone();
	}

	/// @brief get ResidueType of this residue
	ResidueType const &
	type() const
	{
		return rsd_type_;
	}

	/// @brief access to the ResidueTypeSet containing our ResidueType
	chemical::ResidueTypeSet const &
	residue_type_set() const
	{
		return rsd_type_.residue_type_set();
	}


	/// @brief get AtomType for an atom in thie residue by its index number
	AtomType const &
	atom_type( int const atomno ) const
	{
		return rsd_type_.atom_type( atomno );
	}

	/// @brief get AtomType for an atom in thie residue by its index number
	chemical::AtomTypeSet const &
	atom_type_set() const
	{
		return rsd_type_.atom_type_set();
	}

	///@brief get atom_type_index for an atom in this residue by its index number
	/// atom_type_index is used to query this atom's AtomType from an AtomTypeSet,
	/// for example, AtomTypeSet[atom_type_index] = AtomType
	Size
	atom_type_index( Size const atomno ) const
	{
		return atoms_[ atomno ].type();
	}

	///@brief get atom charge for an atom in this residue by its index number
	Real
	atomic_charge( int const atomno ) const
	{
		return rsd_type_.atomic_charge( atomno );
	}


	///@brief get index number of an atom in this residue by its name.
	/// for example: atom_index("CA") = 2 for normal amino acid
	Size
	atom_index( std::string const & atm ) const
	{
		return rsd_type_.atom_index( atm );
	}

	// NOTE: AtomIndices == vector1< Size >

	///@brief The indices of the polar hydrogens.
	AtomIndices const &
	Hpos_polar() const
	{
		return rsd_type_.Hpos_polar();
	}

	/// @brief the indices of the orbitals
	AtomIndices const &
	Orbitals_index() const
	{
		return rsd_type_.Orbitals_index();
	}

	/// @brief
	AtomIndices const &
	atoms_with_orb_index() const
	{
		return rsd_type_.atoms_with_orb_index();
	}

	AtomIndices const &
	sc_orbitals() const
	{
		return rsd_type_.sc_lp_index();
	}

	AtomIndices const &
	bb_orbitals() const
	{
		return rsd_type_.bb_lp_index();
	}

	/// @brief all backbone atoms, heavy plus hydrogen
	AtomIndices const &
	all_bb_atoms() const {
		return rsd_type_.all_bb_atoms();
	}

	/// @brief the indices of all aromatic hydrogens
	AtomIndices const &
	Haro_index() const{
		return rsd_type_.Haro_index();
	}

	/// @brief the indices of all polar hydrogens
	AtomIndices const &
	Hpol_index() const{
		return rsd_type_.Hpol_index();
	}

	///@brief The indices of the apolar hydrogens, for carbon h-bond calculations.
	AtomIndices const &
	Hpos_apolar() const
	{
		return rsd_type_.Hpos_apolar();
	}

	///@brief The indices of the polar sidechain hydrogens.
	AtomIndices const &
	Hpos_polar_sc() const
	{
		return rsd_type_.Hpos_polar_sc();
	}




	///@brief the indices of the hbond acceptor atoms
	AtomIndices const &
	accpt_pos() const
	{
		return rsd_type_.accpt_pos();
	}

	///@brief the indices of the sidechain hbond acceptor atoms
	AtomIndices const &
	accpt_pos_sc() const
	{
		return rsd_type_.accpt_pos_sc();
	}

	///@brief our atoms, const -- a vector1 of Atom objects
	Atoms const &
	atoms() const
	{
		return atoms_;
	}

	///@brief our atoms, non-const
	Atoms &
	atoms()
	{
		return atoms_;
	}

	///@brief begin interator, to iterate over atoms
	Atoms::iterator atom_begin() { return atoms_.begin(); }
	///@brief end interator, to iterate over atoms
	Atoms::iterator atom_end  () { return atoms_.end  (); }

	Atoms::const_iterator atom_begin() const { return atoms_.begin(); }
	Atoms::const_iterator atom_end  () const { return atoms_.end  (); }

	///@brief should be safe, given the atom ordering rules?
	Atoms::const_iterator sidechainAtoms_begin() const
	{
		return atoms_.begin() + first_sidechain_atom() - 1;
	}
	Atoms::const_iterator heavyAtoms_end() const
	{
		return atoms_.begin() + nheavyatoms() - 1;
	}

	///@brief const, get Atom object (xyz and atom_type) of an atom by its index number in this residue
	Atom const &
	atom( Size const atm_index ) const
	{
		return atoms_[ atm_index ];
	}

	///@brief non-const, get Atom object (xyz and atom_type) of an atom by its index number in this residue
	Atom &
	atom( Size const atm_index )
	{
		return atoms_[ atm_index ];
	}


	///@brief const, get Atom object (xyz and atom_type) of an atom by its atom name
	///
	///@note slower but safer than hard-coding an integer index in code where you need a specific atom
	Atom const &
	atom( std::string const & atm_name ) const
	{
		return atoms_[ atom_index( atm_name ) ];
	}

	///@brief non-const, get Atom object (xyz and atom_type) of an atom by its atom name
	///
	///@note slower but safer than hard-coding an integer index in code where you need a specific atom
	Atom &
	atom( std::string const & atm_name )
	{
		return atoms_[ atom_index( atm_name ) ];
	}

	///@brief get an atom's position in this residue by atom's index number
	Vector const &
	xyz( Size const atm_index ) const
	{
		return atoms_[ atm_index ].xyz();
	}

	///@brief get an atom's position in this residue by atom's name
	Vector const &
	xyz( std::string const & atm_name ) const
	{
		return atom( atm_name ).xyz();
	}

	///@brief set an atom's position in this residue by atom's index number
	void
	set_xyz( core::Size const atm_index, Vector const & xyz_in )
	{
		atoms_[ atm_index ].xyz( xyz_in );
	}

	/// set an atom's position in this residue by atom's name
	void
	set_xyz( std::string const & atm_name, Vector const & xyz_in )
	{
		atom( atm_name ).xyz( xyz_in );
	}


	///@brief number of atoms in this residue
	Size
	natoms() const
	{
		return rsd_type_.natoms();
	}

	///@brief number of heavyatoms
	Size
	nheavyatoms() const
	{
		return rsd_type_.nheavyatoms();
	}

	///@brief The index of the last backbone heavyatom
	///
	///@note  The heavyatoms come first in atom ordering,
	///first backbone then sidechain. The hydrogens follow the order
	/// of their attached heavyatom.
	Size
	last_backbone_atom() const
	{
		return rsd_type_.last_backbone_atom();
	}

	///@brief the index of the first sidechain heavy atom
	Size
	first_sidechain_atom() const
	{
		return rsd_type_.first_sidechain_atom();
	}

	///@brief the index of the first sidechain hydrogen atom
	Size
	first_sidechain_hydrogen() const
	{
		return rsd_type_.first_sidechain_hydrogen();
	}

	///@brief (polymers only) The atom-index of the atom in this residue which connects to the residue before us in sequence
	///for example, for amino acid, residue.lower_connect_atom() = atom_index("N")
	Size
	lower_connect_atom() const
	{
		return rsd_type_.lower_connect_atom();
	}

	///@brief (polymers only) The atom-index of the atom in this residue which connects to the residue after us in sequence
	/// for example, for amino acid, residue.upper_connect_atom() = atom_index("C")
	Size
	upper_connect_atom() const
	{
		return rsd_type_.upper_connect_atom();
	}

	///@brief get upper_connection for this residue
	/// a ResidueConnection has internal coords info
	/// on how to build the atom in the next residue which
	/// connects to this residue
	chemical::ResidueConnection const &
	upper_connect() const
	{
		return rsd_type_.upper_connect();
	}

	/// @brief get lower_connection for this residue
	/// a ResidueConnection has internal coords info
	/// on how to build the atom in the previous residue which
	/// connects to this residue
	chemical::ResidueConnection const &
	lower_connect() const
	{
		return rsd_type_.lower_connect();
	}

	bool connections_match( Residue const & other ) const;

	/// @brief number of ResidueConnections, counting polymeric residue connections
	Size
	n_residue_connections() const
	{
		return rsd_type_.n_residue_connections();
	}

	Size
	n_polymeric_residue_connections() const {
		return rsd_type_.n_polymeric_residue_connections();
	}

	Size
	n_non_polymeric_residue_connections() const {
		return rsd_type_.n_non_polymeric_residue_connections();
	}


	/// @brief get a connection for this residue
	/// a ResidueConnection has internal coords info
	/// on how to build the atom in a different residue which
	/// connects to this residue
	chemical::ResidueConnection const &
	residue_connection( int const resconn_index ) const
	{
		return rsd_type_.residue_connection( resconn_index );
	}

	Size
	residue_connect_atom_index( Size const resconn_id ) const {
		return rsd_type_.residue_connect_atom_index( resconn_id );
	}


	Size
	connected_residue_at_resconn( Size const resconn_index ) const {
		return connect_map_[ resconn_index ].resid();
	}

	chemical::ResConnID
	connect_map( Size resconn_index ) const {
		return connect_map_[ resconn_index ];
	}

	void
	clear_residue_connections();

	void
	copy_residue_connections_from( Residue const & src );

	bool
	has_incomplete_connection() const;

	/// @brief check atom for complete connectivity
	bool
	has_incomplete_connection(
		core::Size const atomno
	) const;

	bool
	connection_incomplete( Size resconnid ) const;

	chemical::ResConnID
	actual_residue_connection( Size resconnid ) const {
		return connect_map_[ resconnid ];
	}

	/// @brief get the residue number of a residue connected to this residue
	Size
	residue_connection_partner( Size const resconn_index ) const
	{
		return connect_map_[ resconn_index ].resid();
	}

	/// attempt to take residue connection info from src_rsd
	void
	copy_residue_connections( Residue const & src_rsd );


	/// @brief get the residue number of a residue connected to this residue
	Size
	residue_connection_conn_id( Size const resconn_index ) const
	{
		return connect_map_[ resconn_index ].connid();
	}


	/// @brief set a connection to this residue by adding its parnter's residue number
	void
	residue_connection_partner(
		Size const resconn_index, // ie, our connid
		Size const otherres,
		Size const other_connid
	);

	/// @brief Distance between a potential residue connection match and the position of the expected atom
	Distance
	connection_distance(
		conformation::Conformation const & conf,
		Size const resconn_index,
		Vector const matchpoint
	) const;

	/// @brief Am I bonded to other?
	/// Meaningful for arbitrary topologies (e.g. circular peptides, disulfides)
	bool
	is_bonded( Residue const & other ) const;

	/// @brief Do I have any pseudobonds to other?
	bool
	is_pseudo_bonded( Residue const & other ) const
	{
		return is_pseudo_bonded( other.seqpos() );
	}

	/// @brief Am I bonded to other?
	/// Looks at all residue connections as opposed to doing arithmetic
	bool
	is_bonded( Size const other_index ) const;

	/// @brief Do I have any pseudobonds to other?
	bool
	is_pseudo_bonded( Size const other_index ) const
	{
		return pseudobonds_.find( other_index ) != pseudobonds_.end();
	}

	/// @brief Am I polymer bonded to other?
  bool
  is_polymer_bonded( Residue const & other ) const;

 /// @brief Am I polymer-bonded to other? checks lower and upper connections
  bool
  is_polymer_bonded( Size const other_index ) const;

	/// @brief Returns the atom-index of my atom which is connected to the other residue

	Size
	connect_atom( Residue const & other ) const;

	/// @brief Returns the vector1 of resconn ids that connect this residue to other
	utility::vector1< Size > const &
	connections_to_residue( Residue const & other ) const
	{
		return connections_to_residue( other.seqpos() );
	}

	/// @brief Returns the vector1 of resconn ids that connect this residue to other
	utility::vector1< Size > const &
	connections_to_residue( Size const other_resid ) const
	{
		assert( connections_to_residues_.find( other_resid ) != connections_to_residues_.end() );
		return connections_to_residues_.find( other_resid )->second;
	}

	PseudoBondCollectionCOP
	get_pseudobonds_to_residue( Size resid ) const;

	std::map< Size, PseudoBondCollectionCOP > const &
	pseudobonds() const {
		return pseudobonds_;
	}


	void
	set_pseudobonds_to_residue( Size resid, PseudoBondCollectionCOP pbs );

	/// @brief number of bonds separating atom at1 and at2
	int
	path_distance( int at1, int at2 ) const
	{
		return rsd_type_.path_distance( at1, at2 );
	}

	/// @brief shortest path distance for an atom to all other residue atoms
	utility::vector1< int > const &
	path_distance( int atom ) const
	{
		return rsd_type_.path_distance( atom );
	}

	/// @brief shortest path distance for any pair atom in this residue
	/// for example path_distances()[atom1][atom2]
	utility::vector1< utility::vector1< int > > const &
	path_distances() const
	{
		return rsd_type_.path_distances();
	}

	/// @brief this atom (by its atomno) is a backbone atom
	bool
	atom_is_backbone( int const atomno ) const
	{
		return rsd_type_.atom_is_backbone( atomno );
	}

	/// @brief quick lookup: is the atom with the given index a hydrogen or not?
	bool
	atom_is_hydrogen( Size const atomno ) const
	{
		return rsd_type_.atom_is_hydrogen( atomno );
	}

	/// @brief First atom-index of the hydrogens attached to atom number "atom"
	Size
	attached_H_begin( int const atom ) const
	{
		return rsd_type_.attached_H_begin( atom );
	}

	/// @brief Last atom-index of the hydrogens attached to atom number "atom"
	Size
	attached_H_end( int const atom ) const
	{
		return rsd_type_.attached_H_end( atom );
	}

	/// @brief For each heavy atom, the atom-index of the first hydrogens attached to it
	AtomIndices const &
	attached_H_begin() const
	{
		return rsd_type_.attached_H_begin();
	}

	/// @brief For each heavy atom, the atom-index of last hydrogens attached to it
	AtomIndices const &
	attached_H_end() const
	{
		return rsd_type_.attached_H_end();
	}

	/// @brief the atom index of this atom's base atom
	Size
	atom_base( int const atomno ) const
	{
		return rsd_type_.atom_base( atomno );
	}

	/// @brief the atom index of this atom's second base atom
	///
	/// abase2 is this atom's first bonded neighbor other than
	/// this atom's base atom (unless it has only one neighbor)
	Size
	abase2( int const atomno ) const
	{
		return rsd_type_.abase2( atomno );
	}

	/// @brief number of chi angles of this residue
	Size
	nchi() const
	{
		return rsd_type_.nchi();
	}

	/// @brief chi rotamers available for this chi angle of this residue
	utility::vector1< std::pair< Real, Real > > const &
	chi_rotamers( Size const chino ) const
	{
		return rsd_type_.chi_rotamers( chino );
	}

	/// @brief atom indices for all bonded neighbor of this atom in this residue
	AtomIndices const &
	bonded_neighbor( int const atm ) const
	{
		return rsd_type_.bonded_neighbor( atm );
	}

	/// @brief atom indices for bonded neighbors to which atom-tree connections are disallowed.
	AtomIndices const &
	cut_bond_neighbor( int const atm ) const
	{
		return rsd_type_.cut_bond_neighbor( atm );
	}


	/// @brief count the number of atoms bonded to a given atom in all residues
	core::Size
	n_bonded_neighbor_all_res(
		core::Size const atomno,
		bool virt = false
	) const;


	/// @brief Convenience synonym for bonded_neighbor
	AtomIndices const &
	nbrs( int const atm ) const
	{
		return rsd_type_.bonded_neighbor( atm );
	}


	/// @brief const accessor of mainchain torsion angles of this residue
	utility::vector1< Real > const &
	mainchain_torsions() const
	{
		return mainchain_torsions_;
	}

	/// @brief accessor of mainchain torsion angles of this residue
	utility::vector1< Real > &
	mainchain_torsions()
	{
		return mainchain_torsions_;
	}

	/// @brief setter of mainchain torsion angles of this residue
	void
	mainchain_torsions( utility::vector1< Real > const & torsions ) {
		mainchain_torsions_ = torsions;
	}

	/// @brief access chi torsion angles of this residue
	utility::vector1< Real > const &
	chi() const
	{
		return chi_;
	}

	/// @brief access chi torsion angles of this residue
	utility::vector1< Real > &
	chi()
	{
		return chi_;
	}

	/// @brief setter for chi torsion angles of this residue
	void
	chi( utility::vector1< Real > const & chis ) {
		chi_ = chis;
	}


	/// @brief atom indices of four atoms defining each chi angle
	utility::vector1< AtomIndices > const &
	chi_atoms() const
	{
		return rsd_type_.chi_atoms();
	}

	/// @brief atom indices of four atoms defining this chi angle
	AtomIndices const &
	chi_atoms( int const chino ) const
	{
		return rsd_type_.chi_atoms( chino );
	}

	/// @brief get a specific main chain torsion angle for this residue
	///
	/// for example, mainchain_torsion(2) will be psi angle
	Real
	mainchain_torsion( Size const torsion ) const
	{
		return mainchain_torsions_[ torsion ];
	}

	/// @brief index numbers of all mainchain atoms
	AtomIndices const &
	mainchain_atoms() const
	{
		return rsd_type_.mainchain_atoms();
	}

	/// @brief number of mainchain atoms
	Size
	mainchain_atom( Size const atm ) const
	{
		return rsd_type_.mainchain_atom( atm );
	}

	///
	Size
	n_mainchain_atoms() const
	{
		return rsd_type_.mainchain_atoms().size();
	}

	/// @brief get a specific chi torsion angle
	Real
	chi( Size const chino ) const
	{
		return chi_[ chino ];
	}

	/// @brief sequence position number of this residue in a certain Conformation
	Size
	seqpos() const
	{
		return seqpos_;
	}

	/// @brief absolute value of separation of sequence positions between two residues
	Size
	polymeric_sequence_distance( Residue const & other ) const
	{
		if ( ! is_polymer() ||  ! other.is_polymer() ) return Size( -1 );
		return ( chain_ == other.chain() ?
			( seqpos_ <= other.seqpos_ ? other.seqpos_ - seqpos_ : seqpos_ - other.seqpos_ ) : Size( -1 ) );
	}

	/// @brief separation of sequence positions between two residues
	/// positive if the other residue is "farther" along the sequence
	int
	polymeric_oriented_sequence_distance( Residue const & other ) const
	{
		if ( ! is_polymer() || ! other.is_polymer() ) return std::numeric_limits<int>::max();
		return ( chain_ == other.chain() ? other.seqpos_ - seqpos_ : std::numeric_limits<int>::max() );
	}

	/// @brief set sequence position numbe for this residue
	void
	seqpos( Size const setting )
	{
		seqpos_ = setting;
	}

	/// @brief get chain id of this residue
	core::Size
	chain() const
	{
		return chain_;
	}

	/// @brief set chain id of this residue
	void
	chain( int const setting )
	{
		chain_ = setting;
	}

	/// @brief does this residue require an actcoord?
	bool
	requires_actcoord() const
	{
		return rsd_type_.requires_actcoord();
	}

	/// @brief index number of atoms to be used to define this residue actcoord.
	AtomIndices const &
	actcoord_atoms() const
	{
		return rsd_type_.actcoord_atoms();
	}

	/// @brief update actcoord for this residue
	void
	update_actcoord();



	///@brief coordinate used for pairE calculations (amino acids only)
	Vector const &
	actcoord() const
	{
		return actcoord_;
	}

	///@brief coordinate used for pairE calculations (amino acids only)
	Vector &
	actcoord()
	{
		return actcoord_;
	}

	///@brief update sequence numbers for this residue and the numbers stored about its non-polymer connections.
	///called by our owning conformation when the sequence numbers are remapped
	void
	update_sequence_numbering( utility::vector1< Size > const & old2new );


	///@brief build coord for an atom from ideal internal coords, used for building missing atoms
	Vector
	build_atom_ideal(
		int const atomno,
		Conformation const & conformation // necessary for context, eg the C of the preceding residue for HN
	) const
	{
		return icoor( atomno ).build( *this, conformation );
	}

	///@brief index number for the atom which is used as center for neighbor definition
	///
	/// for example, C-beta atom for some amino acid
	Size
	nbr_atom() const
	{
		return rsd_type_.nbr_atom();
	}

	///@brief distance cutoff which is used as radius for neighbor definition
	Real
	nbr_radius() const
	{
		return rsd_type_.nbr_radius();
	}

	///
	Vector const &
	nbr_atom_xyz() const
	{
		return atoms_[ rsd_type_.nbr_atom() ].xyz();
	}

	/////////////
	// properties

	///@brief residue is a polymer type
	bool
	is_polymer() const
	{
		return rsd_type_.is_polymer();
	}

	///@brief residue is amino acid
	bool
	is_protein() const
	{
		return rsd_type_.is_protein();
	}

	///@brief residue is DNA
	bool
	is_DNA() const
	{
		return rsd_type_.is_DNA();
	}

	///@brief residue is RNA
	bool
	is_RNA() const
	{
		return rsd_type_.is_RNA();
	}

	///@brief residue is nucleic acid
	bool
	is_NA() const
	{
		return rsd_type_.is_NA();
	}

	bool
	is_ligand() const
	{
		return rsd_type_.is_ligand();
	}


  ///@brief residue is surface
  bool
  is_surface() const
  {
    return rsd_type_.is_surface();
  }

  ///@brief residue has orbitals
  bool
  has_orbitals() const
  {
	  return rsd_type_.has_orbitals();
  }

  ///@brief residue has orbitals
  bool
  has_sc_orbitals() const
  {
	  return rsd_type_.has_sc_orbitals();
  }

  bool
  has_bb_orbitals() const
  {
	  return rsd_type_.has_bb_orbitals();
  }

  ///@brief residue is polor
	bool
	is_polar() const
	{
		return rsd_type_.is_polar();
	}

	///@brief is residue apolar?
	bool
	is_apolar() const
	{
		if(rsd_type_.is_polar() || rsd_type_.is_aromatic() || rsd_type_.is_charged()){
			return false;
		}else {
			return true;
		}
	}


	///@brief residue is charged
	bool
	is_charged() const
	{
		return rsd_type_.is_charged();
	}

	///@brief residue is aromatic
	bool
	is_aromatic() const
	{
		return rsd_type_.is_aromatic();
	}

	///@brief residue has a type of terminus variant?
	bool
	is_terminus() const
	{
		return rsd_type_.is_terminus();
	}

	///@brief residue is upper terminus variant type
	bool
	is_upper_terminus() const
	{
		return rsd_type_.is_upper_terminus();
	}

	///@brief residue is lower terminus variant type
	bool
	is_lower_terminus() const
	{
		return rsd_type_.is_lower_terminus();
	}

	/// @brief  Generic property access -- SLOW!!!!!
	bool
	has_property( std::string const & property ) const
	{
		return rsd_type_.has_property( property );
	}

	/// @brief  Generic variant access -- SLOW!!!!!
	bool
	has_variant_type( chemical::VariantType const & variant_type ) const
	{
		return rsd_type_.has_variant_type( variant_type );
	}

	/////////////////////////////
	///@brief name strings
	std::string const &
	atom_name( int const atm ) const
	{
		return rsd_type_.atom_name( atm );
	}

	/// @brief name of the mm_atom type
	std::string const &
	mm_atom_name(int const atom) const
	{
		return rsd_type_.mm_atom_name(atom);
	}


	///@brief get our (unique?) residue name
	std::string const &
	name() const
	{
		return rsd_type_.name();
	}

	///@brief get our 3-letter code
	std::string const &
	name3() const
	{
		return rsd_type_.name3();
	}

	//@brief/ get our 1-letter code
	char
	name1() const
	{
		return rsd_type_.name1();
	}

	///@brief our traditional rsd type, if any
	/**
		 Used for knowledge-based scores, dunbrack, etc.
		 could be "aa_unk".  AA is enumeration.
	**/
	chemical::AA const &
	aa() const
	{
		return rsd_type_.aa();
	}


	/// @brief access atom's internal coords
	chemical::AtomICoor const &
	icoor( int const atm ) const
	{
		return rsd_type_.icoor( atm );
	}

	/// @brief Sets the chi angle assuming that change propagates according to the atom_base tree
	void
	set_chi( int const chino, Real const setting );

	/// @brief wrapper function to set all chi values using the set_chi function
	void
	set_all_chi( utility::vector1< Real > const & chis );


	/// @brief this residue has an atom of this name?
	bool
	has( std::string const & atm ) const
	{
		return rsd_type_.has( atm );
	}

	/// @brief build coords for atoms which are missing from this residue assuming ideal internal coords
	void
	fill_missing_atoms(
		utility::vector1< bool > missing,
		Conformation const & conformation
	);

	/// @brief  Select the three atoms to be used for orienting
	void
	select_orient_atoms(
		Size & center,
		Size & nbr1,
		Size & nbr2
	) const;

	/// @brief  Orient our coords onto those of src, using the atoms from select_orient_atoms
	void
	orient_onto_residue( Residue const & src );

	/// @brief  Orient our coords onto those of src, using the atoms specified in the input
	void
	orient_onto_residue(
		Residue const & src,
		utility::vector1< std::pair< std::string, std::string > > const & atom_pairs
	);

	/// @brief place this rotamer at the sequence position occupied by src by reorienting the ideal sc-coords to match
	void
	place(
		Residue const & src,
		Conformation const & conformation,
		bool preserve_c_beta = false
	);

	/// @brief Applies a transform of the form Rx + v, where R is a rotation
	/// matrix, V is a vector, and x is the original position in xyz space.
	void
	apply_transform_Rx_plus_v(
		numeric::xyzMatrix< Real > R,
		Vector v
	);



	/////////////////////////////////////////////////////////////////////////////
	// private methods
	/////////////////////////////////////////////////////////////////////////////

private:

	/// @brief  Updates connections_to_residues_ using connect_map_
	void
	update_connections_to_residues();


	/// @brief apply transform of rotation R and translation V for all atoms downstream
	/// @note note this is not for general atom tree folding. only used in set_chi in which
	/// changes for a chi angle is fast propagated within one residue and not to invoke
	/// folding the whole atom tree.
	void
	apply_transform_downstream(
		int const atomno,
		numeric::xyzMatrix< Real > const & R,
		Vector const & v
	);


	void
	determine_nonstandard_polymer_status();

	/// @brief Assignment operator does not work for class Residue.
	/// This function is intentionally unimplemented and private.
	Residue const &
	operator = ( Residue const & rhs );



	/////////////////////////////////////////////////////////////////////////////
	// data
	/////////////////////////////////////////////////////////////////////////////

private:

	/// @brief our Residue type
	//-- should be CAP, perhaps?
	ResidueType const & rsd_type_;

	/// @brief our conformation atoms (not kinematic atom pointers) with xyz positiona and atom type
	Atoms atoms_;

	/// @brief the sequence position
	Size seqpos_;

	/// @brief the chain id number, starting from 1
	core::Size chain_; ///TODO make this a core::Size

	/// @brief our chi (sidechain) torsion angles
	utility::vector1< Real > chi_;

	/// @brief our (possibly empty) backbone torsion angles
	utility::vector1< Real > mainchain_torsions_;

	/// @brief the action coordinate, an interaction centroid for knowledge-based terms like fa-pair
	/// in fact, only for fa-pair
	Vector actcoord_;


	/////////////////////////////////
	/// Inter-residue connection data

	/// @brief true iff is_polymer() and either upper_connect or lower_connect (if they exist)
	/// do not connect to seqpos()+1 or seqpos()-1
	bool nonstandard_polymer_;

	/// @brief map between connection ids on this residue and the
	/// connection points on other residues to which its bonded
	utility::vector1< chemical::ResConnID > connect_map_;

	/// @brief lists for each connected residue of the connection points on this residue
	/// that connect the pair.
	std::map< Size, utility::vector1< Size > > connections_to_residues_;

	/// @brief other residues within 4 bonds (connected through PseudoBonds)
	/// may include this residue (intra-residue pseudo-bonds)
	std::map< Size, PseudoBondCollectionCOP > pseudobonds_;

};

std::ostream & operator << ( std::ostream & os, Residue const & res );

} // conformation
} // core



#endif
