// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author Phil Bradley


// Unit headers
#include <core/conformation/util.hh>

// Package headers
#include <core/conformation/Conformation.hh>
// AUTO-REMOVED #include <core/kinematics/util.hh>
#include <core/id/AtomID_Map.hh>
#include <core/id/TorsionID.hh>
//#include <core/id/types.hh>
// AUTO-REMOVED #include <core/kinematics/constants.hh>
#include <core/kinematics/Stub.hh>

// Project headers
#include <core/conformation/Residue.hh>
#include <core/conformation/ResidueFactory.hh>
// AUTO-REMOVED #include <core/chemical/VariantType.hh>

// AUTO-REMOVED #include <core/util/basic.hh>

// ObjexxFCL headers
// #include <ObjexxFCL/FArray1A.hh>
// #include <ObjexxFCL/FArray2A.hh>
// #include <ObjexxFCL/FArray3A.hh>
// #include <ObjexxFCL/format.hh>
// #include <ObjexxFCL/string.functions.hh> // Pretty output

// Numeric headers
#include <numeric/util.hh>
#include <numeric/xyz.functions.hh>

// Utility headers
#include <utility/exit.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/pointer/access_ptr.hh>

// C++ headers
// #include <algorithm>
#include <cassert>
// #include <cstddef>
// #include <iostream>
// #include <fstream>
#include <core/util/Tracer.hh>

static core::util::Tracer TR( "core.conformation.util" );


namespace core {
namespace conformation {



void
orient_residue_for_ideal_bond(
	Residue & moving_rsd,
	chemical::ResidueConnection const & moving_connection,
	Residue const & fixed_rsd,
	chemical::ResidueConnection const & fixed_connection,
	Conformation const & conformation // unused
)
{

	// confirm that both connections are defined entirely by atoms internal to their residues,
	// so we don't need the conformation info (which might not be correct for the moving_rsd if we
	// are in the process of adding it to the conformation)
	//
	runtime_assert( moving_connection.icoor().is_internal() && fixed_connection.icoor().is_internal() );

	// a1 ---  a2  --- (a3)
	//
	//        (b2) ---  b3  --- b4
	//
	// a3 and b2 are the pseudo-atoms built from their respective residues using the ideal geometry in the
	// corresponding ResidueConnection objects
	//
	Vector
		a1(  fixed_connection.icoor().stub_atom2().xyz(  fixed_rsd, conformation ) ),
		b4( moving_connection.icoor().stub_atom2().xyz( moving_rsd, conformation ) ),

		a2(  fixed_rsd.xyz(  fixed_connection.atomno() ) ),
		b3( moving_rsd.xyz( moving_connection.atomno() ) ),

		a3(  fixed_connection.icoor().build(  fixed_rsd, conformation ) ),
		b2( moving_connection.icoor().build( moving_rsd, conformation ) );

	// we want to move b2 to align with a2 and b3 to align with a3. Torsion about the a2->a3 bond
	// (ie the inter-residue bond) determined by atoms a1 and b4 (torsion set to 0 by default).
	//
	kinematics::Stub A( a3, a2, a1 ), B( b3, b2, b4 );

	for ( Size i=1; i<= moving_rsd.natoms(); ++i ) {
		moving_rsd.set_xyz(i, A.local2global( B.global2local( moving_rsd.xyz(i) ) ) );
	}

	moving_rsd.actcoord() = A.local2global( B.global2local( moving_rsd.actcoord() ) );
}




///////////////////////////////////////////////////////////////////////////////////////
void
insert_ideal_mainchain_bonds(
	Size const seqpos,
	Conformation & conformation
)
{
	using namespace id;

	Residue const & rsd( conformation.residue( seqpos ) );

	// create a mini-conformation with
	Conformation idl;
	{
		ResidueOP idl_rsd( ResidueFactory::create_residue( rsd.type() ) );
		idl.append_residue_by_bond( *idl_rsd );
	}

	int idl_pos(1);

	// intra-residue mainchain bonds and angles
	Size const nbb( rsd.n_mainchain_atoms() );
	runtime_assert( nbb >= 2 ); // or logic gets a bit trickier
	for ( Size i=2; i<= nbb; ++i ) {
		AtomID
			bb1_idl( rsd.mainchain_atom(i-1), idl_pos ),
			bb1    ( rsd.mainchain_atom(i-1),  seqpos ),
			bb2_idl( rsd.mainchain_atom(  i), idl_pos ),
			bb2    ( rsd.mainchain_atom(  i),  seqpos );

		// bond length
		conformation.set_bond_length( bb1, bb2, idl.bond_length( bb1_idl, bb2_idl ) );

		if ( i<nbb ) {
			AtomID
				bb3_idl( rsd.mainchain_atom(i+1), idl_pos ),
				bb3    ( rsd.mainchain_atom(i+1),  seqpos );

			// bond angle
			conformation.set_bond_angle( bb1, bb2, bb3, idl.bond_angle( bb1_idl, bb2_idl, bb3_idl ) );
		}
	}

	if ( seqpos>1 && rsd.is_polymer() && !rsd.is_lower_terminus() ) {
		ResidueOP prev_rsd( ResidueFactory::create_residue( conformation.residue( seqpos-1 ).type() ) );
		idl.prepend_polymer_residue_before_seqpos( *prev_rsd, 1, true );
		idl_pos = 2;

		Size const nbb_prev( prev_rsd->n_mainchain_atoms() );
		AtomID
			bb1_idl( prev_rsd->mainchain_atom( nbb_prev ), idl_pos-1 ),
			bb1    ( prev_rsd->mainchain_atom( nbb_prev ),  seqpos-1 ),
			bb2_idl( rsd.mainchain_atom( 1 ), idl_pos ),
			bb2    ( rsd.mainchain_atom( 1 ),  seqpos ),
			bb3_idl( rsd.mainchain_atom( 2 ), idl_pos ),
			bb3    ( rsd.mainchain_atom( 2 ),  seqpos );
		conformation.set_bond_angle( bb1, bb2, bb3, idl.bond_angle( bb1_idl, bb2_idl, bb3_idl ) );
	}

	if ( seqpos < conformation.size() && rsd.is_polymer() && !rsd.is_upper_terminus() ) {
		ResidueOP next_rsd( ResidueFactory::create_residue( conformation.residue( seqpos+1 ).type() ) );
		idl.append_polymer_residue_after_seqpos( *next_rsd, idl_pos, true );

		AtomID
			bb1_idl( rsd.mainchain_atom( nbb-1 ), idl_pos ),
			bb1    ( rsd.mainchain_atom( nbb-1 ),  seqpos ),
			bb2_idl( rsd.mainchain_atom( nbb   ), idl_pos ),
			bb2    ( rsd.mainchain_atom( nbb   ),  seqpos ),
			bb3_idl( next_rsd->mainchain_atom( 1 ), idl_pos+1 ),
			bb3    ( next_rsd->mainchain_atom( 1 ),  seqpos+1 );

		// bond angle
		conformation.set_bond_angle( bb1, bb2, bb3, idl.bond_angle( bb1_idl, bb2_idl, bb3_idl ) );

		// bond length
		conformation.set_bond_length( bb2, bb3, idl.bond_length( bb2_idl, bb3_idl ) );
	}



}

/// @details  Just sets the two bond angles and the bond length between seqpos and seqpos+1

void
insert_ideal_bonds_at_polymer_junction(
	Size const seqpos,
	Conformation & conformation
)
{
	using namespace id;

	Residue const & rsd( conformation.residue( seqpos ) );
	runtime_assert( seqpos < conformation.size() && rsd.is_polymer() && !rsd.is_upper_terminus() );

	if ( conformation.fold_tree().is_cutpoint( seqpos ) ) {
		TR.Warning << "insert_ideal_bonds_at_polymer_junction: seqpos (" << seqpos << ") is a foldtree cutpoint, " <<
			"returning!" << std::endl;
	}

	// create a mini-conformation with ideal bond lengths and angles
	Conformation idl;
	{
		ResidueOP idl_rsd( ResidueFactory::create_residue( rsd.type() ) );
		idl.append_residue_by_bond( *idl_rsd );
	}

	int idl_pos(1);

	Size const nbb( rsd.n_mainchain_atoms() );
	runtime_assert( nbb >= 2 ); // or logic gets a bit trickier

	ResidueOP next_rsd( ResidueFactory::create_residue( conformation.residue( seqpos+1 ).type() ) );
	idl.append_polymer_residue_after_seqpos( *next_rsd, idl_pos, true /* append with ideal geometry */ );

	AtomID
		bb1_idl( rsd.mainchain_atom( nbb-1 ), idl_pos ),
		bb1    ( rsd.mainchain_atom( nbb-1 ),  seqpos ),
		bb2_idl( rsd.mainchain_atom( nbb   ), idl_pos ),
		bb2    ( rsd.mainchain_atom( nbb   ),  seqpos ),
		bb3_idl( next_rsd->mainchain_atom( 1 ), idl_pos+1 ),
		bb3    ( next_rsd->mainchain_atom( 1 ),  seqpos+1 ),
		bb4_idl( next_rsd->mainchain_atom( 2 ), idl_pos+1 ),
		bb4    ( next_rsd->mainchain_atom( 2 ),  seqpos+1 );

	// bond angle
	conformation.set_bond_angle( bb1, bb2, bb3, idl.bond_angle( bb1_idl, bb2_idl, bb3_idl ) );

	// bond length
	conformation.set_bond_length( bb2, bb3, idl.bond_length( bb2_idl, bb3_idl ) );

	// bond angle
	conformation.set_bond_angle( bb2, bb3, bb4, idl.bond_angle( bb2_idl, bb3_idl, bb4_idl ) );

	// fix up atoms that depend on the atoms across the bond for their torsion offsets
	conformation.rebuild_polymer_bond_dependent_atoms( seqpos );

}


///////////////////////////////////////////////////////////////////////////////////////
void
idealize_position(
	Size const seqpos,
	Conformation & conformation
)
{
	using namespace id;

	runtime_assert( seqpos > 0 );
	runtime_assert( conformation.size() >= seqpos );
	runtime_assert( conformation.size() > 0 );
	Residue const & rsd( conformation.residue( seqpos ) );

	//// create a mini-conformation with completely ideal residue ( and nbrs, if appropriate )

	Conformation idl;
	{
		ResidueOP idl_rsd( ResidueFactory::create_residue( rsd.type() ) );
		idl.append_residue_by_bond( *idl_rsd );
	}

	int idl_pos(1);
	bool lower_connect( false ), upper_connect( false );

	if ( rsd.is_polymer() ) {
		// add polymer nbrs?
		if ( seqpos > 1 && !rsd.is_lower_terminus() && !conformation.fold_tree().is_cutpoint( seqpos-1 ) ) {
			lower_connect = true;
			ResidueOP prev_rsd( ResidueFactory::create_residue( conformation.residue( seqpos-1 ).type() ) );
			idl.prepend_polymer_residue_before_seqpos( *prev_rsd, 1, true );
			idl_pos = 2;
		}

		if ( seqpos < conformation.size() && !rsd.is_upper_terminus() && !conformation.fold_tree().is_cutpoint( seqpos ) ) {
			upper_connect = true;
			ResidueOP next_rsd( ResidueFactory::create_residue( conformation.residue( seqpos+1 ).type() ) );
			idl.append_polymer_residue_after_seqpos( *next_rsd, idl_pos, true );
		}
	}

	//// now set the torsion angles in the ideal conformation... This is to prepare for replacing rsd with
	//// the idealized version

	// chi angles
	for ( Size i=1; i<= rsd.nchi(); ++i ) {
		idl.set_torsion( TorsionID( idl_pos, CHI, i ), rsd.chi( i ) );
	}
	// mainchain torsions, if polymer residue
	if ( rsd.is_polymer() ) {
		Size const nbb( rsd.n_mainchain_atoms() );
		for ( Size i=1; i<= nbb; ++i ) {
			//if ( ( !lower_connect && i == 1 ) || ( !upper_connect && i >= nbb-1 ) ) continue;
			idl.set_torsion( TorsionID( idl_pos, BB, i ), rsd.mainchain_torsion( i ) );
		}
	}


	//// now we copy the mainchain bond geometry from ideal conf into the conf to be idealized
	if ( rsd.is_polymer() ) {

		// intra-residue mainchain bonds and angles
		Size const nbb( rsd.n_mainchain_atoms() );
		runtime_assert( nbb >= 2 ); // or logic gets a bit trickier
		for ( Size i=2; i<= nbb; ++i ) {
			AtomID
				bb1_idl( rsd.mainchain_atom(i-1), idl_pos ),
				bb1    ( rsd.mainchain_atom(i-1),  seqpos ),
				bb2_idl( rsd.mainchain_atom(  i), idl_pos ),
				bb2    ( rsd.mainchain_atom(  i),  seqpos );

			// bond length
			conformation.set_bond_length( bb1, bb2, idl.bond_length( bb1_idl, bb2_idl ) );

			if ( i<nbb ) {
				AtomID
					bb3_idl( rsd.mainchain_atom(i+1), idl_pos ),
					bb3    ( rsd.mainchain_atom(i+1),  seqpos );

				// bond angle
				conformation.set_bond_angle( bb1, bb2, bb3, idl.bond_angle( bb1_idl, bb2_idl, bb3_idl ) );
			}
		}

		if ( lower_connect ) {
			Residue const & prev_rsd( idl.residue( idl_pos-1 ) );

			Size const nbb_prev( prev_rsd.n_mainchain_atoms() );
			AtomID
				bb1_idl( prev_rsd.mainchain_atom( nbb_prev ), idl_pos-1 ),
				bb1    ( prev_rsd.mainchain_atom( nbb_prev ),  seqpos-1 ),
				bb2_idl(      rsd.mainchain_atom(        1 ), idl_pos   ),
				bb2    (      rsd.mainchain_atom(        1 ),  seqpos   ),
				bb3_idl(      rsd.mainchain_atom(        2 ), idl_pos   ),
				bb3    (      rsd.mainchain_atom(        2 ),  seqpos   );
			conformation.set_bond_angle( bb1, bb2, bb3, idl.bond_angle( bb1_idl, bb2_idl, bb3_idl ) );
		}

		if ( upper_connect ) {
			Residue const & next_rsd( idl.residue( idl_pos+1 ) );

			AtomID
				bb1_idl(      rsd.mainchain_atom( nbb-1 ), idl_pos   ),
				bb1    (      rsd.mainchain_atom( nbb-1 ),  seqpos   ),
				bb2_idl(      rsd.mainchain_atom( nbb   ), idl_pos   ),
				bb2    (      rsd.mainchain_atom( nbb   ),  seqpos   ),
				bb3_idl( next_rsd.mainchain_atom(     1 ), idl_pos+1 ),
				bb3    ( next_rsd.mainchain_atom(     1 ),  seqpos+1 );

			// bond angle
			conformation.set_bond_angle( bb1, bb2, bb3, idl.bond_angle( bb1_idl, bb2_idl, bb3_idl ) );

			// bond length
			conformation.set_bond_length( bb2, bb3, idl.bond_length( bb2_idl, bb3_idl ) );
		}
	} // rsd.is_polymer()


	//// now we orient the ideal residue onto the existing residue and replace it
	ResidueOP idl_rsd( idl.residue( idl_pos ).clone() );

	// EXTREMELY SUBTLE DANGEROUS THING ABOUT HAVING Residue const &'s lying around:
	// if we put "rsd" in place of conformation.residue( seqpos ) below, this will not behave properly.
	// that's because the xyz's in rsd don't get updated properly to reflect the changes to the conformation
	// a "refold" is only triggered by another call to conformation.residue
	// May want to consider making Residue smarter about this kind of thing, ie knowing its from a
	// Conformation... acting as an observer... could get pretty tricky though...
	//
	idl_rsd->orient_onto_residue( conformation.residue( seqpos ) ); // can't use rsd here!
	conformation.replace_residue( seqpos, *idl_rsd, false );

}


///////////////////////////////////////////////////////////////////////////////////////
// NOTE: conformation is needed for polymer neighbours
// @author Barak 07/2009
bool
is_ideal_position(
	Size const seqpos,
	Conformation const & conf,
 	Real theta_epsilon,
	Real D_epsilon
)
{
	using namespace core::kinematics;
	runtime_assert( conf.size() >= seqpos  &&  seqpos >= 1 );

	Residue const rsd( conf.residue( seqpos ) );

	// I. Create mini-conformations for both idealized and original residue + nbrs, if appropriate )
	Conformation miniconf, miniconf_idl;
	{
		miniconf.append_residue_by_bond( rsd );
		ResidueOP prsd_idl( ResidueFactory::create_residue( rsd.type() ) );
		miniconf_idl.append_residue_by_bond( *prsd_idl );
	}
	// add polymer nbrs if needed
	int seqpos_miniconf(1);
	bool lower_connect( false ), upper_connect( false );
	if ( rsd.is_polymer() ) {
		// prepending previous neighbour
		if ( seqpos > 1 && !rsd.is_lower_terminus() && !conf.fold_tree().is_cutpoint( seqpos-1 ) ) {
			lower_connect = true;
			seqpos_miniconf = 2; // pushed forward by prependedq residue
			Residue const & prev_rsd = conf.residue( seqpos-1 );
			miniconf.safely_prepend_polymer_residue_before_seqpos // TODO: safely is probably unneeded here, verify this point
				( prev_rsd, 1, false /*build_ideal_geom*/);
			ResidueOP pprev_rsd_idl( ResidueFactory::create_residue( prev_rsd.type() ) );
			miniconf_idl.safely_prepend_polymer_residue_before_seqpos
				( *pprev_rsd_idl, 1, true /*build_ideal_geom*/);
		}
		// appending next neighbour
		if ( seqpos < conf.size() && !rsd.is_upper_terminus() && !conf.fold_tree().is_cutpoint( seqpos ) ) {
			upper_connect = true;
			Residue const & next_rsd = conf.residue( seqpos+1 );
			miniconf.safely_append_polymer_residue_after_seqpos
				( next_rsd, seqpos_miniconf, false /*build_ideal_geom*/ );
			ResidueOP pnext_rsd_idl( ResidueFactory::create_residue( next_rsd.type() ) );
			miniconf_idl.safely_append_polymer_residue_after_seqpos
				( *pnext_rsd_idl, seqpos_miniconf, true /*build_ideal_geom*/ );
		}
	}


 	// II. Compare angle and length of all residue bonded atoms in the new mini-conformations
	for ( Size atom = 1, atom_end = miniconf.residue( seqpos_miniconf ).natoms();
				atom <= atom_end;
				++atom )
		{
			id::AtomID aid(atom, seqpos_miniconf);
			if( miniconf.atom_tree().atom(aid).is_jump() ) // skip non-bonded atoms
				continue;
			id::DOF_ID dof_theta(aid, id::THETA); // bond angle
			id::DOF_ID dof_D(aid, id::D); // bond length
			if( ! numeric::equal_by_epsilon(
					miniconf.dof( dof_theta ), miniconf_idl.dof( dof_theta ),	theta_epsilon	)	) {
				TR << "Non-ideal residue detected: "
					 << " Residue #" << seqpos << " atom #" << atom << "( " << rsd.atom_name( atom ) << " ) " << ": "
					 << " Ideal theta=" << miniconf_idl.dof( dof_theta)
					 << ", Inspected theta=" << miniconf.dof( dof_theta )
					 << " (in Radians)"
					 << std::endl;
				return false;
			}
			if( ! numeric::equal_by_epsilon(
					miniconf.dof( dof_D), miniconf_idl.dof (dof_D), D_epsilon	) ) {
				TR << "Non-ideal residue detected: "
					 << " Residue #" << seqpos << " atom #" << atom << "( " << rsd.atom_name( atom ) << " ) " << ": "
					 << " Ideal D=" << miniconf_idl.dof( dof_D)
					 << ", Inspected D=" << miniconf.dof( dof_D )
					 << std::endl;
				return false;
			}
		}

 	return true;
}



/// @details  For building variant residues, eg
/// @note  Need conformation for context in case we have to rebuild atoms, eg backbone H

void
copy_residue_coordinates_and_rebuild_missing_atoms(
	Residue const & source_rsd,
	Residue & target_rsd,
	Conformation const & conformation
)
{

	Size const natoms( target_rsd.natoms() );

	utility::vector1< bool > missing( natoms, false );
	bool any_missing( false );

	for ( Size i=1; i<= natoms; ++i ) {
		std::string const & atom_name( target_rsd.atom_name(i) );
		if ( source_rsd.has( atom_name ) ) {
			target_rsd.atom( i ).xyz( source_rsd.atom( atom_name ).xyz() );
		} else {
			TR.Debug << "copy_residue_coordinates_and_rebuild_missing_atoms: missing atom " << target_rsd.name() << ' ' <<
				atom_name << std::endl;
			any_missing = true;
			missing[i] = true;
		}
	}

	if ( any_missing ) {
		target_rsd.seqpos( source_rsd.seqpos() ); // in case fill_missing_atoms needs context info
		target_rsd.chain ( source_rsd.chain () );
		target_rsd.fill_missing_atoms( missing, conformation );
	}

}

/// @details  Helper function for below
std::ostream &
print_atom( id::AtomID const & id, Conformation const & conf, std::ostream & os )
{
	Residue const & rsd( conf.residue(id.rsd() ) );
	os << rsd.atom_name( id.atomno() ) << " (" << id.atomno() << ") " << rsd.name() << ' ' << rsd.seqpos();
	return os;
}

///
void
show_atom_tree(
	kinematics::tree::Atom const & atom,
	Conformation const & conf, std::ostream & os
) {
	os << "ATOM "; print_atom( atom.id(), conf, os ) << std::endl;
	os << "CHILDREN: ";
	for ( Size i=0; i< atom.n_children(); ++i ) {
		print_atom( atom.child(i)->id(), conf, os ) << ' ';
	}
	os << std::endl;
	for ( Size i=0; i< atom.n_children(); ++i ) {
		show_atom_tree( *atom.child(i), conf, os );
	}
}


} // namespace kinematics
} // namespace core
