// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragments/FragSet.cc
/// @brief  set of fragments for a certain alignment frame
/// @author Oliver Lange (olange@u.washington.edu)
/// @author James Thompson (tex@u.washington.edu)
/// @date   Wed Oct 20 12:08:31 2007
///

// Unit Headers
#include <core/fragment/ConstantLengthFragSet.hh>

// Package Headers
#ifdef WIN32
#include <core/fragment/FragID.hh>
#endif

#include <core/fragment/BBTorsionSRFD.hh>
#include <core/fragment/Frame.hh>
#include <core/fragment/FragData.hh>
#include <core/fragment/ConstantLengthFragSetIterator_.hh>

// Project Headers
// AUTO-REMOVED #include <core/pose/Pose.hh>
#include <core/types.hh>


// ObjexxFCL Headers
// AUTO-REMOVED #include <ObjexxFCL/format.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility headers
#include <utility/vector1.fwd.hh>
#include <utility/io/izstream.hh>
#include <utility/pointer/owning_ptr.hh>
#include <core/util/Tracer.hh>
#include <ostream>

//Auto Headers
#include <core/fragment/FrameIterator.hh>
#include <utility/exit.hh>

//Auto using namespaces
namespace ObjexxFCL { namespace fmt { } } using namespace ObjexxFCL::fmt; // AUTO USING NS
//Auto using namespaces end




namespace core {
namespace fragment {

using namespace kinematics;
using namespace ObjexxFCL;

static util::Tracer tr("core.fragments.ConstantLengthFragSet");
// preliminary reader method --- reads classic rosetta++ frag files

ConstantLengthFragSet::~ConstantLengthFragSet() {}

ConstantLengthFragSet::ConstantLengthFragSet( Size frag_length, std::string filename ) {
		set_max_frag_length( frag_length );
		read_fragment_file(filename);
	}

ConstantLengthFragSet::ConstantLengthFragSet( FragSet const& fragments ) {
	set_max_frag_length( fragments.max_frag_length() );
	Parent::add( fragments ); 	// if fragments are of differing length assertion in add will fail.
}

///@brief there is only one Frame per position, end / max_overlap are ignored
Size ConstantLengthFragSet::region(
	MoveMap const& mm,
	core::Size start,
	core::Size end,
	core::Size , //min_overlap not used
	core::Size , //min_length not used
	FrameList &frames
) const {
	Size count( 0 );
	for ( Size pos=start; pos<=frames_.size() && pos<=end; pos++ ) {
		if ( frames_[pos] ) {
			if ( frames_[pos]->is_applicable( mm ) && frames_[pos]->is_valid() ) {
				frames.push_back( frames_[ pos ] );
				count++;
			}
		}
	}
	return count;
}

/// @brief Accessor for the Frame at the specified insertion position. Returns false if
/// there is no frame at the specified position.
// Size ConstantLengthFragSet::frames(
// 	Size pos,
// 	Frame & frame
// ) {
// 	if ( frames_[pos] ) {
// 		frame = *frames_[pos];
// 		return 1;
// 	}
// 	return 0;
// }

void ConstantLengthFragSet::add_( FrameOP aframe ) {
	//this method will only be called if the frame has been incompatible with existing frames
	// in case of ConstantLengthFragSet aka single-kind-of Frame FragSet this means position at aframe->start() is empty

	if ( max_frag_length() ) runtime_assert( aframe->length() == max_frag_length() );
	Size seqpos( aframe->start() );
	if ( frames_.size() < seqpos ) {
		frames_.resize( seqpos, NULL );
	}
	runtime_assert( frames_[ seqpos ] == 0 ); //I consider this as error... don't add incompatible frames to a ConstantLengthFragSet
	frames_[ seqpos ] = aframe;
}


void ConstantLengthFragSet::read_fragment_file( std::string filename, Size top25, Size ncopies, bool bAnnotation ) {
	//	std::cout << "Reading fragments from file "
	//					<< filename << "!" << std::endl;

	utility::io::izstream data( filename );
	if ( !data.good() ) {
		std::cerr << "Open failed for file: " << data.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string line;

	Size insertion_pos = 1;
	FragDataOP current_fragment( NULL );
	FrameOP frame;// = new Frame( insertion_pos, fragment_len );

	Size n_frags( 0 );
	while ( getline( data, line ) ) {
		using namespace ObjexxFCL;
		using namespace ObjexxFCL::fmt;
		// skip blank lines
		if ( line == "" || line == " " ) {
			// add current_fragment to the appropriate frame
			if ( current_fragment && frame ) {
				if ( !top25 || frame->nr_frags() < top25*ncopies ) {
					current_fragment->set_valid(); //it actually containts data
					if ( !frame->add_fragment( current_fragment ) ) {
						tr.Fatal << "Incompatible Fragment in file: " << data.filename() << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					} else {
						for ( Size i = 2; i <= ncopies; i++ ) frame->add_fragment( current_fragment );
					}
				}
			}

			// create a new current_fragment
			current_fragment = NULL;//new FragData;
			continue;
		}

		// skip lines beginning with # characters, representing comments
		if ( line.substr(0,1) == "#" ) {
			continue;
		}

		// skip lines that are too short?? --- such a stupid thing
		if ( line.length() < 22 ) {
			tr.Warning << "line too short: Skipping line '" << line << "'" << std::endl;
			continue;
		}

		// skip position and neighbor lines for now
		if ( line.find("position") != std::string::npos ) {
			std::istringstream in( line );
			//			insertion_pos = int_of( line.substr( 17, 6 ) );
			std::string tag;
			in >> tag;
			in >> insertion_pos;
			// create a new frame
			//if (frame) add( frame );
			if ( frame && frame->is_valid() ) {
				add( frame );
				n_frags = std::max( n_frags, frame->nr_frags() );
			}
			frame = new Frame( insertion_pos );
			continue;
		}

		int aa_index;
		std::string pdbid;
		char chain, ss, aa;
		Real phi, psi, omega, weight;

		pdbid    =           line.substr(  1, 4  );
		chain    = char_of(  line.substr(  6, 1  ) );
		aa_index = int_of(   line.substr(  8, 5  ) );
		aa       = char_of(  line.substr( 14, 1  ) );
		ss       = char_of(  line.substr( 16, 1  ) );
		phi      = float_of( line.substr( 18, 9  ) );
		psi      = float_of( line.substr( 27, 9  ) );
		omega    = float_of( line.substr( 36, 9  ) );
		if ( line.length() > 46 ) {
			weight   = float_of( line.substr( 47, 6  ) );
		}

		utility::pointer::owning_ptr< BBTorsionSRFD > res_torsions( new BBTorsionSRFD(3,ss,aa) ); // 3 protein torsions
		res_torsions->set_torsion   ( 1, phi   ); // ugly numbers 1-3, but pose.set_phi also uses explicit numbers
		res_torsions->set_torsion   ( 2, psi   );
		res_torsions->set_torsion   ( 3, omega );
		res_torsions->set_secstruct ( ss );
		if ( !current_fragment ) {
			if ( bAnnotation ) current_fragment = new AnnotatedFragData( pdbid, aa_index );
			else current_fragment = new FragData;
		}
 		current_fragment->add_residue( res_torsions );
	} // while ( getline( data, line ) )
	//if (frame) add( frame ); //add last frame
	if ( frame && frame->is_valid() ) {
		add( frame );
		n_frags = std::max( n_frags, frame->nr_frags() );
	}
	tr.Info << "finished reading top " << n_frags << " "
		//<< ( top25 ? top25 : frame->nr_frags() ) << " "
		<< max_frag_length() << "mer fragments from file " << filename
		<< std::endl;
} // read_fragment_file( string fragment_file )
/*
	std::ostream& operator<<(std::ostream& out, const FragmentSet& fs )
		{
			using namespace ObjexxFCL::fmt;

			FragDataOP f;
			int n_positions = fs.get_npositions();
			for ( int i = 1; i <= n_positions; ++i ) {
				int nfrags = fs.get_nfrags( i );
				for ( int j = 1; j <= nfrags; ++j ) {
					// f = fs.get_fragment(i,j);
					//out << i << '\t' << j << std::endl;
					//out << f << std::endl;
					for ( int k = 1; k <= f->get_length(); k++ ) {
						out << ' '
								// << f->get_pdbid()  						<< ' '
								// << f->get_chain() 						<< ' '
								<< f->get_secstruct(k) 			  << '\t'
								<< F( 9, 3, f->get_phi(k)   ) << '\t'
								<< F( 9, 3, f->get_psi(k)   ) << '\t'
								<< F( 9, 3, f->get_omega(k) ) << '\t'
								<< std::endl;
					}
					out << std::endl;
				}
			}
			return out;
		} // std::ostream& operator<<(std::ostream& out, const FragmentSet& fs
*/

FrameIterator ConstantLengthFragSet::begin() const
{
	return FrameIterator( new ConstantLengthFragSetIterator_( frames_.begin(), frames_.end() ) );
}

FrameIterator ConstantLengthFragSet::end() const
{
	return FrameIterator( new ConstantLengthFragSetIterator_( frames_.end(), frames_.end() ) );
}



} //fragment
} //core
