// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragments/FragSet.cc
/// @brief  set of fragments for a certain alignment frame
/// @author Oliver Lange (olange@u.washington.edu)
/// @author James Thompson (tex@u.washington.edu)
/// @date   Wed Oct 20 12:08:31 2007
///

// Unit Headers
#include <core/fragment/FragSet.hh>

// Package Headers
// AUTO-REMOVED #include <core/fragment/BBTorsionSRFD.hh>
#include <core/fragment/Frame.hh>
#include <core/fragment/FrameIteratorWorker_.hh>
#include <core/fragment/FrameList.hh>
#include <core/fragment/FragData.hh>


// Project Headers
#include <core/kinematics/MoveMap.hh>
// AUTO-REMOVED #include <core/pose/Pose.hh>
#include <core/types.hh>


// ObjexxFCL Headers

// Utility headers
#include <utility/vector1.fwd.hh>
#include <utility/exit.hh>
// AUTO-REMOVED #include <utility/io/izstream.hh>
#include <utility/pointer/owning_ptr.hh>

#include <core/util/Tracer.hh>

#include <ostream>
// AUTO-REMOVED #include <set>

//Auto Headers
#include <core/fragment/FragID.hh>
#include <core/fragment/FrameIterator.hh>


namespace core {
namespace fragment {

using namespace kinematics;

static util::Tracer tr("core.fragment");

///@brief return a list of frames that all sample the specified region, assume all motions are allowed
Size
FragSet::region_all(
	core::Size start,
	core::Size end,
	core::Size min_overlap,
	core::Size min_length,
	FrameList &frames
) const {
	kinematics::MoveMap move_map;
	move_map.set_bb( true );
	move_map.set_chi( true );
	move_map.set_jump( true );
	return region( move_map, start, end, min_overlap, min_length, frames );
}


	// put all fragments in FragID_list into this FragSet.
	// this function has the following effect:
	//      fragments that belong to the same frame are copied into a new frame
	//      the frame gets added. If all fragments of a frame are in the list, the frame is just added as is
	//
void FragSet::insert_fragID_list( FragID_List& list ) {
	utility::vector1< bool > handled(list.size(), false );
	Size pos( 1 );
	for ( FragID_List::iterator it=list.begin(), eit=list.end(); it!=eit; ++it,++pos ) {
		if ( !handled[ pos ] ) {
			FrameOP new_frame = it->frame().clone();
			Size spos( pos );
			for ( FragID_List::iterator sit=it; sit!=eit; ++sit, spos++ ) {
				if ( sit->frame_ptr() == it->frame_ptr() ) {
					handled[ spos ] = true;
					new_frame->add_fragment( &( sit->fragment() ) );
					new_frame->clone_cache_data( it->frame(), it->id(), new_frame->nr_frags() /* last added fragment */ );
				};
			}

			tr.Debug << pos << ": add frame " << new_frame << " " << new_frame->nr_frags() << std::endl;
			add( new_frame );
		} // handled
	}
}

void FragSet::add( FragID const& frag_id ) {
	Frame const& aFrame( frag_id.frame() );
	//std::cerr << "FragSet::add_frame " << std::endl;
	runtime_assert( aFrame.nr_frags() ); // do not allow insertion of empty frames --> makes frag_id iterator sooo much easier
	Size start ( aFrame.start() );
	Size end ( aFrame.end() );
	Size length( aFrame.length() );

	if ( min_pos() > start ) {
		set_min_pos( start );
	};

	if ( max_pos() < end ) {
		set_max_pos( end );
	};

	//	tr.Trace << "frag length " << length << " ( " << max_frag_length() <<  " ) " << std::endl;
	if ( length > max_frag_length() ) {
		tr.Trace << "set max frag length " << length << std::endl;
		set_max_frag_length( length );
	}

	// now add frame:
	FrameList present_frames;
	Size nr_present = frames( start, present_frames );
	if ( nr_present ) {
		for ( FrameList::iterator it = present_frames.begin(),
						eit = present_frames.end(); it!=eit; ++it ) {
			if ( (*it)->is_mergeable( aFrame ) ) {
				Size const new_id( (*it)->add_fragment( const_cast< FragData*>( frag_id.fragment_ptr().get() ) ) );
				(*it)->clone_cache_data( aFrame, frag_id.id(), new_id );
				return; //finished early
			}
		}
	}
	//didn't found mergable frames at this sequence position
	// make a new empty frame for this Fragment
	FrameOP new_frame = aFrame.clone();
	Size const new_id( new_frame->add_fragment( const_cast< FragData*>( frag_id.fragment_ptr().get() ) ) );
	new_frame->clone_cache_data( aFrame, frag_id.id(), new_id );
	add_( new_frame );
}

void
FragSet::generate_insert_map( MoveMap const& mm, InsertMap &insert_map, InsertSize &insert_size ) const {
	tr.Debug << "generate insert map from Movemap:\n";
	for ( Size i = 1; i<=max_pos(); i++) {
		if ( mm.get_bb( i ) ) tr.Debug << "*";
		else tr.Debug << "x";
	}
	tr.Debug << std::endl;
	typedef std::map< Size, Size> InsertSet;
	InsertSet insert_set;
	// want to use FrameIterator but doesn't work as const method...
	FragSet& me( *const_cast<FragSet*>(this) );
	for ( FrameIterator it=me.begin(), eit=me.end(); it!=eit; ++it ) {
		Size size ( it->is_valid() ? it->is_applicable( mm ) : 0 );
		if ( size ) {
			if ( insert_set[ it->start() ] < size ) insert_set[ it->start() ] = size;
		}
	}

	insert_map.clear();
	insert_size.clear();
	if( insert_set.size() != 0){
		insert_size.resize( insert_set.rbegin()->first ); //the largest residue in insert_map
	}
	// now copy it into a simple vector of numbers
	//for ( std::set< std::pair< Size, Size > >::const_iterator it=insert_set.begin(), eit=insert_set.end();
	for ( InsertSet::const_iterator it=insert_set.begin(), eit=insert_set.end();
				it!=eit; ++it ) {
		insert_map.push_back( it->first );
		insert_size[ it->first ] = it->second;
	}
}

Size FragSet::size() const {
	// want to use FrameIterator but doesn't work as const method... sometime somebody will add ConstFrameIterator :-)
	FragSet& me( *const_cast<FragSet*>(this) );
	Size tot = 0;
	for ( FrameIterator it=me.begin(), eit=me.end(); it!=eit; ++it ) {
		tot+=(*it)->nr_frags();
	}
	return tot; //frames_.size();
}

Size FragSet::nr_frames() const {
	// want to use FrameIterator but doesn't work as const method... sometime somebody will add ConstFrameIterator :-)
	FragSet& me( *const_cast<FragSet*>(this) );
	Size tot = 0;
	for ( FrameIterator it=me.begin(), eit=me.end(); it!=eit; ++it ) {
		tot+=1;
	}
	return tot; //frames_.size();
}

void
FragSet::add( FrameOP aFrame ) {
	//std::cerr << "FragSet::add_frame " << std::endl;
	runtime_assert( aFrame->nr_frags() ); // do not allow insertion of empty frames --> makes frag_id iterator sooo much easier
	Size start ( aFrame->start() );
	Size end ( aFrame->end() );
	Size length( aFrame->length() );

	if ( min_pos() > start ) {
		set_min_pos( start );
	};

	if ( max_pos() < end ) {
		set_max_pos( end );
	};

	//	tr.Trace << "frag length " << length << " ( " << max_frag_length() <<  " ) " << std::endl;
	if ( length > max_frag_length() ) {
		tr.Trace << "set max frag length " << length << std::endl;
		set_max_frag_length( length );
	}

	// now add frame:
	FrameList present_frames;
	Size nr_present = frames( start, present_frames );
	if ( !nr_present ) {
		add_( aFrame );
	} else {
		for ( FrameList::iterator it = present_frames.begin(),
						eit = present_frames.end(); it!=eit; ++it ) {
			if ( (*it)->is_mergeable( *aFrame ) ) {
				(*it)->merge( *aFrame );
				return; //finished early
			}
		}
		//didn't found mergable frames at this sequence position
		add_( aFrame );
	}
}

void
FragSet::add( FrameList const& frames ) {
	for ( FrameList::const_iterator it=frames.begin(), eit=frames.end(); it!=eit; ++it ) {
		add( *it );
	}
}

void
FragSet::add( FragSet const& cframes ) {
	// ( doesn't change FragSet, but uses the non-const FrameIterator... )
	// eventually I might give the FrameIterator the correct const behaviour...
	FragSet &frames = const_cast< FragSet& >( cframes );
	for ( FrameIterator it=frames.begin(), eit=frames.end(); it!=eit; ++it ) {
		add( *it );
	}
}

std::ostream& operator<< (std::ostream& out, FragSet const& cfrags ) {
	FragSet &frags = const_cast< FragSet& >( cfrags );
	for ( FrameIterator it = frags.begin(),
					eit = frags.end(); it!=eit; ++it ) {
		out << *(*it);
	}
	return out;
}

} //fragment
} //core
