// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragment/picking/BoundedCollector.hh
/// @brief
/// @author Dominik Gront (dgront@chem.uw.edu.pl)

#ifndef INCLUDED_core_fragment_picking_BoundedCollector_HH
#define INCLUDED_core_fragment_picking_BoundedCollector_HH

// package headers
#include <core/fragment/picking/LazySortedVector1.hh>
#include <core/fragment/picking/BoundedPriorityQueue.hh>
#include <core/fragment/picking/scores/FragmentScoreMap.hh>
#include <core/fragment/picking/FragmentCandidate.hh>
#include <core/fragment/picking/BoundedCollector.fwd.hh>
#include <core/fragment/picking/CandidatesCollector.hh>
#include <core/fragment/picking/scores/FragmentScoreManager.hh>

// utility headers
#include <core/types.hh>

#include <core/util/prof.hh>

/// ObjexxFCL headers
#include <ObjexxFCL/format.hh>

#include <iostream>
//#include <algorithm>

namespace core {
namespace fragment {
namespace picking {


/// @brief Keeps the N best fragments candidates for the final selection
/// @detailed The purpose of a collector is to keep the best fragment candidates to the end
/// of vall processing. In particular, the capacity of this collector may be larger than
/// the number of fragments one wants to get
template<class StrictWeakOrdering>
class BoundedCollector: public CandidatesCollector {
public:

	/// @brief create a collector for a given size of a query sequence
	BoundedCollector(Size query_size, Size max_frags_per_pos,
			StrictWeakOrdering fragment_comparator,Size n_score_terms,Size buffer_factor = 5) {

		FragmentCandidateOP worst_f = new FragmentCandidate(1,1,0,1);
		scores::FragmentScoreMapOP worst_s = new scores::FragmentScoreMap(n_score_terms);
		for(Size i=1;i<=n_score_terms;i++)
		    worst_s->set_score_component(99999.9999,i);

		for (Size i = 1; i <= query_size; i++) {
			LazySortedVector1<std::pair<FragmentCandidateOP,
					scores::FragmentScoreMapOP>, StrictWeakOrdering> queue(
					fragment_comparator, max_frags_per_pos,max_frags_per_pos*buffer_factor);
			queue.set_worst( std::pair<FragmentCandidateOP,
			                scores::FragmentScoreMapOP>(worst_f,worst_s) );
			storage_.push_back(queue);
		}
	}

	/// @brief  Insert a fragment candidate to the container
	inline bool add(
			std::pair<FragmentCandidateOP, scores::FragmentScoreMapOP> new_canditate) {

		return storage_[new_canditate.first->get_first_index_in_query()].push(
				new_canditate);
	}

	/// @brief  Check how many candidates have been already collected for a given position
	inline Size count_candidates(Size seq_pos) {
		return storage_[seq_pos].size();
	}

	/// @brief  Check how many candidates have been already collected for all positions
	inline Size count_candidates() {

		Size response = 0;
		for(Size i=1;i<=storage_.size();++i)
			response += storage_[i].size();
		return response;
	}

	/// @brief  Check the size of query sequence that this object knows.
	/// This is mainly to be able to check if it is the same as in the other parts of
	/// fragment picking machinery.
	inline Size query_length() {
		return storage_.size();
	}

	/// @brief returns all stored fragment candidates that begins at a given position in a query
	inline utility::vector1<std::pair<FragmentCandidateOP,
			scores::FragmentScoreMapOP> >& get_candidates(
			Size position_in_query) {
		return storage_.at(position_in_query).expose_data();
	}

	inline void clear() {

	    for (Size i_pos = 1; i_pos <= storage_.size(); ++i_pos)
		storage_[i_pos].clear();
	}

	/// @brief prints how many candidates have been collected for each position and how good they are
	void print_report(std::ostream& out, scores::FragmentScoreManagerOP scoring) {
		using namespace ObjexxFCL::fmt;
		out
				<< "\n pos  count   best     worst  | pos  count   best    worst   | pos  count    best    worst  |\n";
		Size cnt = 0;
		for (Size i_pos = 1; i_pos <= storage_.size(); ++i_pos) {

			if (storage_[i_pos].size() <= 1) {
				out << I(4, i_pos) << "      0                   |";
			} else {
				out << I(4, i_pos) << " " << I(6, storage_[i_pos].size())
						<< " ";
				out << F(8, 2, scoring->total_score(
						storage_[i_pos].peek_front().second)) << " ";
				out << F(8, 2, scoring->total_score(
						storage_[i_pos].peek_back().second)) << " |";
			}
			++cnt;
			if (cnt % 3 == 0)
				out << '\n';
		}
		out << std::endl;
	}

private:
	utility::vector1<LazySortedVector1<pair<FragmentCandidateOP,
			scores::FragmentScoreMapOP> , StrictWeakOrdering> > storage_;
};

} // picking
} // fragment
} // core


#endif /* INCLUDED_core_fragment_picking_BoundedCollector_HH */
