// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/fragment/picking/scores/RamaScore.hh
/// @brief  a base class for fragment scoring
/// @author Dominik Gront (dgront@chem.uw.edu.pl)

#ifndef INCLUDED_core_fragment_picking_scores_RamaScore_HH
#define INCLUDED_core_fragment_picking_scores_RamaScore_HH

// type headers
#include <core/types.hh>

// package headers
#include <core/fragment/picking/FragmentCandidate.fwd.hh>
#include <core/fragment/picking/FragmentPicker.hh>

#include <core/fragment/picking/scores/CachingScoringMethod.hh>
#include <core/fragment/picking/scores/FragmentScoreMap.hh>

#include <core/conformation/SecondaryStructure.fwd.hh>

namespace core {
namespace fragment {
namespace picking {
namespace scores {

/// @brief  RamaScore score counts identical residues
/// @detailed Resulting score is the number of identical residues
/// on corresponding positions in a vall fragment and a query sequence
class RamaScore: public CachingScoringMethod {
public:

	RamaScore(Size, Real, bool, std::string&,std::string);

	RamaScore(Size, Real, bool, std::string&, core::conformation::SecondaryStructureOP,std::string);

	void SetupRamaTables();

	void do_caching(VallChunkOP);
	void clean_up();

	bool score(FragmentCandidateOP f, FragmentScoreMapOP empty_map);
	bool cached_score(FragmentCandidateOP f, FragmentScoreMapOP empty_map);

	/// @brief prints a detailed explanation how a fragment score has been computed
	/// @detailed besides extensive output, the method should return the same result as score()
	bool describe_score(FragmentCandidateOP f, FragmentScoreMapOP empty_map,
			std::ostream& out);

	std::string & get_prediction_name() { return prediction_name_; }
private:
	Real minScoreAllowed_;
	std::string& query_;
	core::conformation::SecondaryStructureOP query_ss_;
	std::string prediction_name_; 

	std::string cached_scores_id_;

	//Sequence Specific Ramachandran Surfaces
	utility::vector1< utility::vector1< utility::vector1< Real > > > sequence_rama_tables_;

	utility::vector1< utility::vector1< Real > > scores_;

};

/// @brief  Maker class that produces a new RamaScore object
class MakeRamaScore: public MakeFragmentScoringMethod {
public:

	MakeRamaScore() :
		MakeFragmentScoringMethod("RamaScore") {
	}

	FragmentScoringMethodOP make(Size priority, Real lowest_acceptable_value, bool use_lowest,
			FragmentPickerOP picker, std::string prediction_id ) {

		if ( prediction_id == "" ) {
			core::conformation::SecondaryStructureOP default_ss = new core::conformation::SecondaryStructure;
			Size query_len = picker->get_query_seq_string().size();
			default_ss->extend( query_len );
			for ( Size i = 1; i <= query_len; ++i ) {
				default_ss->set_fractions(i, 0.333, 0.333, 0.334);
			}

			//			std::cout << "CO_PREDICTION_ID " << prediction_id << std::endl;
			std::string default_prediction_name("uniform_default");
			return (FragmentScoringMethodOP) new RamaScore(priority,
				lowest_acceptable_value, use_lowest, picker->get_query_seq_string(), default_ss,default_prediction_name);
		} else {

			//std::cout << "PREDICTION_ID " << prediction_id << std::endl;
			return (FragmentScoringMethodOP) new RamaScore(priority,
				lowest_acceptable_value, use_lowest, picker->get_query_seq_string(), picker->get_query_ss(prediction_id),prediction_id);
		}
	}
};

} // scores
} // picking
} // fragment
} // core


#endif /* INCLUDED_core_fragment_picking_scores_RamaScore_HH */
