// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/grid/Pockets/QuaternionFingerprint.cc
/// @brief  core::grid::Pockets::QuaternionFingerprint functions
/// @author Ragul Gowthaman

#include <iostream>
#include <iomanip>
#include <fstream>
#include <ostream>
#include <string>
#include <sstream>
#include <cmath>
#include <map>

// Protocol Headers
#include <numeric/constants.hh>
#include <core/grid/Pockets/QuaternionFingerprint.hh>
#include <core/grid/Pockets/PocketGrid.hh>

// Core Headers

#include <core/options/option.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/options/keys/fingerprint.OptionKeys.gen.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <core/id/AtomID_Map.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/Conformation.hh>
#include <core/chemical/AtomType.hh>
#include <core/types.hh>
#include <numeric/xyzMatrix.hh>



// Utility Headers

using namespace core;
using namespace core::scoring;
using namespace core::optimization;
using namespace std;


namespace core {
namespace grid {
namespace Pockets {



Quaternion Quaternion::get_conjugate() const{
	Quaternion conj_q;
	conj_q.q1(-q1_);
	conj_q.q2(-q2_);
	conj_q.q3(-q3_);
	conj_q.q4(q4_);
	return conj_q;
}



QuaternionFingerprintBase::QuaternionFingerprintBase () :
	ReferenceCount()
{
	origin_.zero();
}



void QuaternionFingerprintBase::print_to_file(std::string const & output_filename) const {
	Quaternion q_offset(0.,0.,0.,0.);
	return print_to_file( output_filename, q_offset);
}

void QuaternionFingerprintBase::print_to_file(std::string const & output_filename, Quaternion const & q_offset) const {

	std::filebuf f1;
	f1.open (output_filename.c_str(), std::ios::out);
	std::ostream os1(&f1);
	std::string f1_info;
	std::stringstream f1_tmp;

	f1_tmp<<"//"<<std::fixed<<std::setprecision(2)<< origin_.x() << "\t" <<std::fixed<<std::setprecision(2)<< origin_.y() << "\t"<<std::fixed<<std::setprecision(3)<< origin_.z() <<std::endl;
	f1_info += f1_tmp.str();
	f1_tmp.str(std::string());

	for (std::list<spherical_coor_pentet>::const_iterator fi = pentet_fingerprint_data_.begin(); fi != pentet_fingerprint_data_.end(); ++fi) {
		spherical_coor_pentet new_pentet;
		apply_rotation_to_point( *fi, q_offset, new_pentet );
		f1_tmp<<std::fixed<<std::setprecision(2)<< new_pentet.rho << "\t" <<std::fixed<<std::setprecision(2)<<new_pentet.q.q1() << "\t"<<std::fixed<<std::setprecision(3)<< new_pentet.q.q2()<< "\t" <<std::fixed<<std::setprecision(2)<<new_pentet.q.q3() << "\t"<<std::fixed<<std::setprecision(3)<< new_pentet.q.q4() <<std::endl;
		f1_info += f1_tmp.str();
		f1_tmp.str(std::string());
	}
	os1<<f1_info;
	f1.close();

	return;
}

void QuaternionFingerprintBase::print_to_pdb(std::string const & output_pdbname) const {
	Quaternion q_offset(0,0,0,0);
	return print_to_pdb( output_pdbname, q_offset);
}

void QuaternionFingerprintBase::print_to_pdb(std::string const & output_pdbname, Quaternion const & q_offset ) const {

	std::filebuf f2;
	f2.open (output_pdbname.c_str(), std::ios::out);
	std::ostream os2(&f2);
	std::string f2_info;
	std::stringstream f2_tmp;
	f2_tmp<<"HETATM   "<<std::setw(2)<<1<<"  C   ORI X   0    "<<std::setw(8)<<std::fixed<<std::setprecision(3)<<origin_.x()<<std::setw(8)<<std::fixed<<std::setprecision(3)<<origin_.y()<<std::setw(8)<<std::fixed<<std::setprecision(3)<<origin_.z()<<std::endl;
	f2_info += f2_tmp.str();
	f2_tmp.str(std::string());
	for (std::list<spherical_coor_pentet>::const_iterator pd = pentet_fingerprint_data_.begin(); pd != pentet_fingerprint_data_.end(); ++pd) {
		spherical_coor_pentet new_pentet;
		//apply_rotation_to_point( *pd, q_offset, new_pentet );
		numeric::xyzVector<core::Real> new_coor;
		convert_spherical_coor_pentet_to_cartesian( *pd, new_coor );
		new_coor += origin_;
		f2_tmp<<"HETATM   "<<std::setw(2)<<1<<"  C   MAP A   1    "<<std::setw(8)<<std::fixed<<std::setprecision(3)<<new_coor.x()<<std::setw(8)<<std::fixed<<std::setprecision(3)<<new_coor.y()<<std::setw(8)<<std::fixed<<std::setprecision(3)<<new_coor.z()<<std::endl;
		f2_info += f2_tmp.str();
		f2_tmp.str(std::string());
	}

	os2<<f2_info;
	f2.close();

	return;
}

void NonPlaidQuaternionFingerprint::setup_from_PlaidQuaternionFingerprint( PlaidQuaternionFingerprint const & pfp ) {
	origin_ = pfp.origin();
	std::copy (pfp.pentet_fingerprint_data().begin(),pfp.pentet_fingerprint_data().end(), pentet_fingerprint_data_.begin());
	return;
}
/*
void NonPlaidQuaternionFingerprint::setup_from_PocketGrid( PocketGrid const & pocket_grid ) {
	int clustNo=1;
	bool smallPocket;
	origin_.zero();
	int divcountr = 0;
	for (std::list<PCluster>::const_iterator cit=pocket_grid.clusters_.clusters_.begin(); cit != pocket_grid.clusters_.clusters_.end(); ++cit){
		if (cit->points_.size()*pow(pocket_grid.stepSize_,3)<pocket_grid.minPockSize_) smallPocket=true;
		else smallPocket=false;
			for (std::list<PCluster::Cxyz>::const_iterator pit=cit->points_.begin(); pit != cit->points_.end(); ++pit){
				if (smallPocket){
					if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_POCKET) continue;
					if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_SURF) continue;
					if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_BURIED) continue;
				}else{
					if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_POCKET) {
						origin_.x() += pit->x*pocket_grid.stepSize_+pocket_grid.xcorn_; origin_.y() += pit->y*pocket_grid.stepSize_+pocket_grid.ycorn_; origin_.z() += pit->z*pocket_grid.stepSize_+pocket_grid.zcorn_;
						divcountr++;
					}
					if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_SURF){
						origin_.x() += pit->x*pocket_grid.stepSize_+pocket_grid.xcorn_; origin_.y() += pit->y*pocket_grid.stepSize_+pocket_grid.ycorn_; origin_.z() += pit->z*pocket_grid.stepSize_+pocket_grid.zcorn_;
						divcountr++;
					}
					if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_BURIED){
						origin_.x() += pit->x*pocket_grid.stepSize_+pocket_grid.xcorn_; origin_.y() += pit->y*pocket_grid.stepSize_+pocket_grid.ycorn_; origin_.z() += pit->z*pocket_grid.stepSize_+pocket_grid.zcorn_;
						divcountr++;
					}
					if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_EDGE){
						origin_.x() += pit->x*pocket_grid.stepSize_+pocket_grid.xcorn_; origin_.y() += pit->y*pocket_grid.stepSize_+pocket_grid.ycorn_; origin_.z() += pit->z*pocket_grid.stepSize_+pocket_grid.zcorn_;
						divcountr++;
					}
				}
				if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.POCKET) continue;
				if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.PO_SURF) continue;
				if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.PO_BURIED) continue;
				if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.PO_EDGE) continue;
			}
			clustNo++;
	}
	origin_ /= divcountr;

	//generate egg_shell
	spherical_coor_pentet new_pentet;
	std::list<spherical_coor_pentet>::iterator it;
	pentet_fingerprint_data_.clear();

	bool smallpocket;
	core::Size searchxmin, searchxmax, searchymin, searchymax, searchzmin, searchzmax,xx,yy,zz,x,y,z;
	numeric::xyzVector<core::Real> coord;
	for (std::list<PCluster>::const_iterator cit=pocket_grid.clusters_.clusters_.begin(); cit != pocket_grid.clusters_.clusters_.end(); ++cit){
		if (cit->points_.size()*pow(pocket_grid.stepSize_,3)<pocket_grid.minPockSize_) smallpocket=true;
		else smallpocket=false;
		for (std::list<PCluster::Cxyz>::const_iterator pit=cit->points_.begin(); pit != cit->points_.end(); ++pit){
			if (smallpocket)  continue;
				if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_POCKET) continue;
				if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_SURF) continue;
				if (pocket_grid.grid_[pit->x][pit->y][pit->z]==pocket_grid.TP_BURIED){
					coord.x() = (pit->x*pocket_grid.stepSize_+pocket_grid.xcorn_);
					coord.y() = (pit->y*pocket_grid.stepSize_+pocket_grid.ycorn_);
					coord.z() = (pit->z*pocket_grid.stepSize_+pocket_grid.zcorn_);
					if (pit->x != 0) {
						searchxmin=pit->x-1;
					}else{
						searchxmin=pit->x;
					}
					if (pit->x != pocket_grid.xdim_-1) {
						searchxmax=pit->x+1;
					}else{
						searchxmax=pit->x;
					}
					if (pit->y != 0) {
						searchymin=pit->y-1;
					}else{
						searchymin=pit->y;
					}
					if (pit->y != pocket_grid.ydim_-1) {
						searchymax=pit->y+1;
					}else{
						searchymax=pit->y;
					}
					if (pit->z != 0) {
						searchzmin=pit->z-1;
					}else{
						searchzmin=pit->z;
					}
					if (pit->z != pocket_grid.zdim_-1) {
						searchzmax=pit->z+1;
					}else{
						searchzmax=pit->z;
					}
					bool found=false;
					for (xx = searchxmin; ( xx<=searchxmax) && ! found; xx++){
						for (yy = searchymin; ( yy<=searchymax) && ! found; yy++){
							for (zz = searchzmin; ( zz<=searchzmax) && ! found; zz++){
								if (xx==pit->x && yy==pit->y && zz==pit->z) continue;
								if ( (pocket_grid.grid_[xx][yy][zz] == pocket_grid.HSURFACE) || (pocket_grid.grid_[xx][yy][zz] == pocket_grid.PSURFACE) || (pocket_grid.grid_[xx][yy][zz] == pocket_grid.T_SURFACE)) {
									found = true;
									coord -= origin_;
									convert_cartesian_to_spherical_coor_pentet( coord, new_pentet );
									pentet_fingerprint_data_.push_back(new_pentet);
								}
							}
						}
					}
				}
		}
	}
	//End egg_shell
	return;
}


void NonPlaidQuaternionFingerprint::setup_from_file(std::string const & input_filename) {

	ifstream inFile(input_filename.c_str());

	if (!inFile) {
		std::cout<< "Can't open input file " << input_filename << std::endl;
		exit(1);
	}

	std::string lineread;
	std::string Line;
	std::string Field;

	spherical_coor_pentet new_pentet;
	std::list<spherical_coor_pentet>::iterator it;
	pentet_fingerprint_data_.clear();

	while (std::getline(inFile, lineread)) {

		std::stringstream sss(lineread);
		std::string Pock_string_phi, Pock_string_psi, Pock_string_rho;
		core::Real Pock_real_phi, Pock_real_psi, Pock_real_rho;

		//parse COM values from line starting with "//"
		if (lineread[0] == '/' && lineread[1] == '/') {
			lineread.erase(0,2);
			std::stringstream com_line(lineread);
			std::getline(com_line, Pock_string_phi, '\t');
			origin_.x() = atof(Pock_string_phi.c_str());
			std::getline(com_line, Pock_string_psi, '\t');
			origin_.y() = atof(Pock_string_psi.c_str());
			std::getline(com_line, Pock_string_rho, '\t');
			origin_.z() = atof(Pock_string_rho.c_str());
			//std::cout<<"setupfromfile"<< " " <<origin_.x()<<" "<<origin_.y()<<" "<<origin_.z()<<std::endl;
				continue;
		}

		std::getline(sss, Pock_string_phi, '\t');
		Pock_real_phi = atof(Pock_string_phi.c_str());
		std::getline(sss, Pock_string_psi, '\t');
		Pock_real_psi = atof(Pock_string_psi.c_str());
		std::getline(sss, Pock_string_rho, '\t');
		Pock_real_rho = atof(Pock_string_rho.c_str());

		new_pentet.phi = Pock_real_phi;
		new_pentet.psi = Pock_real_psi;
		new_pentet.rho = Pock_real_rho;
		pentet_fingerprint_data_.push_back(new_pentet);

	}
	inFile.close();
	return;
}
	*/

void PlaidQuaternionFingerprint::build_from_pose(core::pose::Pose const & input_pose) {

  using namespace core::options;
	core::Real const radius_scale = option[ OptionKeys::fingerprint::atom_radius_scale ];
	core::Real const atom_buffer = option[ OptionKeys::fingerprint::atom_radius_buffer ];


	origin_.zero();

	for (core::Real a = 1.0; a > -1.049; a+= -0.5){
		q_vector.push_back (a);
	}

	core::Size const vector_length = q_vector.size();

	RhoVector.clear();
	RhoVector.resize( vector_length );

	for (Size k = 0; k < vector_length; ++k){
		(RhoVector[k]).resize( vector_length );
		for (Size l = 0; l < vector_length; ++l){
			(RhoVector[k][l]).resize( vector_length );
			for (Size m = 0; m < vector_length; ++m){
				(RhoVector[k][l][m]).resize( vector_length, -1.0 );
			}
		}
	}

	core::Size lig_res_num = 0;
	for ( int j = 1, resnum = input_pose.total_residue(); j <= resnum; ++j ) {
		if (!input_pose.residue(j).is_protein()){
			lig_res_num = j;
			break;
		}
	}

	if (lig_res_num != 0){
		conformation::Residue const & curr_rsd=input_pose.conformation().residue(lig_res_num);

		for(Size i = 1, i_end = curr_rsd.nheavyatoms(); i <= i_end; ++i) {
			origin_.x() +=  curr_rsd.atom(i).xyz()(1);
			origin_.y() +=  curr_rsd.atom(i).xyz()(2);
			origin_.z() +=  curr_rsd.atom(i).xyz()(3);
		}
		origin_ /= curr_rsd.nheavyatoms();

		std::cout<<"Ligand COM"<<" "<<origin_.x()<<" "<<origin_.y()<<" "<<origin_.z()<<std::endl;

		core::Real atomX,atomY,atomZ,atom_radius;
		Quaternion qAngle;
		int nest_k,nest_l,nest_m,nest_n;

		for(Size i = 1, i_end = curr_rsd.nheavyatoms(); i <= i_end; ++i) {
			atomX = 0;  atomY = 0;   atomZ = 0;   atom_radius = 0;
			atomX =  curr_rsd.atom(i).xyz()(1)-origin_.x();
			atomY =  curr_rsd.atom(i).xyz()(2)-origin_.y();
			atomZ =  curr_rsd.atom(i).xyz()(3)-origin_.z();
			//atom radius * 0.9 to shrink a little to match with protein surface
			atom_radius = ( curr_rsd.atom_type(i).lj_radius() - atom_buffer ) * radius_scale;
			if ( atom_radius < 0. ) continue;

			for (Size k = 0; k < q_vector.size(); ++k){
				for (Size l = 0; l < q_vector.size(); ++l){
					for (Size m = 0; m < q_vector.size(); ++m){
						for (Size n = 0; n < q_vector.size(); ++n){

							qAngle.q1(q_vector[k]);
							qAngle.q2(q_vector[l]);
							qAngle.q3(q_vector[m]);
							qAngle.q4(q_vector[n]);

							nest_k = k;
							nest_l = l;
							nest_m = m;
							nest_n = n;
							std::cout<<"vectorvals "<<" "<<nest_k<< " "<<nest_l<<" "<<nest_m<<" "<<nest_n<<std::endl;
							std::cout<<"vectorklmn "<<" "<<q_vector[k]<< " "<<q_vector[l]<<" "<<q_vector[m]<<" "<<q_vector[n]<<std::endl;
							std::cout<<"Ligand "<<" "<<qAngle.q1()<< " "<<qAngle.q2()<<" "<<qAngle.q3()<<" "<<qAngle.q4()<<std::endl;
							Find_Intersect(qAngle,atomX,atomY,atomZ,atom_radius,nest_k,nest_l,nest_m,nest_n);
						}
					}
				}
			}
			std::cout<<" end "<<std::endl;
		}

		spherical_coor_pentet p_pentet;
		std::list<spherical_coor_pentet>::iterator it;
		pentet_fingerprint_data_.clear();


			for (Size k = 0; k < q_vector.size(); ++k){
				for (Size l = 0; l < q_vector.size(); ++l){
					for (Size m = 0; m < q_vector.size(); ++m){
						for (Size n = 0; n < q_vector.size(); ++n){

							if (RhoVector[k][l][m][n] > 0.) {
								p_pentet.q.q1(q_vector[k]);
								p_pentet.q.q2(q_vector[l]);
								p_pentet.q.q3(q_vector[m]);
								p_pentet.q.q4(q_vector[n]);
								p_pentet.rho = RhoVector[k][l][m][n];
								pentet_fingerprint_data_.push_back(p_pentet);
							}

						}
					}
				}
			}
	}
}

void PlaidQuaternionFingerprint::Find_Intersect(Quaternion const & qAngle, core::Real const & atomX, core::Real const & atomY, core::Real const & atomZ, core::Real const & atom_radius, int const & nest_k, int const & nest_l, int const & nest_m, int const & nest_n){

	numeric::xyzVector<core::Real> xyz_vec;
	numeric::xyzVector<core::Real> unit_vec;
	unit_vec.x() = 1.0;
	unit_vec.y() = 0.0;
	unit_vec.z() = 0.0;
	xyz_vec = apply_rotation_to_vector(qAngle, unit_vec);
	std::cout<<" xyzvec "<<xyz_vec.x()<< " " <<xyz_vec.y()<< " " << xyz_vec.z()<<std::endl;

	//Define line with two points: (dirX,dirY,dirZ) and Origin (0,0,0)
	//compute (dirX,dirY,dirZ) from phi/psi with randon large_dist, relative to Origin
	//Reference: http://local.wasp.uwa.edu.au/~pbourke/geometry/sphereline/
	core::Real large_dist(999.);
	xyz_vec *= large_dist;
	std::cout<<" mult large_dist "<<xyz_vec.x()<< " " <<xyz_vec.y()<< " " << xyz_vec.z()<<std::endl;
	core::Real dirX,dirY,dirZ,dot_direction;
	dirX = xyz_vec.x();
	dirY = xyz_vec.y();
	dirZ = xyz_vec.z();
	std::cout<<"dirxyz "<<dirX<< " " <<dirY<< " " << dirZ<<std::endl;
	// figure out whether vector points towards atom or away from atom
	dot_direction = (dirX*atomX) + (dirY*atomY) + (dirZ*atomZ);
	if ( dot_direction < 0.0 ){
		return;
	}

	// setup our quadratic equation
	core::Real a = (dirX*dirX) + (dirY*dirY) + (dirZ*dirZ);
	core::Real b = 2.0 * ( (dirX*(-atomX)) + (dirY*(-atomY)) + (dirZ*(-atomZ)) );
	core::Real c = atomX*atomX + atomY*atomY + atomZ*atomZ - (atom_radius * atom_radius);

	// test for intersection
	core::Real inside_sqrt = b * b - 4 * a * c;

	if ( std::abs(inside_sqrt) < 0.00001 ) {
		//      std::cout << "Line is tangent to atom\n" << std::endl;
		core::Real mu = -b / ( 2 * a);
		core::Real x = mu *dirX;
		core::Real y = mu *dirY;
		core::Real z = mu *dirZ;
		core::Real dist = sqrt( x*x + y*y + z*z );
		if (dist > RhoVector[nest_k][nest_l][nest_m][nest_n]){
			RhoVector[nest_k][nest_l][nest_m][nest_n] = dist;
		}
		std::cout << "Distance from Origin is " << dist << std::endl;
		return;
	} else if ( inside_sqrt < 0 ) {
		return;
	} else {
		std::cout << "Line intersects atom\n" << std::endl;
		core::Real mu1 = -(b-sqrt(inside_sqrt)) / ( 2 * a);
		core::Real mu2 = -(b+sqrt(inside_sqrt)) / ( 2 * a);
		core::Real x1 =  mu1 * dirX;
		core::Real y1 =  mu1 * dirY;
		core::Real z1 =  mu1 * dirZ;
		core::Real dist1 = sqrt( x1*x1 + y1*y1 + z1*z1 );
		core::Real x2 = mu2 * dirX;
		core::Real y2 = mu2 * dirY;
		core::Real z2 = mu2 * dirZ;
		core::Real dist2 = sqrt( x2*x2 + y2*y2 + z2*z2 );
		core::Real max_dist = dist1;
		if ( dist2 > max_dist ) max_dist = dist2;
		if ((max_dist == dist1)){
			if (dist1 > RhoVector[nest_k][nest_l][nest_m][nest_n]){
				RhoVector[nest_k][nest_l][nest_m][nest_n] = dist1;
			}
		}
		else if ((max_dist == dist2)){
			if (dist2 > RhoVector[nest_k][nest_l][nest_m][nest_n]){
				RhoVector[nest_k][nest_l][nest_m][nest_n] = dist2;
			}
		}
		return;
	}
	return;
}


	/*
core::Real PlaidQuaternionFingerprint::fp_compare( QuaternionFingerprintBase const & fp ) const {
	return fp_compare( fp, 0., 0. );
}

core::Real PlaidQuaternionFingerprint::fp_compare( QuaternionFingerprintBase const & fp, Quaternion const & q_offset ) const {

  using namespace core::options;
	core::Real deviation_cutoff = option[ OptionKeys::fingerprint::dist_deviation_cutoff ];
	core::Real rho_weight = option[ OptionKeys::fingerprint::nonplaid_rho_weight ];


	// Loop over 1D list from fp
	// Get the two angles for this fp point
	// For these two angles, interpolate using the 2D matrix of "self" to predict the distance for these angles
	// Subtract the "predicted" distance from the distance of the fp point
	// Square this, add to running sum, move on to next fp point

	core::Real Total_score = 0, packing_score = 0, steric_score = 0;
	core::Size npf_points = fp.pentet_fingerprint_data().size();
	for (std::list<spherical_coor_pentet>::const_iterator ni = fp.pentet_fingerprint_data().begin(); ni != fp.pentet_fingerprint_data().end(); ++ni) {

		spherical_coor_pentet new_pentet;
		apply_rotation_to_point( *ni, q_offset, new_pentet );

		core::Real non_plaid_rho = new_pentet.rho;
		core::Real non_plaid_psi = new_pentet.psi;
		core::Real non_plaid_phi = new_pentet.phi;
		correct_phi_psi( non_plaid_phi, non_plaid_psi );

		//std::cout<<" phi "<<non_plaid_phi<<" psi "<<non_plaid_psi<<" rho "<<non_plaid_rho<<std::endl;

		assert( non_plaid_psi > -180.00001 );
		assert( non_plaid_psi < 180.00001 );
		assert( non_plaid_phi > -0.00001 );
		assert( non_plaid_phi < 180.00001 );

		std::pair<core::Real, core::Real> plaid_phi_range;
		std::pair<core::Real, core::Real> plaid_psi_range;

		//Find nearest angles in pf for a given npf angle

		core::Size plaid_phi_pos1, plaid_phi_pos2;
		std::vector<core::Real>::const_iterator phi_it = std::lower_bound(PhiVector.begin(), PhiVector.end(), non_plaid_phi);
		if (phi_it == PhiVector.end()){
			plaid_phi_range = std::make_pair(PhiVector.back(), PhiVector.back());
			plaid_phi_pos1 = std::distance( PhiVector.begin() , (phi_it - 1) );
			plaid_phi_pos2 = std::distance( PhiVector.begin() , (phi_it) );
		} else if (phi_it == PhiVector.begin()){
			plaid_phi_range =  std::make_pair(PhiVector.front(), PhiVector.front());
			plaid_phi_pos1 = std::distance( PhiVector.begin() , (phi_it - 1) );
			plaid_phi_pos2 = std::distance( PhiVector.begin() , (phi_it) );
		} else {
			plaid_phi_range = std::make_pair(*(phi_it - 1), *(phi_it));
			plaid_phi_pos1 = std::distance( PhiVector.begin() , (phi_it - 1) );
			plaid_phi_pos2 = std::distance( PhiVector.begin() , (phi_it) );
		}

		//Find corresponding index values for nearest angles
		plaid_phi_pos1 = std::distance( PhiVector.begin() , (phi_it - 1) );
		plaid_phi_pos2 = std::distance( PhiVector.begin() , (phi_it) );

		core::Size plaid_psi_pos1, plaid_psi_pos2;
		std::vector<core::Real>::const_iterator psi_it = std::lower_bound(PsiVector.begin(), PsiVector.end(), non_plaid_psi);
		if (psi_it == PsiVector.end()){
			plaid_psi_range = std::make_pair(PsiVector.back(), PsiVector.back());
			plaid_psi_pos1 = std::distance(PsiVector.begin() , (psi_it - 1));
			plaid_psi_pos2 = std::distance(PsiVector.begin() , (psi_it));
		} else if (psi_it == PsiVector.begin()){
			plaid_psi_range =  std::make_pair(PsiVector.front(), PsiVector.front());
			plaid_psi_pos1 = std::distance(PsiVector.begin() , (psi_it - 1));
			plaid_psi_pos2 = std::distance(PsiVector.begin() , (psi_it));
		} else {
			plaid_psi_range = std::make_pair(*(psi_it - 1), *(psi_it));
			plaid_psi_pos1 = std::distance(PsiVector.begin() , (psi_it - 1));
			plaid_psi_pos2 = std::distance(PsiVector.begin() , (psi_it));
		}

		core::Real plaid_phi_first = plaid_phi_range.first;
		core::Real plaid_phi_second = plaid_phi_range.second;
		core::Real plaid_psi_first = plaid_psi_range.first;
		core::Real plaid_psi_second = plaid_psi_range.second;

		//std::cout << "The nearest phi  angles of "<< non_plaid_phi <<" are "<< plaid_phi_range.first << " and " << plaid_phi_range.second <<std::endl;
		//std::cout << "The nearest phi index of "<< non_plaid_phi <<" are "<< plaid_phi_pos1 << " and " << plaid_phi_pos2 <<std::endl;
		//std::cout << "The nearest psi angles of "<< non_plaid_psi <<" are "<< plaid_psi_range.first << " and " << plaid_psi_range.second <<std::endl;
		//std::cout << "The nearest psi index of "<< non_plaid_psi <<" are "<< plaid_psi_pos1 << " and " << plaid_psi_pos2 <<std::endl;

		core::Real dist_deviation, add_weight;
		core::Real predicted_rho = Interpolate(plaid_phi_pos1, plaid_phi_pos2, plaid_psi_pos1, plaid_psi_pos2, non_plaid_phi, non_plaid_psi);
		dist_deviation = std::abs( non_plaid_rho - predicted_rho );
		if (predicted_rho > non_plaid_rho) {
			// penalize steric clash
			steric_score += rho_weight * dist_deviation;
		} else {
			// if no steric clash and within a threshold, give a bonus
			if (dist_deviation > deviation_cutoff) {
				dist_deviation = deviation_cutoff;
			}
			packing_score += dist_deviation;
		}
		//std::cout <<" actual rho "<< non_plaid_rho <<" predicted rho "<< predicted_rho<<" score_contribution "<< score_weight*dist_deviation <<std::endl;
	}

	Total_score = (steric_score + packing_score)/npf_points;
	//std::cout <<" steric_score "<< steric_score <<" packing_score "<< packing_score <<" Total_score "<< Total_score <<std::endl;
	return Total_score;
}

core::Real PlaidQuaternionFingerprint::Interpolate(core::Real const & plaid_phi_pos1, core::Real const & plaid_phi_pos2, core::Real const & plaid_psi_pos1, core::Real const & plaid_psi_pos2, core::Real const & non_plaid_phi, core::Real const & non_plaid_psi) const {

	core::Real plaid_rho1, plaid_rho2, plaid_rho3, plaid_rho4, Interpolated_rho;
	plaid_rho1 = RhoVector[plaid_phi_pos1][plaid_psi_pos1];
	plaid_rho2 = RhoVector[plaid_phi_pos2][plaid_psi_pos1];
	plaid_rho3 = RhoVector[plaid_phi_pos1][plaid_psi_pos2];
	plaid_rho4 = RhoVector[plaid_phi_pos2][plaid_psi_pos2];
	//std::cout<<"four rhos "<<plaid_rho1<<" "<<plaid_rho2<<" "<<plaid_rho3<<" "<<plaid_rho4<<std::endl;

	//bilinear interpolation Ref:Wikipedia
	Interpolated_rho = ((plaid_rho1/((PhiVector[plaid_phi_pos2]-PhiVector[plaid_phi_pos1])*(PsiVector[plaid_psi_pos2]-PsiVector[plaid_psi_pos1])))*(PhiVector[plaid_phi_pos2]-non_plaid_phi)*(PsiVector[plaid_psi_pos2]-non_plaid_psi)) + ((plaid_rho2/((PhiVector[plaid_phi_pos2]-PhiVector[plaid_phi_pos1])*(PsiVector[plaid_psi_pos2]-PsiVector[plaid_psi_pos1])))*(non_plaid_phi - PhiVector[plaid_phi_pos1])*(PsiVector[plaid_psi_pos2]-non_plaid_psi)) + ((plaid_rho3/((PhiVector[plaid_phi_pos2]-PhiVector[plaid_phi_pos1])*(PsiVector[plaid_psi_pos2]-PsiVector[plaid_psi_pos1])))*(PhiVector[plaid_phi_pos2]-non_plaid_phi)*(non_plaid_psi-PsiVector[plaid_psi_pos1])) + ((plaid_rho4/((PhiVector[plaid_phi_pos2]-PhiVector[plaid_phi_pos1])*(PsiVector[plaid_psi_pos2]-PsiVector[plaid_psi_pos1])))*(non_plaid_phi-PhiVector[plaid_phi_pos1])*(non_plaid_psi-PsiVector[plaid_psi_pos1]));

	return Interpolated_rho;
}


	core::Real PlaidQuaternionFingerprint::find_optimal_rotation( QuaternionFingerprintBase const & fp, core::Real const & q_increment, Quaternion & optimal_q ) const {

	core::Real curr_score;
	core::Real best_score = std::numeric_limits<core::Real>::max();
	core::Size phi_steps = core::Size ( 360. / q_increment );
	core::Size psi_steps = core::Size ( 360. / q_increment );

	core::Real curr_phi=0.;
  for (core::Size i = 0; i <= phi_steps; ++i ){
		core::Real curr_psi=0.;
		for (core::Size j = 0; j <= psi_steps; ++j ){
			core::Real curr_score = fp_compare( fp, curr_phi, curr_psi );
			if ( curr_score < best_score ) {
				best_score = curr_score;
				optimal_q = curr_q;
			}
			curr_psi += q_increment;
		}
		curr_phi += q_increment;
	}
	return best_score;

}

void NonPlaidQuaternionFingerprint::apply_rotation( Quaternion const & q_offset ){
  for (std::list<spherical_coor_pentet>::iterator pd = pentet_fingerprint_data_.begin(); pd != pentet_fingerprint_data_.end(); ++pd) {
		spherical_coor_pentet orig_pentet = *pd;
		apply_rotation_to_point( orig_pentet, q_offset, *pd );
	}
}

void NonPlaidQuaternionFingerprint::move_origin( numeric::xyzVector<core::Real> const & new_origin ){

	numeric::xyzVector<core::Real> fp_coor;
  for (std::list<spherical_coor_pentet>::iterator pd = pentet_fingerprint_data_.begin(); pd != pentet_fingerprint_data_.end(); ++pd) {
		// find cartesian coors relative to old origin
		convert_spherical_coor_pentet_to_cartesian( *pd, fp_coor );
		// move cartesian coors from old origin to new origin
    fp_coor += origin_ - new_origin;
		// convert from cartesian coors to polar
		convert_cartesian_to_spherical_coor_pentet( fp_coor, *pd );
	}
	origin_ = new_origin;
}
	*/

	//Ragul Fill ths
void apply_rotation_to_point(spherical_coor_pentet const & orig_pentet, Quaternion const & q_offset, spherical_coor_pentet & new_pentet ) {
	Quaternion conj_q_offset;
	conj_q_offset = q_offset.get_conjugate();
	new_pentet = orig_pentet;
	return;
}



numeric::xyzVector<core::Real> apply_rotation_to_vector(Quaternion const & qrot, numeric::xyzVector<core::Real> const & vec){
	numeric::xyzVector<core::Real> rot_vec;
	Quaternion conj_qrot, vecQuat, resQuat;
	vecQuat.q1(vec.x());
	vecQuat.q2(vec.y());
	vecQuat.q3(vec.z());
	vecQuat.q4(0.0);
	conj_qrot = qrot.get_conjugate();
	resQuat = multiply_Quaternions(vecQuat,conj_qrot);
	resQuat = multiply_Quaternions(qrot,resQuat);
	rot_vec.x() = resQuat.q1();
	rot_vec.y() = resQuat.q2();
	rot_vec.z() = resQuat.q3();
	return rot_vec;
}

void convert_spherical_coor_pentet_to_cartesian( spherical_coor_pentet const & pentet, numeric::xyzVector<core::Real> & coord ) {

	numeric::xyzVector<core::Real> unit_vec;
	unit_vec.x() = 1.0;
	unit_vec.y() = 0.0;
	unit_vec.z() = 0.0;
	coord = get_matrix(pentet) * unit_vec;
}


numeric::xyzVector<core::Real> get_cartesian(Quaternion const & quat){

	numeric::xyzVector<core::Real> unit_vec, coord;
	unit_vec.x() = 1.0;
	unit_vec.y() = 0.0;
	unit_vec.z() = 0.0;
	coord = get_matrix(quat) * unit_vec;
	return coord;
}

numeric::xyzMatrix<core::Real> get_matrix( spherical_coor_pentet const & pentet){
	numeric::xyzMatrix<core::Real> rot;

	core::Real x2 = pentet.q.q1() * pentet.q.q1();
	core::Real y2 = pentet.q.q2() * pentet.q.q2();
	core::Real z2 = pentet.q.q3() * pentet.q.q3();
	core::Real w2 = pentet.q.q4() * pentet.q.q4();
	core::Real xy = pentet.q.q1() * pentet.q.q2();
	core::Real xz = pentet.q.q1() * pentet.q.q3();
	core::Real yz = pentet.q.q2() * pentet.q.q3();
	core::Real wx = pentet.q.q4() * pentet.q.q1();
	core::Real wy = pentet.q.q4() * pentet.q.q2();
	core::Real wz = pentet.q.q4() * pentet.q.q3();

	rot.xx(w2+x2-y2-z2);
	rot.xy(2*xy+2*wz);
	rot.xz(2*xz-2*wy);
	rot.yx(2*xy-2*wz);
	rot.yy(w2-x2+y2-z2);
	rot.yz(2*yz-2*wx);
	rot.zx(2*xz-2*wy);
	rot.zy(2*yz-2*wx);
	rot.zz(w2-x2-y2+z2);
	return rot;
	}

numeric::xyzMatrix<core::Real> get_matrix(Quaternion const & quat){
	numeric::xyzMatrix<core::Real> rot;

	core::Real x2 = quat.q1() * quat.q1();
	core::Real y2 = quat.q2() * quat.q2();
	core::Real z2 = quat.q3() * quat.q3();
	core::Real w2 = quat.q4() * quat.q4();
	core::Real xy = quat.q1() * quat.q2();
	core::Real xz = quat.q1() * quat.q3();
	core::Real yz = quat.q2() * quat.q3();
	core::Real wx = quat.q4() * quat.q1();
	core::Real wy = quat.q4() * quat.q2();
	core::Real wz = quat.q4() * quat.q3();

	rot.xx(w2+x2-y2-z2);
	rot.xy(2*xy+2*wz);
	rot.xz(2*xz-2*wy);
	rot.yx(2*xy-2*wz);
	rot.yy(w2-x2+y2-z2);
	rot.yz(2*yz-2*wx);
	rot.zx(2*xz-2*wy);
	rot.zy(2*yz-2*wx);
	rot.zz(w2-x2-y2+z2);
	return rot;
	}

Quaternion multiply_Quaternions(Quaternion const & quat1, Quaternion const & quat2) {
	Quaternion quat_product;

	core::Real x1 = quat1.q1();
	core::Real y1 = quat1.q2();
	core::Real z1 = quat1.q3();
	core::Real w1 = quat1.q4();

	core::Real x2 = quat2.q1();
	core::Real y2 = quat2.q2();
	core::Real z2 = quat2.q3();
	core::Real w2 = quat2.q4();

	quat_product.q1( (w1 * w2) - (x1*x2) - (y1*y2) -(z1*z2) );
	quat_product.q2( (w1 * x2) + (x1*w2) + (y1*z2) -(z1*y2) );
	quat_product.q3( (w1 * y2) - (x1*z2) - (y1*w2) -(z1*x2) );
	quat_product.q4( (w1 * z2) - (x1*y2) - (y1*x2) -(z1*w2) );
	return quat_product;
}

core::Real get_length(Quaternion const & quat){
	return sqrt(quat.q1()*quat.q1() + quat.q2()*quat.q2() + quat.q3()*quat.q3() + quat.q4()*quat.q4());
}


Quaternion normalaize_Quaternion(Quaternion const & quat){
	Quaternion normalized_quat;
	core::Real length = get_length(quat);
	normalized_quat.q1(quat.q1()/length);
	normalized_quat.q2(quat.q2()/length);
	normalized_quat.q3(quat.q3()/length);
	normalized_quat.q4(quat.q4()/length);
	return normalized_quat;
}




} // Pockets
} // grid
} // core



