// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/io/database/open.cc
/// @brief  Functions for opening database files
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


// Unit headers
#include <core/io/database/open.hh>

// Project headers
#include <core/types.hh>
#include <core/options/option.hh>

// Utility headers
#include <utility/exit.hh>
#include <utility/io/izstream.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/file/PathName.hh>

// C++ headers
#include <cstdlib>
#include <iostream>

#include <core/util/Tracer.hh>
using core::util::T;

// option key includes

#include <core/options/keys/in.OptionKeys.gen.hh>


using core::util::Error;
using core::util::Warning;

namespace core {
namespace io {
namespace database {

static core::util::Tracer TR("core.io.database");

/// @brief Open a database file on a provided stream
bool
open(
	utility::io::izstream & db_stream,
	std::string const & db_file
)
{
	db_stream.close();
	db_stream.clear();
	db_stream.open( full_name( db_file ) );

	if ( db_stream ) { // Open succeeded
		TR << "Database file opened: " << db_stream.filename() << std::endl;
		return true;
	} else { // Open failed
		Error() << "ERROR: Database file open failed for: " << db_stream.filename() << std::endl;
		std::exit( EXIT_FAILURE );
		db_stream.close();
		db_stream.clear();
		return false;
	}
}


/// @brief Full-path database file name
std::string
full_name(
	std::string const & db_file,
	bool warn // = true
)
{
	using namespace core::options;
	using namespace core::options::OptionKeys;

	for(Size i = 1, i_end = option[ in::path::database ]().size(); i <= i_end; ++i) {
		std::string fname = option[ in::path::database ](i).name() + db_file;
		if( utility::file::file_exists(fname) || utility::file::file_exists(fname + ".gz") ) return fname;
	}
	// Don't exit -- sometimes caller wants to check if file exists (e.g. Dunbrack .bin file)
	//utility_exit_with_message("Unable to locate database file "+db_file);
	if (warn) Warning() << "Unable to locate database file " << db_file << std::endl;
	return option[ in::path::database ](1).name() + db_file;
}


} // namespace database
} // namespace io
} // namespace core
