// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/operation/TaskOperations.cc
/// @brief
/// @author Andrew Leaver-Fay (leaverfa@email.unc.edu)

// Unit Headers
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/operation/TaskOperationCreators.hh>

// Project Headers
#include <core/pack/task/PackerTask.hh>
#include <core/pack/rotamer_set/RotamerCouplings.hh>
#include <core/pack/rotamer_set/RotamerSetOperation.hh>
#include <core/pack/task/RotamerSampleOptions.hh>
#include <core/options/option.hh>
#include <core/util/Tracer.hh>
#include <boost/foreach.hpp>
#define foreach BOOST_FOREACH

// Utility Headers
#include <utility/exit.hh>
#include <utility/Tag/Tag.hh>

// option key includes
#include <core/options/keys/packing.OptionKeys.gen.hh>

namespace core {
namespace pack {
namespace task {
namespace operation {

using util::t_warning;
using util::t_info;
using util::t_debug;
static util::Tracer TR("core.pack.task.operation.TaskOperations",t_info);
using namespace utility::Tag;

/// BEGIN RestrictToRepacking

RestrictToRepacking::~RestrictToRepacking() {}

TaskOperationOP RestrictToRepackingCreator::create_task_operation() const
{
	return new RestrictToRepacking;
}

TaskOperationOP RestrictToRepacking::clone() const
{
	return new RestrictToRepacking( *this );
}

void
RestrictToRepacking::apply( pose::Pose const &, PackerTask & task ) const
{
	task.restrict_to_repacking();
}

void
RestrictToRepacking::parse_tag( TagPtr )
{}

/// BEGIN RestrictResidueToRepacking
RestrictResidueToRepacking::~RestrictResidueToRepacking() {}

TaskOperationOP RestrictResidueToRepackingCreator::create_task_operation() const
{
	return new RestrictResidueToRepacking;
}

TaskOperationOP RestrictResidueToRepacking::clone() const
{
	return new RestrictResidueToRepacking( *this );
}

void
RestrictResidueToRepacking::apply( pose::Pose const &, PackerTask & task ) const
{
	for(utility::vector1< core::Size >::const_iterator it(residues_to_restrict_to_repacking_.begin()), end(residues_to_restrict_to_repacking_.end());
			it != end; ++it)
		{
			assert( *it );
			task.nonconst_residue_task(*it).restrict_to_repacking();
		}
	return;
}

void
RestrictResidueToRepacking::include_residue( core::Size resid ) { residues_to_restrict_to_repacking_.push_back(resid); }

void
RestrictResidueToRepacking::clear() { residues_to_restrict_to_repacking_.clear(); }

void
RestrictResidueToRepacking::parse_tag( TagPtr tag )
{
	include_residue( tag->getOption< core::Size >( "resnum", 0 ) );
}

/// BEGIN RestrictAbsentCanonicalAAS
RestrictAbsentCanonicalAAS::RestrictAbsentCanonicalAAS()
	:	parent(),
		resid_(1)
{
	keep_aas_.assign( chemical::num_canonical_aas, false );
}

RestrictAbsentCanonicalAAS::RestrictAbsentCanonicalAAS( core::Size resid, utility::vector1< bool > keep  )
	:
	parent()
{
	keep_aas( keep );
	include_residue( resid );
}

RestrictAbsentCanonicalAAS::~RestrictAbsentCanonicalAAS(){}

TaskOperationOP RestrictAbsentCanonicalAASCreator::create_task_operation() const
{
	return new RestrictAbsentCanonicalAAS;
}

TaskOperationOP RestrictAbsentCanonicalAAS::clone() const
{
	return new RestrictAbsentCanonicalAAS( *this );
}

void
RestrictAbsentCanonicalAAS::apply( pose::Pose const &, PackerTask & task ) const
{
	if( resid_ == 0 ){// restrict all residues
		for( core::Size i( 1 ); i <= task.total_residue(); ++i )
			task.nonconst_residue_task( i ).restrict_absent_canonical_aas( keep_aas_ );
	}
	else
		task.nonconst_residue_task( resid_ ).restrict_absent_canonical_aas( keep_aas_ );
}

void
RestrictAbsentCanonicalAAS::keep_aas( std::string const keep )
{
	using namespace chemical;
	runtime_assert( keep_aas_.size() == num_canonical_aas );
	utility::vector1< bool > canonical_aas_to_keep( num_canonical_aas, false );
	foreach( char const c, keep ){
		if ( oneletter_code_specifies_aa( c ) ) {
			canonical_aas_to_keep[ aa_from_oneletter_code( c ) ] = true;
		} else {
			TR << "aa letter " << c << " does not not correspond to a canonical AA"<<std::endl;
			utility_exit();
		}
	}
	keep_aas( canonical_aas_to_keep );
}

void
RestrictAbsentCanonicalAAS::keep_aas( utility::vector1< bool > const keep )
{
	runtime_assert( keep.size() == chemical::num_canonical_aas );
	keep_aas_ = keep;
}

void
RestrictAbsentCanonicalAAS::include_residue( core::Size const resid )
{
	resid_ = resid;
}

void
RestrictAbsentCanonicalAAS::parse_tag( TagPtr tag )
{
	include_residue( tag->getOption< core::Size >( "resnum", 0 ) );
	keep_aas( tag->getOption< std::string >( "keep_aas" ) );
}


//BEGIN DisallowIfNonnative
DisallowIfNonnative::DisallowIfNonnative():
	disallowed_aas_ ( chemical::num_canonical_aas, false ),
	allowed_aas_( invert_vector( disallowed_aas_ ) )
{}

DisallowIfNonnative::DisallowIfNonnative( utility::vector1< bool > disallowed_aas):
	disallowed_aas_( disallowed_aas ),
	allowed_aas_( invert_vector(disallowed_aas) )
{}

DisallowIfNonnative::DisallowIfNonnative( utility::vector1< bool > disallowed_aas, utility::vector1<core::Size> res_selection):
	residue_selection_( res_selection ),
	disallowed_aas_( disallowed_aas ),
	allowed_aas_( invert_vector(disallowed_aas) )
{}

DisallowIfNonnative::~DisallowIfNonnative(){}

TaskOperationOP DisallowIfNonnativeCreator::create_task_operation() const
{
	return new DisallowIfNonnative;
}

TaskOperationOP DisallowIfNonnative::clone() const
{
	return new DisallowIfNonnative( *this );
}

void DisallowIfNonnative::clear(){
	allowed_aas_.clear();
	disallowed_aas_.clear();
	residue_selection_.clear();
}

//private function to invert disallowed aas into allowed aas
utility::vector1< bool >
DisallowIfNonnative::invert_vector( utility::vector1< bool > disallowed_aas){
	utility::vector1< bool > inverted_vec;
	for(core::Size ii=1; ii<=disallowed_aas_.size(); ii++ ){
		inverted_vec.push_back( ! disallowed_aas[ii] );
	}
	return inverted_vec;
}

void DisallowIfNonnative::apply( pose::Pose const &, PackerTask & task ) const
{
 	runtime_assert( allowed_aas_.size() == chemical::num_canonical_aas );
	if ( residue_selection_.empty() ){ //if no residue defined then do all residues
		for (core::Size ii = 1; ii<= task.total_residue(); ii++)
			task.nonconst_residue_task( ii ).restrict_nonnative_canonical_aas( allowed_aas_ );
	}
	else{  //if a residue is defined then do only on that residue
		for(core::Size jj = 1; jj <= residue_selection_.size(); jj++ )
			task.nonconst_residue_task( residue_selection_[jj] ).restrict_nonnative_canonical_aas( allowed_aas_ );
	}
}

//helper functions for DisallowIfNonnative
void
DisallowIfNonnative::disallow_aas( utility::vector1< bool > const & cannonical_disallowed ){
	runtime_assert( cannonical_disallowed.size() == chemical::num_canonical_aas );
	disallowed_aas_ = cannonical_disallowed;
	allowed_aas_ = invert_vector( disallowed_aas_ );
}
void DisallowIfNonnative::disallow_aas( std::string const & aa_string ){
	using namespace chemical;
	utility::vector1< bool > aa_vector ( chemical::num_canonical_aas, false );
	for ( std::string::const_iterator it( aa_string.begin() ), end( aa_string.end() );
				it != end; ++it ) {
		if ( oneletter_code_specifies_aa( *it ) ) {
			aa_vector[ aa_from_oneletter_code( *it ) ] = true;
		} else {
			std::ostringstream os;
			os << "aa letter " << *it << " does not not correspond to a canonical AA";
			utility_exit_with_message( os.str() );
		}
	}
	disallowed_aas_ = aa_vector;
	allowed_aas_ = invert_vector( disallowed_aas_ );
}
	//functions to restrict what residues are looked at by operation
	//selections are additive
void DisallowIfNonnative::restrict_to_residue( core::Size const & resid){
	residue_selection_.push_back( resid );
}
void DisallowIfNonnative::restrict_to_residue( utility::vector1< core::Size > const & residues){
	for(core::Size ii=1; ii<=residues.size(); ii++)
		residue_selection_.push_back( residues[ii] );
}

void DisallowIfNonnative::parse_tag( TagPtr tag )
{
	restrict_to_residue( tag->getOption< core::Size >( "resnum", 0 ) );
	disallow_aas( tag->getOption< std::string >( "disallow_aas" ) );
}

//BEGIN RotamerExplosion
RotamerExplosion::RotamerExplosion(){}

RotamerExplosion::RotamerExplosion( core::Size const resid, ExtraRotSample const sample_level, core::Size const chi ) :
	resid_( resid ),
	chi_( chi ),
	sample_level_( sample_level )
{}

RotamerExplosion::~RotamerExplosion() {}

TaskOperationOP RotamerExplosionCreator::create_task_operation() const
{
	return new RotamerExplosion;
}

TaskOperationOP RotamerExplosion::clone() const
{
	return( new RotamerExplosion( *this ) );
}

void
RotamerExplosion::apply( core::pose::Pose const &, PackerTask & task ) const
{
	ResidueLevelTask & restask( task.nonconst_residue_task( resid_ ) );
  if( chi_ > 0 ) restask.or_ex1_sample_level( sample_level_ );
	if( chi_ > 1 ) restask.or_ex2_sample_level( sample_level_ );
	if( chi_ > 2 ) restask.or_ex3_sample_level( sample_level_ );
	if( chi_ > 3 ) restask.or_ex4_sample_level( sample_level_ );
	restask.or_include_current( false );
}

void
RotamerExplosion::parse_tag( TagPtr tag )
{
	resid( tag->getOption< core::Size >( "resnum" ) );
	chi( tag->getOption< core::Size >( "chi" ) );
	sample_level( EX_THREE_THIRD_STEP_STDDEVS );// hardcoded and ugly, but this is probably too much to expect the user to set
}

void
RotamerExplosion::resid( core::Size const r )
{
	resid_ = r;
}

void
RotamerExplosion::chi( core::Size const c )
{
	chi_ = c;
}

void
RotamerExplosion::sample_level( ExtraRotSample const s )
{
	sample_level_ = s;
}


/// BEGIN InitializeFromCommandline

InitializeFromCommandline::~InitializeFromCommandline() {}

TaskOperationOP InitializeFromCommandlineCreator::create_task_operation() const
{
	return new InitializeFromCommandline;
}

TaskOperationOP InitializeFromCommandline::clone() const
{
	return new InitializeFromCommandline( *this );
}

void
InitializeFromCommandline::apply( pose::Pose const &, PackerTask & task ) const
{
	task.initialize_from_command_line();
}

/// BEGIN InitializeFromCommandline

InitializeExtraRotsFromCommandline::~InitializeExtraRotsFromCommandline() {}

TaskOperationOP InitializeExtraRotsFromCommandlineCreator::create_task_operation() const
{
	return new InitializeExtraRotsFromCommandline;
}

TaskOperationOP InitializeExtraRotsFromCommandline::clone() const
{
	return new InitializeExtraRotsFromCommandline( *this );
}

void
InitializeExtraRotsFromCommandline::apply( pose::Pose const &, PackerTask & task ) const
{
	task.initialize_extra_rotamer_flags_from_command_line();
}


/// BEGIN IncludeCurrent

IncludeCurrent::~IncludeCurrent() {}

TaskOperationOP IncludeCurrentCreator::create_task_operation() const
{
	return new IncludeCurrent;
}

TaskOperationOP IncludeCurrent::clone() const
{
	return new IncludeCurrent( *this );
}

void
IncludeCurrent::apply( pose::Pose const &, PackerTask & task ) const
{
	task.or_include_current(true);
}

/// BEGIN ReadResfile

ReadResfile::ReadResfile() : parent()
{
}

ReadResfile::ReadResfile( std::string const & filename )
	: parent(),
		resfile_filename_( filename )
{}

ReadResfile::~ReadResfile() {}

TaskOperationOP ReadResfileCreator::create_task_operation() const
{
	return new ReadResfile;
}

TaskOperationOP ReadResfile::clone() const
{
	return new ReadResfile( *this );
}

void
ReadResfile::apply( pose::Pose const &, PackerTask & task ) const
{
	using namespace options;
	using namespace OptionKeys;
	if ( resfile_filename_.empty() ) {
		task.read_resfile( option[ packing::resfile].value().at(1) );
	} else {
		task.read_resfile( resfile_filename_ );
	}
}

void
ReadResfile::default_filename()
{
	///.value().at(1) gets the value of the first resfile in the FileVector (like [1])
	resfile_filename_ = core::options::option[ core::options::OptionKeys::packing::resfile].value().at(1);
}

void
ReadResfile::filename( std::string const & filename )
{
	resfile_filename_ = filename;
}

std::string const & ReadResfile::filename() const
{
	return resfile_filename_;
}

void
ReadResfile::parse_tag( TagPtr tag )
{
	if ( tag->hasOption("filename") ) resfile_filename_ = tag->getOption<std::string>("filename");
	// special case: if "COMMANDLINE" string specified, use commandline option setting
	if ( resfile_filename_ == "COMMANDLINE" ) default_filename();
}


/// BEGIN SetRotamerCouplings

SetRotamerCouplings::SetRotamerCouplings()
{}

SetRotamerCouplings::~SetRotamerCouplings()
{}

SetRotamerCouplings::SetRotamerCouplings( SetRotamerCouplings const & src )
:
	parent(),
	rotamer_couplings_( src.rotamer_couplings_ )
{}

SetRotamerCouplings const &
SetRotamerCouplings::operator = ( SetRotamerCouplings const & rhs )
{
	rotamer_couplings_ = rhs.rotamer_couplings_;
	return *this;
}

TaskOperationOP SetRotamerCouplingsCreator::create_task_operation() const
{
	return new SetRotamerCouplings;
}

TaskOperationOP SetRotamerCouplings::clone() const
{
	return new SetRotamerCouplings( *this );
}

void
SetRotamerCouplings::apply( pose::Pose const &, PackerTask & task ) const
{
	task.rotamer_couplings( rotamer_couplings_ );
}

void
SetRotamerCouplings::set_couplings( rotamer_set::RotamerCouplingsOP couplings )
{
	rotamer_couplings_ = couplings;
}

/// BEGIN AppendRotamer

AppendRotamer::AppendRotamer()
	: rotamer_operation_(0)
{}

AppendRotamer::~AppendRotamer()
{}

AppendRotamer::AppendRotamer( rotamer_set::RotamerOperationOP rotamer_operation )
 : rotamer_operation_( rotamer_operation )
{}

AppendRotamer::AppendRotamer( AppendRotamer const & src )
: parent(), rotamer_operation_( src.rotamer_operation_ )
{}

TaskOperationOP AppendRotamerCreator::create_task_operation() const
{
	return new AppendRotamer;
}

TaskOperationOP AppendRotamer::clone() const
{
	return new AppendRotamer( *this );
}

void
AppendRotamer::apply( pose::Pose const &, PackerTask & task ) const
{
	task.append_rotamer_operation( rotamer_operation_ );
}

void
AppendRotamer::set_rotamer_operation(
	rotamer_set::RotamerOperationOP rotamer_operation
)
{
	rotamer_operation_ = rotamer_operation;
}


/// BEGIN AppendRotamerSet

AppendRotamerSet::AppendRotamerSet()
	: rotamer_set_operation_(0)
{}

AppendRotamerSet::~AppendRotamerSet()
{}

AppendRotamerSet::AppendRotamerSet( rotamer_set::RotamerSetOperationOP rotamer_set_operation )
 : rotamer_set_operation_( rotamer_set_operation )
{}

AppendRotamerSet::AppendRotamerSet( AppendRotamerSet const & src )
: parent(), rotamer_set_operation_( src.rotamer_set_operation_ )
{}

TaskOperationOP AppendRotamerSetCreator::create_task_operation() const
{
	return new AppendRotamerSet;
}

TaskOperationOP AppendRotamerSet::clone() const
{
	return new AppendRotamerSet( *this );
}

void
AppendRotamerSet::apply( pose::Pose const &, PackerTask & task ) const
{
	task.append_rotamerset_operation( rotamer_set_operation_ );
}

void
AppendRotamerSet::set_rotamer_set_operation(
	rotamer_set::RotamerSetOperationOP rotamer_set_operation
)
{
	rotamer_set_operation_ = rotamer_set_operation;
}



/// BEGIN PreserveCBeta

PreserveCBeta::~PreserveCBeta() {}

TaskOperationOP PreserveCBetaCreator::create_task_operation() const
{
	return new PreserveCBeta;
}

TaskOperationOP PreserveCBeta::clone() const
{
	return new PreserveCBeta( *this );
}

void
PreserveCBeta::apply( pose::Pose const &, PackerTask & task ) const
{
	task.or_preserve_c_beta( true );
}

/// BEGIN PreventRepacking
PreventRepacking::~PreventRepacking() {}

TaskOperationOP PreventRepackingCreator::create_task_operation() const
{
	return new PreventRepacking;
}

TaskOperationOP PreventRepacking::clone() const
{
	return new PreventRepacking( *this );
}

void
PreventRepacking::apply( pose::Pose const &, PackerTask & task ) const
{
	for(utility::vector1< core::Size >::const_iterator it(residues_to_prevent_.begin()), end(residues_to_prevent_.end());
			it != end; ++it)
		{task.nonconst_residue_task(*it).prevent_repacking();}
	return;
}

void
PreventRepacking::include_residue( core::Size resid ) { residues_to_prevent_.push_back(resid); }

void
PreventRepacking::clear() { residues_to_prevent_.clear(); }

void
PreventRepacking::parse_tag( TagPtr tag )
{
	include_residue( tag->getOption< core::Size >( "resnum", 0 ) );
}


// BEGIN RestrictYSDesign
RestrictYSDesign::~RestrictYSDesign() {}
RestrictYSDesign::RestrictYSDesign() : gly_switch_( false )  {}
RestrictYSDesign::RestrictYSDesign( RestrictYSDesign const & src ) :
	parent(), YSresids_( src.YSresids_ ), gly_switch_( src.gly_switch_ )
{}
RestrictYSDesign::RestrictYSDesign( utility::vector1< core::Size > const & resids ) {
	gly_switch_ = false;
	YSresids_ = resids;
}

void
RestrictYSDesign::apply( pose::Pose const &, PackerTask & task ) const {
	utility::vector1<bool> restrict_to_aa( 20, false );
	for( utility::vector1<core::Size>::const_iterator res_it=YSresids_.begin(); res_it!=YSresids_.end(); ++res_it ) {
		if( gly_switch_ ) restrict_to_aa[chemical::aa_from_name( "GLY" )] = true;;
		restrict_to_aa[chemical::aa_from_name( "TYR" )] = true;
		restrict_to_aa[chemical::aa_from_name( "SER" )] = true;
		task.nonconst_residue_task(*res_it).restrict_absent_canonical_aas( restrict_to_aa );
	}
}

TaskOperationOP RestrictYSDesignCreator::create_task_operation() const
{
	return new RestrictYSDesign;
}

TaskOperationOP RestrictYSDesign::clone() const {
	return new RestrictYSDesign( *this );
}

void
RestrictYSDesign::include_resid( core::Size const resid ) { YSresids_.push_back(resid); }

void
RestrictYSDesign::include_gly( bool const gly ) { gly_switch_ = gly; }


	//////////////////////////////////////////////////////
	// This class could easily be expanded to handle sample_level, etc.
	// Someone has probably already written this, and should replace this
	// dinky thing.
	ExtraRotamers::ExtraRotamers(){}

	ExtraRotamers::ExtraRotamers( core::Size const resid, core::Size const chi ):
		resid_( resid ),
		chi_( chi )
	{}

	ExtraRotamers::~ExtraRotamers() {}

	TaskOperationOP ExtraRotamersCreator::create_task_operation() const
	{
		return new ExtraRotamers;
	}

	TaskOperationOP ExtraRotamers::clone() const
	{
		return( new ExtraRotamers( *this ) );
	}

	void
	ExtraRotamers::apply( core::pose::Pose const &, PackerTask & task ) const
	{
		ResidueLevelTask & restask( task.nonconst_residue_task( resid_ ) );
		if( chi_ == 1 ) restask.or_ex1( true );
		if( chi_ == 2 ) restask.or_ex2( true );
		if( chi_ == 3 ) restask.or_ex3( true );
		if( chi_ == 4 ) restask.or_ex4( true );
	}


	//////////////////////////////////////////////////////
	// This class could easily be expanded ...
	// Someone has probably already written this, and should replace this
	// dinky thing.
	ExtraChiCutoff::ExtraChiCutoff() {}

	ExtraChiCutoff::ExtraChiCutoff( core::Size const resid, core::Size const extrachi_cutoff):
		resid_( resid ),
		extrachi_cutoff_( extrachi_cutoff )
	{}

	ExtraChiCutoff::~ExtraChiCutoff() {}

	TaskOperationOP ExtraChiCutoffCreator::create_task_operation() const
	{
		return new ExtraChiCutoff;
	}

	TaskOperationOP ExtraChiCutoff::clone() const
	{
		return( new ExtraChiCutoff( *this ) );
	}

	void
	ExtraChiCutoff::apply( core::pose::Pose const &, PackerTask & task ) const
	{
		task.nonconst_residue_task( resid_ ).and_extrachi_cutoff( extrachi_cutoff_ );
	}



} //namespace operation
} //namespace task
} //namespace pack
} //namespace core
