// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   core/scoring/constraints/PocketConstraint.cc
///
/// @brief
/// @author David Johnson


#include <core/scoring/constraints/PocketConstraint.hh>
#include <core/grid/Pockets/PocketGrid.hh>

#include <core/options/option.hh>
#include <core/options/keys/OptionKeys.hh>
#include <core/scoring/ScoreType.hh>
#include <core/pose/PDBInfo.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/constraints/XYZ_Func.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/Conformation.hh>
#include <core/chemical/AtomType.hh>
#include <core/util/Tracer.hh>
#include <string>
#include <ObjexxFCL/string.functions.hh>
#include <fstream>
#include <iostream>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
#include <core/options/keys/pocket_grid.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <sys/time.h>

namespace core {
namespace scoring {
namespace constraints {


static core::util::Tracer TR("core.scoring.constraints.PocketConstraint");

PocketConstraint::PocketConstraint(
    pose::Pose const & pose):
  Constraint( core::scoring::pocket_constraint )
{
  using namespace core::options;
  // for now, set the constraint to depend on ALL atom positions, ie. if ANYTHING moves we have to update the constraint
  // later, we could pre-define residues near the selected residue and make the constraint depend only on these
  // This is the residue we'll backrub around!!
  //int const central_relax_pdb_number = option[ OptionKeys::pocket_grid::central_relax_pdb_num ];
  std::string resid(option[ OptionKeys::pocket_grid::central_relax_pdb_num ]);
  int  central_relax_pdb_number;
  char chain = ' ';
  std::size_t fpos( resid.find(':') );
  if ( fpos != std::string::npos ) {
    central_relax_pdb_number = ObjexxFCL::int_of( resid.substr(0,fpos) );
    if (fpos != resid.size()-1 ) {
      chain = resid[ fpos+1 ];
    }
  } else {
    central_relax_pdb_number = ObjexxFCL::int_of( resid );
  }
  seqpos_ = 0;

  for ( int j = 1, resnum = pose.total_residue(); j <= resnum; ++j ) {
    if ( pose.pdb_info()->number(j) == central_relax_pdb_number ) {
      //seqpos_ = j;
      if (chain != ' '){
        if ( pose.pdb_info()->chain(j) == chain ) {
          seqpos_ = j;
        }
      }else{
        seqpos_ = j;
      }
    }
  }

	//  Do not crash yet;
	//      if ( seqpos_ == 0 ) {
	//      std::cout << "ERROR!! Could not find residue to backrub around" << std::endl;
	//      exit(1);
	//      }

  dumppdb_=option[ OptionKeys::pocket_grid::pocket_dump_pdbs ]();
	totalres_=pose.total_residue();

  if ( seqpos_ != 0 ) {
    pocketgrid_ = new core::grid::Pockets::PocketGrid( pose.conformation().residue(seqpos_) );
  }

  // JK NOTE: WE'RE NOT USING THE "FUNC" SYSTEM, THIS COULD BE ADDED LATER....

}

PocketConstraint::PocketConstraint( const PocketConstraint& old ):
  Constraint( core::scoring::pocket_constraint )
{
  seqpos_ = old.seqpos_;
  totalres_ = old.totalres_;
  pocketgrid_ = old.pocketgrid_;
  atom_ids_ = old.atom_ids_;
  dumppdb_ = old.dumppdb_;
}


PocketConstraint::~PocketConstraint() {}


void PocketConstraint::set_target_res( pose::Pose const & pose, Size new_seqpos ){
  if (new_seqpos>pose.total_residue()){
    std::cout << "ERROR!! Invalid residue to backrub around" << std::endl;
    exit(1);
  }
  if ( seqpos_ != 0 ) {
		seqpos_=new_seqpos;
    pocketgrid_ = new core::grid::Pockets::PocketGrid( pose.conformation().residue(seqpos_) );
  }else{
    std::cout << "ERROR!! Invalid residue to backrub around" << std::endl;
    exit(1);
  }
}


void PocketConstraint::set_target_res_pdb( pose::Pose const & pose, std::string resid ){
    //std::cout<<size_x_<<" "<<size_y_<<" "<<size_z_<<"\n";
  int  central_relax_pdb_number;
  char chain = ' ';
  std::size_t fpos( resid.find(':') );
  if ( fpos != std::string::npos ) {
    central_relax_pdb_number = ObjexxFCL::int_of( resid.substr(0,fpos) );
    if (fpos != resid.size()-1 ) {
      chain = resid[ fpos+1 ];
    }
  } else {
    central_relax_pdb_number = ObjexxFCL::int_of( resid );
  }

  seqpos_ = 0;
  for ( int j = 1, resnum = pose.total_residue(); j <= resnum; ++j ) {
    if ( pose.pdb_info()->number(j) == central_relax_pdb_number ) {
      //seqpos_ = j;
      if (chain != ' '){
        if ( pose.pdb_info()->chain(j) == chain ) {
          seqpos_ = j;
        }
      }else{
        seqpos_ = j;
      }
    }
  }

  if ( seqpos_ != 0 ) {
    pocketgrid_ = new core::grid::Pockets::PocketGrid( pose.conformation().residue(seqpos_) );
  } else {
    std::cout << "ERROR!! Invalid residue to backrub around" << std::endl;
    exit(1);
  }

}




// Calculates a score for this constraint using XYZ_Func, and puts the UNWEIGHTED score into
// emap. Although the current set of weights currently is provided, Constraint objects
// should put unweighted scores into emap.
void
PocketConstraint::score( XYZ_Func const & xyz_func, EnergyMap const & weights, EnergyMap & emap ) const
{
  //std::cout<<size_x_<<" "<<size_y_<<" "<<size_z_<<"\n";
  if ( weights[ this->score_type() ] == 0 ) return;
  if (seqpos_==0){
    std::cout << "ERROR!! Invalid residue to backrub around" << std::endl;
    exit(1);
  }

  conformation::Residue const & curr_rsd( xyz_func.residue(seqpos_) );

	pocketgrid_->autoexpanding_pocket_eval( curr_rsd, xyz_func, totalres_ );

  core::Real cst_val = -1.;

  //  core::Real largestPocketVol=pocketgrid_->netTargetPocketVolume();

	if (dumppdb_) pocketgrid_->dumpGridToFile();

	//	core::Real vol=pocketgrid_->targetPocketVolume(surf_score, bur_score);
	//	core::Real sa=pocketgrid_->targetPocketSolventSurface();
	//	core::Real psa=pocketgrid_->targetPocketProteinSurface();
	//	core::Real hpsa=pocketgrid_->targetPocketHydrophobicProteinSurface();
	//	core::Real ppsa=pocketgrid_->targetPocketPolarProteinSurface();
	//	core::Real nps=pocketgrid_->targetPocketHeuristicScore();

  //core::Real largestPocketVol=pocketgrid_->largestTargetPocketVolume();
  core::Real largestPocketVol=pocketgrid_->netTargetPocketVolume();
  //core::Real largestPocketVol=0;

  cst_val *= (largestPocketVol);
  //cst_val *= (vol);
  //cst_val *= (vol*vol/sa);
  //  std::cout<<"Vol: "<<vol<<" Solvent surface: "<<sa<<" Prot surface: "<<psa<<" Hprot surface: "<<hpsa<<" Pprot surface: "<<ppsa<<" New Score: "<<nps<<std::endl;
  //std::cout<<"done4\n";
  //  cst_val *= distance(CB_curr,CA_curr);
  emap[ this->score_type() ] += cst_val;
  //std::cout<<cst_val<<" done5\n";

}


void
PocketConstraint::fill_f1_f2(
    AtomID const & ,
    XYZ_Func const & ,
    Vector & ,
    Vector & ,
    EnergyMap const & weights
    ) const
{

  if ( weights[ this->score_type() ] == 0 ) return;

  using namespace core::options;
  if (!option[ OptionKeys::constraints::pocket_zero_derivatives ]()){
    TR << "ERROR - derivatives not yet implemented for PocketConstraints." << std::endl;
    std::exit(1);
  }

  return;

}


ConstraintOP PocketConstraint::clone() const {
  return ConstraintOP( new PocketConstraint( *this ) );
}


} // namespace constraints
} // namespace scoring
} // namespace core
