// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author

#ifndef INCLUDED_core_scoring_electron_density_util_HH
#define INCLUDED_core_scoring_electron_density_util_HH

#include <core/types.hh>
#include <core/scoring/ScoreFunction.fwd.hh>

#include <numeric/xyzMatrix.fwd.hh>
#include <numeric/xyzVector.hh>

#include <ObjexxFCL/FArray3D.hh>

#include <iostream>


namespace core {
namespace scoring {
namespace electron_density {

/// @brief update scorefxn with density scores from commandline
void add_dens_scores_from_cmdline_to_scorefxn( core::scoring::ScoreFunction &scorefxn_  );

/// @brief trilinear interpolation with periodic boundaries
template <class S>
core::Real interp_linear(
	ObjexxFCL::FArray3D< S > const & data ,
	numeric::xyzVector< core::Real > const & idxX) {

	int pt000[3], pt111[3];
	core::Real fpart[3],neg_fpart[3];
	int grid[3];
	grid[0] = data.u1(); grid[1] = data.u2(); grid[2] = data.u3();

	// find bounding grid points
	pt000[0] = (int)(floor(idxX[0])) % grid[0]; if (pt000[0] <= 0) pt000[0]+= grid[0];
	pt000[1] = (int)(floor(idxX[1])) % grid[1]; if (pt000[1] <= 0) pt000[1]+= grid[1];
	pt000[2] = (int)(floor(idxX[2])) % grid[2]; if (pt000[2] <= 0) pt000[2]+= grid[2];
	pt111[0] = (pt000[0]+1); if (pt111[0]>grid[0]) pt111[0] = 1;
	pt111[1] = (pt000[1]+1); if (pt111[1]>grid[1]) pt111[1] = 1;
	pt111[2] = (pt000[2]+1); if (pt111[2]>grid[2]) pt111[2] = 1;

	// interpolation coeffs
	fpart[0] = idxX[0]-floor(idxX[0]); neg_fpart[0] = 1-fpart[0];
	fpart[1] = idxX[1]-floor(idxX[1]); neg_fpart[1] = 1-fpart[1];
	fpart[2] = idxX[2]-floor(idxX[2]); neg_fpart[2] = 1-fpart[2];

	S retval = (S)0.0;
	retval+= neg_fpart[0]*neg_fpart[1]*neg_fpart[2] * data(pt000[0],pt000[1],pt000[2]);
	retval+= neg_fpart[0]*neg_fpart[1]*    fpart[2] * data(pt000[0],pt000[1],pt111[2]);
	retval+= neg_fpart[0]*    fpart[1]*neg_fpart[2] * data(pt000[0],pt111[1],pt000[2]);
	retval+= neg_fpart[0]*    fpart[1]*    fpart[2] * data(pt000[0],pt111[1],pt111[2]);
	retval+= fpart[0]*neg_fpart[1]*neg_fpart[2] * data(pt111[0],pt000[1],pt000[2]);
	retval+= fpart[0]*neg_fpart[1]*    fpart[2] * data(pt111[0],pt000[1],pt111[2]);
	retval+= fpart[0]*    fpart[1]*neg_fpart[2] * data(pt111[0],pt111[1],pt000[2]);
	retval+= fpart[0]*    fpart[1]*    fpart[2] * data(pt111[0],pt111[1],pt111[2]);

	return retval;
}

/// @brief spline interpolation with periodic boundaries
core::Real interp_spline( ObjexxFCL::FArray3D< double > & coeffs ,
                          numeric::xyzVector<core::Real> const & idxX );

/// @brief precompute spline coefficients (float array => double coeffs)
void spline_coeffs( ObjexxFCL::FArray3D< double > &data, ObjexxFCL::FArray3D< double > & coeffs);

/// @brief precompute spline coefficients (double array => double coeffs)
void spline_coeffs( ObjexxFCL::FArray3D< float > &data, ObjexxFCL::FArray3D< double > & coeffs);

// TODO ??? complex spline interpolation


} // namespace constraints
} // namespace scoring
} // namespace core

#endif
