// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   core/scoring/geometric_solvation/HBondDatabase.cc
/// @brief  Database containing params for HBondEnergy
/// @author John Karanicolas
/// @author Matthew O'Meara


// Unit Headers
#include <core/scoring/hbonds/HBondDatabase.hh>

// Package Headers
#include <core/scoring/hbonds/types.hh>
#include <core/scoring/hbonds/HBondOptions.hh>
#include <core/scoring/hbonds/polynomial.hh>
#include <core/scoring/hbonds/HBondTypeManager.hh>
#include <core/scoring/hbonds/constants.hh>
#include <core/scoring/hbonds/FadeInterval.hh>

// Project Headers
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/AtomTypeSet.hh>
#include <core/scoring/ScoringManager.fwd.hh>

#include <core/io/database/open.hh>
#include <core/util/Tracer.hh>

// Utility Headers
#include <utility/io/izstream.hh>
#include <utility/exit.hh>
#include <utility/string_util.hh>
#include <utility/vector1.hh>


// Boost Headers
#include <boost/tokenizer.hpp>

// C++ headers
#include <cmath>


namespace core {
namespace scoring {
namespace hbonds {

	using namespace std;
	using namespace utility;

	static util::Tracer tr("core.scoring.hbonds.HBondDatabase");
	// Initialize private static data
	map< const string, HBondDatabaseCOP > HBondDatabase::initialized_databases_;


HBondDatabase::HBondDatabase():
	initialized_(false),
	hb_options_( new HBondOptions() ),
	HBFadeInterval_lookup_by_name_(),
	HBFadeInterval_lookup_(),
	AHdist_short_fade_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	AHdist_long_fade_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosBAH_fade_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosAHD_fade_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	HBPoly1D_lookup_by_name_(),
	HBPoly1D_lookup_(),
	AHdist_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosBAH_short_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosBAH_long_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosAHD_short_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosAHD_long_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	chi_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	weight_type_lookup_(HB_EVAL_TYPE_COUNT, hbw_NONE)
{
	initialize();
}

HBondDatabase::HBondDatabase(
	HBondOptionsCOP hb_options
) :
	initialized_(false),
	hb_options_( hb_options ),
	HBFadeInterval_lookup_by_name_(),
	HBFadeInterval_lookup_(),
	AHdist_short_fade_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	AHdist_long_fade_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosBAH_fade_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosAHD_fade_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	HBPoly1D_lookup_by_name_(),
	HBPoly1D_lookup_(),
	AHdist_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosBAH_short_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosBAH_long_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosAHD_short_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	cosAHD_long_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	chi_poly_lookup_(HB_EVAL_TYPE_COUNT, NULL),
	weight_type_lookup_(HB_EVAL_TYPE_COUNT, hbw_NONE)
{
	initialize();
}

HBondDatabase::HBondDatabase(
	const HBondDatabase & src
) :
	ReferenceCount( src ),
	initialized_(src.initialized_),
	hb_options_( src.hb_options_ ),
	HBFadeInterval_lookup_by_name_(src.HBFadeInterval_lookup_by_name_),
	HBFadeInterval_lookup_(src.HBFadeInterval_lookup_),
	AHdist_short_fade_lookup_(src.AHdist_short_fade_lookup_),
	AHdist_long_fade_lookup_(src.AHdist_long_fade_lookup_),
	cosBAH_fade_lookup_(src.cosBAH_fade_lookup_),
	cosAHD_fade_lookup_(src.cosAHD_fade_lookup_),
	HBPoly1D_lookup_by_name_(src.HBPoly1D_lookup_by_name_),
	HBPoly1D_lookup_(src.HBPoly1D_lookup_),
	AHdist_poly_lookup_(src.AHdist_poly_lookup_),
	cosBAH_short_poly_lookup_(src.cosBAH_short_poly_lookup_),
	cosBAH_long_poly_lookup_(src.cosBAH_long_poly_lookup_),
	cosAHD_short_poly_lookup_(src.cosAHD_short_poly_lookup_),
	cosAHD_long_poly_lookup_(src.cosAHD_long_poly_lookup_),
	chi_poly_lookup_(src.chi_poly_lookup_),
	weight_type_lookup_(src.weight_type_lookup_)
{}

HBondDatabaseCOP
HBondDatabase::get_database(){

	HBondOptionsOP hb_options( new HBondOptions() );

	map< string const, HBondDatabaseCOP >::const_iterator
		param_db = initialized_databases_.find(hb_options->params_database_tag());
	if(param_db == initialized_databases_.end()){
		return new HBondDatabase( hb_options );
	}
	return param_db->second();
}

HBondDatabaseCOP
HBondDatabase::get_database( string tag ){

	map< string const, HBondDatabaseCOP >::const_iterator
		param_db = initialized_databases_.find(tag);
	if(param_db == initialized_databases_.end()){
		return new HBondDatabase( new HBondOptions( tag ) );
	}
	return param_db->second();
}



HBondDatabase::~HBondDatabase(){}


/// @details initialize hydrogen bond parameters
void
HBondDatabase::initialize()
{
	if(initialized_){
		tr << "Re-intializing HBond Database when it has already been initialized!";
	}
	initialized_databases_[ hb_options_->params_database_tag() ] = this;

	initialize_HBPoly1D();
	initialize_HBFadeInterval();
	initialize_HBEval();
	initialized_ = true;
}

/// @details has the database already been initialized?
bool
HBondDatabase::initialized() const
{
	return initialized_;
}


void
HBondDatabase::initialize_HBFadeInterval()
{
	string HBFadeInterval_fname = "scoring/score_functions/hbonds/" + hb_options_->params_database_tag() + "/HBFadeIntervals.csv";

	utility::io::izstream s;
	if (!io::database::open(s, HBFadeInterval_fname)){
		stringstream message;
		message << "Unable to open hbond parameter file HBFadeInterval:" << std::endl;
		message << "'" << HBFadeInterval_fname << "'";
		utility_exit_with_message(message.str());
	}

	string line;
	vector1<string> tokens;
	Size id;
	string fade_interval_name;
	Real min0, fmin, fmax, max0;
	while ( getline( s, line ) ) {
		tokens = string_split( line, ',');
		Size ntokens = 7;
		if (tokens.size() != ntokens){
			stringstream message;
			message << "FadeInterval definition line does not have the expected number of fields " << std::endl;
			message << "Expected '" << ntokens << "' tokens but found '" << tokens.size() << "' tokens. " << std::endl;
			message << line << std::endl;
			utility_exit_with_message(message.str());
		}

		Size i(1);
		{ stringstream buf;buf.precision(16); buf << tokens[i]; i++; buf >> id; }
		{ stringstream buf;buf.precision(16); buf << tokens[i]; i++; buf >> fade_interval_name; }
		{ stringstream buf;buf.precision(16); buf << tokens[i]; i++; buf >> min0; }
		{ stringstream buf;buf.precision(16); buf << tokens[i]; i++; buf >> fmin; }
		{ stringstream buf;buf.precision(16); buf << tokens[i]; i++; buf >> fmax; }
		{ stringstream buf;buf.precision(16); buf << tokens[i]; i++; buf >> max0; }

		FadeIntervalOP fade_interval(new FadeInterval(min0, fmin, fmax, max0));

		if( HBFadeInterval_lookup_.size() + 1 != id ){
			stringstream message;
			message << "The id fields in the HBFadeInterval file '" << HBFadeInterval_fname << "'" << std::endl;
			message << "are out of order or missing: Expected id: " << HBFadeInterval_lookup_.size() + 1 << " but instead found id: " << id << std::endl;
			message << "on line: '" << line << "'" << std::endl;
			utility_exit_with_message(message.str());
		}
		HBFadeInterval_lookup_.push_back(fade_interval);
		HBFadeInterval_lookup_by_name_[fade_interval_name] = fade_interval;
	}
}

	/// @details read one dimensional polynomial definition file
	// File Format:
	//    -fields are space delimited
	//    -Columns are: polynomial_name, geometric_dimension, xmin, xmax, root1, root2, degree, c_a, c_b, ..., c_k
void
HBondDatabase::initialize_HBPoly1D()
{

	string HBPoly1D_fname = "scoring/score_functions/hbonds/" + hb_options_->params_database_tag() + "/HBPoly1D.csv";

	utility::io::izstream s;
	if (!io::database::open(s, HBPoly1D_fname)){
		stringstream message;
		message << "Unable to open hbond parameter file HBPoly1D:" << std::endl;
		message << HBPoly1D_fname;
		utility_exit_with_message(message.str());
	}

	string line;
	vector1<string> tokens;
	Size id;
	string polynomial_name;
	string geo_dim_name;
	HBGeoDimType geometric_dimension;
	Real xmin, xmax, root1, root2;
	Size degree;
	vector1< Real > coefficients_;
	while ( getline( s, line ) ) {
		tokens = string_split( line, ',');
		Size ntokens = 20;
		if (tokens.size() != ntokens){
			stringstream message;
			message << "Polynomial definition line does not have enough fields" << std::endl;
			message << "Expected " << ntokens << " tokens but found " << tokens.size() << " tokens. " << std::endl;
			message << line << std::endl;
			utility_exit_with_message(message.str());
		}

		Size i(1);
		{ stringstream buf; buf << tokens[i]; i++; buf >> id;}
		{ stringstream buf; buf << tokens[i]; i++; buf >> polynomial_name;}
		i++; // classic name field
		{ stringstream buf; buf << tokens[i]; i++; buf >> geo_dim_name;
			geometric_dimension = HBondTypeManager::geo_dim_type_from_name( geo_dim_name ); }
		{ stringstream buf; buf << tokens[i]; i++; buf >> xmin;}
		{ stringstream buf; buf << tokens[i]; i++; buf >> xmax;}
		{ stringstream buf; buf << tokens[i]; i++; buf >> root1;}
		{ stringstream buf; buf << tokens[i]; i++; buf >> root2;}
		{ stringstream buf; buf << tokens[i]; i++; buf >> degree;}

		vector1< Real > coefficients_;
		Real c;
		while( i <= tokens.size()){
			stringstream buf; buf << tokens[i]; i++; buf >> c;
			coefficients_.push_back(c);
		}

		Polynomial_1dOP p(new Polynomial_1d(
			polynomial_name,
			geometric_dimension,
			xmin, xmax, root1, root2,
			degree,
			coefficients_));

		if( HBPoly1D_lookup_.size() + 1 != id ){
			stringstream message;
			message << "The id fields in the HBPoly1D file '" << HBPoly1D_fname << "'" << std::endl;
			message << "are out of order or missing: Expected id: " << HBPoly1D_lookup_.size() + 1 << " but instead found id: " << id << std::endl;
			message << "on line: '" << line << "'" << std::endl;
			utility_exit_with_message(message.str());
		}
		HBPoly1D_lookup_.push_back(p);
		HBPoly1D_lookup_by_name_[polynomial_name] = p;

	}
}

	///@details read one dimensional polynomial definition file
	// File Format:
	//    -fields are space delimited
	//    -Columns are: HBDonChemType HBAccChemType, HBSeqSep, AHDis_poly_name, cosBAH_poly_name, cosAHD_poly_name, chi_poly_name
void
HBondDatabase::initialize_HBEval()
{

	string HBEval_fname = "scoring/score_functions/hbonds/" + hb_options_->params_database_tag() + "/HBEval.csv";

	utility::io::izstream s;
	io::database::open(s, HBEval_fname);

	vector1<string> tokens;
	string line;
	string AHdist_poly_name, cosBAH_short_poly_name, cosBAH_long_poly_name,
		cosAHD_short_poly_name, cosAHD_long_poly_name, chi_poly_name,
		don_chem_type_name, acc_chem_type_name, seq_sep_type_name,
		AHdist_short_fade_name, AHdist_long_fade_name,
		cosBAH_fade_name, cosAHD_fade_name;
	HBDonChemType don_chem_type;
	HBAccChemType acc_chem_type;
	HBSeqSep seq_sep_type;
	Polynomial_1dCOP AHdist_poly, cosBAH_short_poly, cosBAH_long_poly,
		cosAHD_short_poly, cosAHD_long_poly, chi_poly;
	FadeIntervalCOP AHdist_short_fade, AHdist_long_fade, cosBAH_fade, cosAHD_fade;
	string weight_type_name;
	HBondWeightType weight_type;

	while (getline(s, line)) {
		tokens = string_split( line, ',');
		if (tokens.size() != 15){
			stringstream message;
			message << "HBond evaluation data line does not have enough fields" << std::endl;
			message << "Expected '" << 15 << "' tokens but found '" << tokens.size() << "' tokens." << std::endl;
			message << line << std::endl;
			utility_exit_with_message(message.str());
		}

		Size i(1);
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> don_chem_type_name;
			don_chem_type = HBDonChemType(HBondTypeManager::don_chem_type_from_name(don_chem_type_name));
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> acc_chem_type_name;
			acc_chem_type = HBAccChemType(HBondTypeManager::acc_chem_type_from_name(acc_chem_type_name));
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> seq_sep_type_name;
			seq_sep_type = HBSeqSep(HBondTypeManager::seq_sep_type_from_name(seq_sep_type_name));
		}

		HBEvalType hbe_type = HBEval_lookup(don_chem_type, acc_chem_type, seq_sep_type);

		if( hbe_type > static_cast<int>(HB_EVAL_TYPE_COUNT) ){
			stringstream message;
			message << "hb_eval_type created from" << std::endl;
			message << "\tdon_chem_type:'"<< don_chem_type_name <<"'" << std::endl;
			message << "\tacc_chem_type:'"<< acc_chem_type_name <<"'" << std::endl;
			message << "\tseq_sep_type: '"<< seq_sep_type_name << "'" << std::endl;
			message << "gives type" << hbe_type << ", which is out of the range valid range (0," << HB_EVAL_TYPE_COUNT << ")" << std::endl;
			utility_exit_with_message(message.str());
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> AHdist_short_fade_name;
			AHdist_short_fade = HBFadeInterval_from_name(AHdist_short_fade_name);
			if(AHdist_short_fade_lookup_[hbe_type]){
				assert(AHdist_short_fade_lookup_[hbe_type] == AHdist_short_fade);
			} else {
				AHdist_short_fade_lookup_[hbe_type] = AHdist_short_fade;
			}
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> AHdist_long_fade_name;
			AHdist_long_fade = HBFadeInterval_from_name(AHdist_long_fade_name);
			if(AHdist_long_fade_lookup_[hbe_type]){
				assert(AHdist_long_fade_lookup_[hbe_type] == AHdist_long_fade);
			} else {
				AHdist_long_fade_lookup_[hbe_type] = AHdist_long_fade;
			}
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> cosBAH_fade_name;
			cosBAH_fade = HBFadeInterval_from_name(cosBAH_fade_name);
			if(cosBAH_fade_lookup_[hbe_type]){
				assert(cosBAH_fade_lookup_[hbe_type] == cosBAH_fade);
			} else {
				cosBAH_fade_lookup_[hbe_type] = cosBAH_fade;
			}
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> cosAHD_fade_name;
			cosAHD_fade = HBFadeInterval_from_name(cosAHD_fade_name);
			if(cosAHD_fade_lookup_[hbe_type]){
				assert(cosAHD_fade_lookup_[hbe_type] == cosAHD_fade);
			} else {
				cosAHD_fade_lookup_[hbe_type] = cosAHD_fade;
			}
		}
		++i; // fade for chi dimension
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> AHdist_poly_name;
			AHdist_poly = HBPoly1D_from_name(AHdist_poly_name);
			if(AHdist_poly_lookup_[hbe_type]){
				assert(AHdist_poly_lookup_[hbe_type] == AHdist_poly);
			} else {
				AHdist_poly_lookup_[hbe_type] = AHdist_poly;
			}
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> cosBAH_short_poly_name;
			cosBAH_short_poly = HBPoly1D_from_name(cosBAH_short_poly_name);
			if(cosBAH_short_poly_lookup_[hbe_type]){
				assert(cosBAH_short_poly_lookup_[hbe_type] == cosBAH_short_poly);
			}{
				cosBAH_short_poly_lookup_[hbe_type] = cosBAH_short_poly;
			}
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> cosBAH_long_poly_name;
			cosBAH_long_poly = HBPoly1D_from_name(cosBAH_long_poly_name);
			if(cosBAH_long_poly_lookup_[hbe_type]){
				assert(cosBAH_long_poly_lookup_[hbe_type] == cosBAH_long_poly);
			} else {
				cosBAH_long_poly_lookup_[hbe_type] = cosBAH_long_poly;
			}
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> cosAHD_short_poly_name;
			cosAHD_short_poly = HBPoly1D_from_name(cosAHD_short_poly_name);
			if(cosAHD_short_poly_lookup_[hbe_type]){
				assert(cosAHD_short_poly_lookup_[hbe_type]);
			} else {
				cosAHD_short_poly_lookup_[hbe_type] = cosAHD_short_poly;
			}
		}
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> cosAHD_long_poly_name;
			cosAHD_long_poly = HBPoly1D_from_name(cosAHD_long_poly_name);
			if(cosAHD_long_poly_lookup_[hbe_type]){
				assert(cosAHD_long_poly_lookup_[hbe_type] == cosAHD_long_poly);
			} else {
				cosAHD_long_poly_lookup_[hbe_type] = cosAHD_long_poly;
			}
		}
//if (tokens[i] != ""){  // chi dimension is not yet defined!
//	stringstream buf;
//	buf << tokens[i];
//	buf >> chi_poly_name;
//	chi_poly = HBPoly1D_from_name(chi_poly_name);
//	if(chi_poly_lookup_[hbe_type]){
//		assert(chi_poly_lookup_[hbe_type] == chi_poly);
//	} else {
//		chi_poly_lookup_[hbe_type] = chi_poly;
//	}
//}
//i++;
		{
			stringstream buf;
			buf << tokens[i]; i++;
			buf >> weight_type_name;
			weight_type = HBondWeightType(HBondTypeManager::weight_type_from_name(weight_type_name));
			if(weight_type_lookup_[hbe_type] != hbw_NONE){
				assert(weight_type_lookup_[hbe_type] == weight_type);
			} else {
				weight_type_lookup_[hbe_type] = weight_type;
			}
		}
	}
}
FadeIntervalCOP
HBondDatabase::HBFadeInterval_from_name(
	string const name
) const {
	map< const string, FadeIntervalCOP >::const_iterator it(HBFadeInterval_lookup_by_name_.find(name));
	if( it == HBFadeInterval_lookup_by_name_.end() ){
		stringstream message;
		message << "Fade Interval '" << name << "' has not been defined.";
		utility_exit_with_message(message.str());
		return NULL;
	} else {
		return it->second;
	}
}



///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
FadeIntervalCOP
HBondDatabase::AHdist_short_fade_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type <<
			"' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}

	FadeIntervalCOP p(AHdist_short_fade_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No short fade interval for AHdist has been defined for hb eval type '"
						<< hb_eval_type << "'" <<std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
FadeIntervalCOP
HBondDatabase::AHdist_long_fade_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	FadeIntervalCOP p(AHdist_long_fade_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No long fade interval for AHdist has been defined for hb eval type '"
						<< hb_eval_type << "'" <<std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
FadeIntervalCOP
HBondDatabase::cosBAH_fade_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	FadeIntervalCOP p(cosBAH_fade_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No fade interval for cosBAH has been defined for hb eval type '"
						<< hb_eval_type << "'" <<std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
FadeIntervalCOP
HBondDatabase::cosAHD_fade_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	FadeIntervalCOP p(cosAHD_fade_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No fade interval for cosAHD has been defined for hb eval type '"
						<< hb_eval_type << "'" <<std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

Polynomial_1dCOP
HBondDatabase::HBPoly1D_from_name(
	string const name
) const {
	map< const string, Polynomial_1dCOP >::const_iterator it(HBPoly1D_lookup_by_name_.find(name));
	if( it == HBPoly1D_lookup_by_name_.end() ){
		stringstream message;
		message << "1d Polynomial '" << name << "' has not been defined.";
		utility_exit_with_message(message.str());
		return NULL;
	} else {
		return it->second;
	}
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
Polynomial_1dCOP
HBondDatabase::AHdist_poly_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	Polynomial_1dCOP p(AHdist_poly_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No AHdist polynomial has been defined for hb eval type '"
						<< hb_eval_type << "'" <<std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
Polynomial_1dCOP
HBondDatabase::cosBAH_short_poly_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	Polynomial_1dCOP p(cosBAH_short_poly_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No cosBAH_short polynomial has been defined for hb eval type '"
						<< hb_eval_type << "'" << std::endl;
		utility_exit_with_message(message.str());
	}
	return p;

}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
Polynomial_1dCOP
HBondDatabase::cosBAH_long_poly_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	Polynomial_1dCOP p(cosBAH_long_poly_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No cosBAH_long polynomial has been defined for hb eval type '"
						<< hb_eval_type << "'" << std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
Polynomial_1dCOP
HBondDatabase::cosAHD_short_poly_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	Polynomial_1dCOP p(cosAHD_short_poly_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No cosAHD_short polynomial has been defined for hb eval type '"
						<< hb_eval_type << "'" << std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
Polynomial_1dCOP
HBondDatabase::cosAHD_long_poly_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	Polynomial_1dCOP p(cosAHD_long_poly_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No cosAHD_long polynomial has been defined for hb eval type '"
						<< hb_eval_type << "'" << std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
Polynomial_1dCOP
HBondDatabase::chi_poly_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	Polynomial_1dCOP p(chi_poly_lookup_[hb_eval_type]);

	if(!p) {
		stringstream message;
		message << "No chi polynomial has been defined for hb eval type '"
						<< hb_eval_type << "'" << std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}

///@details use core::scoring::hbonds::get_hbond_weight_type(...) to
///compute hb_eval_type.
HBondWeightType
HBondDatabase::weight_type_lookup(
	Size const hb_eval_type
) const {
	if( hb_eval_type < 1 || hb_eval_type > HB_EVAL_TYPE_COUNT ){
		stringstream message;
		message << "HBond eval type '" << hb_eval_type
						<< "' is out side of the valid range (1," << HB_EVAL_TYPE_COUNT << ")";
		utility_exit_with_message(message.str());
	}
	HBondWeightType p(weight_type_lookup_[hb_eval_type]);

	if(p == hbw_NONE) {
		stringstream message;
		message << "No weight type has been defined for hb eval type '"
						<< hb_eval_type << "'" << std::endl;
		utility_exit_with_message(message.str());
	}
	return p;
}


} // geometric_solvation
} // namespace scoring
} // namespace core

