// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/UnfoldedStateEnergy.cc
/// @brief  Unfolded state energy method implementation; energies based on eneriges of residues in fragments
/// @author Ron Jacak (ronj@email.unc.edu)

// Unit headers
#include <core/scoring/methods/UnfoldedStateEnergy.hh>
#include <core/scoring/methods/UnfoldedStateEnergyCreator.hh>

// Package headers
#include <core/scoring/methods/ContextIndependentOneBodyEnergy.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/UnfoldedStatePotential.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>
#include <core/util/Tracer.hh>

// Project headers
#include <core/conformation/Residue.hh>

static core::util::Tracer TR("core.scoring.methods.UnfoldedStateEnergy");

namespace core {
namespace scoring {
namespace methods {


/// @details This must return a fresh instance of the UnfoldedStateEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
UnfoldedStateEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const & options
) const {

	if ( options.has_method_weights( unfolded ) ) {
		return new UnfoldedStateEnergy( options.unfolded_energies_type(), options.method_weights( unfolded ) );
	}
	return new UnfoldedStateEnergy( options.unfolded_energies_type() );
}

ScoreTypes
UnfoldedStateEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( unfolded );
	return sts;
}


 UnfoldedStateEnergy::UnfoldedStateEnergy( std::string const & type ) :
	 parent( new UnfoldedStateEnergyCreator ),
	 type_( type ),
	 unf_state_potential_(ScoringManager::get_instance()->get_UnfoldedStatePotential( type ) ),
	 score_type_weights_( unf_state_potential_.get_unfoled_potential_file_weights() )
{}

 UnfoldedStateEnergy::UnfoldedStateEnergy( std::string const & type, utility::vector1< Real > const & vector_weights_in ):
	parent( new UnfoldedStateEnergyCreator ),
	type_( type ),
	unf_state_potential_( ScoringManager::get_instance()->get_UnfoldedStatePotential( type ) )
{
	store_weights_in_emap( vector_weights_in );
#ifndef NDEBUG
	TR << "instantiating class with weights: " << score_type_weights_ << std::endl;
#endif
}

 UnfoldedStateEnergy::UnfoldedStateEnergy( std::string const & type, const EnergyMap & emap_in ):
	parent( new UnfoldedStateEnergyCreator ),
	type_( type ),
	unf_state_potential_( ScoringManager::get_instance()->get_UnfoldedStatePotential( type ) ),
	score_type_weights_( emap_in )
 {}

UnfoldedStateEnergy::~UnfoldedStateEnergy() {}

EnergyMethodOP
UnfoldedStateEnergy::clone() const {
	return new UnfoldedStateEnergy( type_, score_type_weights_ );
}


void
UnfoldedStateEnergy::store_weights_in_emap( utility::vector1< Real > const & v ) {
	assert( v.size() == scoring::n_score_types );

	// below assumes that the vector of Reals coming in contains the weights for each of the score types in the
	// scoring namespace enumeration, and more importantly, in the same order.
	for ( Size ii=1; ii < scoring::n_score_types; ++ii ) {
		score_type_weights_[ (ScoreType) ii ] = v[ii];
	}
}


void
UnfoldedStateEnergy::residue_energy(
	conformation::Residue const & rsd,
	pose::Pose const &,
	EnergyMap & emap
) const
{

	// this function is becoming a no-op because unfolded state energies can only be comptued when a weight
	// set has been established.  weight sets are only determined in the optE protocol after raw energies have
	// been obtained. this class will just serve as a placeholder for an energy that will be computed later.
	//emap[ scoring::unfolded ] = 0.0;

	// in an update to this function, this function will now return an energy depending on the
	// weights that were passed in when the class was instantiated.  if no weights were passed in, a zero'd
	// the member variable energy map is zero'd out and the function will effectively return 0.0.
	// during the first phase of the optE protocol, this function will return 0.0.  once a weight set has
	// been established, this class will get recreated with that weight set and this function will return
	// meaningful energies.

	EnergyMap unweighted_unfolded_energies;
	unf_state_potential_.unfolded_state_energy( rsd.type().name3(), unweighted_unfolded_energies );

	// don't forget to include the weight for the entire term. no, wait, that weight should get applied later
	// in the scoring process. this method should just return the unweighted, unfolded state energy.
	emap[ unfolded ] += unweighted_unfolded_energies.dot( score_type_weights_ );

	return;
}


Real
UnfoldedStateEnergy::eval_dof_derivative( id::DOF_ID const &, id::TorsionID const &, pose::Pose const &,
	ScoreFunction const &, EnergyMap const & ) const {
	return 0.0;
}

void
UnfoldedStateEnergy::indicate_required_context_graphs( utility::vector1< bool > & ) const {}


} // methods
} // scoring
} // core

