// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
#include <core/pose/Pose.hh>

#include <numeric/MathVector_operations.hh>
#include <core/chemical/util.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/Patch.hh>

#include <core/scoring/DerivVectorPair.hh>

#include <core/scoring/orbitals/OrbitalsLookup.hh>
#include <core/scoring/orbitals/OrbitalsScore.hh>
#include <core/scoring/orbitals/OrbitalsScoreCreator.hh>

#include <core/conformation/Residue.hh>

#include <core/scoring/ScoringManager.hh>
#include <core/chemical/AtomType.hh>

#include <core/scoring/ContextGraphTypes.hh>

#include <core/scoring/EnergyMap.hh>
#include <core/util/Tracer.hh>

#include <utility/vector1.hh>
#include <numeric/xyzTriple.hh>

#include <core/options/option.hh>
#include <core/options/keys/lp.OptionKeys.gen.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>
#include <core/scoring/etable/Etable.hh>

#include <numeric/xyzVector.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>


namespace core{
namespace scoring{
namespace orbitals{

methods::EnergyMethodOP
OrbitalsScoreCreator::create_energy_method(
		methods::EnergyMethodOptions const & options
) const{
	return new OrbitalsScore(options);
}

ScoreTypes
OrbitalsScoreCreator::score_types_for_method() const{
	ScoreTypes sts;
	sts.push_back( orbitals_hpol );
	sts.push_back( orbitals_haro );
	sts.push_back(orbitals_orbitals);
	return sts;
}

static core::util::Tracer TR("core.scoring.orbitals_hpol");

OrbitalsScore::OrbitalsScore(methods::EnergyMethodOptions const & options) :
	parent( new OrbitalsScoreCreator ),
	lookup_table_(core::scoring::ScoringManager::get_instance()->get_OrbitalsLookupTable()),
	lookup_Etable_(core::scoring::etable::EtableEnergy( *( ScoringManager::get_instance()->etable( options.etable_type() )), options)),
	max_dist_squared_(25.0)
	//etable_( *ScoringManager::get_instance()->etable( options.etable_type() ))

{

}



methods::EnergyMethodOP OrbitalsScore::clone() const
{
	return new OrbitalsScore(*this);
}

void OrbitalsScore::setup_for_scoring(pose::Pose &, const ScoreFunction &) const
{
	//core::chemical::PatchOP patch_sc = new core::chemical::Patch();
	//patch_sc->read_file("sc_orbitals.txt");

//	for(core::Size res_num = 1; res_num <= pose.n_residue(); ++res_num){
	//	core::conformation::Residue const & res = pose.residue(res_num);
	//	if(res.has_sc_orbitals() ){
			//core::chemical::ResidueTypeOP new_rsd_type(patch_sc->apply(res.type()));
			//replace_pose_residue_copying_existing_coordinates( pose, res.seqpos(), *new_rsd_type );
			//res.type().add_variant_type("SC_ORBITALS");
		//	core::chemical::add_variant_type_to_pose_residue(pose, "SC_ORBITALS", res_num);
	//	}
/*		if(res.has_bb_orbitals() ){
			//res.type().add_variant_type("BB_ORBITALS");
			//core::chemical::add_variant_type_to_pose_residue(pose, "BB_ORBITALS", res_num);
		}*/



//	}


}





void OrbitalsScore::setup_for_derivatives(pose::Pose &, const ScoreFunction &) const
{
	// do nothing for now, otherwise do: pose.update_residue_neighbors();
}



bool OrbitalsScore::defines_intrares_energy(core::scoring::EnergyMap const &) const{
	return false;
}




void OrbitalsScore::eval_intrares_energy(
		const core::conformation::Residue &,
		const core::pose::Pose &,
		const core::scoring::ScoreFunction &,
		core::scoring::EnergyMap &
) const {}



//to determine the atomic interaction cutoff, I scanned a range between 15-0.5A and repacked 100 structures. I used the command:
//time ~combss/mini/mini/bin/fixbb.linuxgccrelease -database ~combss/mini/minirosetta_database/ -l ../pdb_list_for_atomic_interaction_cutoff_100_pdbs -ex1 -ex2 -packing:repack_only -score:weights orbitals_hpol -constant_seed -linmem_ig 10 -lp:atomic_cutoff 15
//Notice that I used the constant seed so that the only thing changing was the atomic interaction cutoff. Below are the results.
//dist        hpol      haro
//15A        -379.281 -87.6969 (did not finish in time, ran out of memory)
// 9A        -385.723 -88.8307
// 8A        -385.402 -89.7303
// 7A        -385.453 -89.3507
// 6A        -385.3   -88.8412
// 5A        -385.371 -89.1318
// 4A        -386.406 -89.1537
// 3A        -385.603 -89.5045
// 2A        -385.671 -89.4091
// 1A        -385.561 -89.2047
// According to this study, it doesnt appear that there is much difference other than time when you increase the atomic interaction cutoff.
// This could be because the other scores are helping to determine the chosen rotamer. For now, I will use 4A as the cutoff as this is where
// the time starts to level off.
core::Distance OrbitalsScore::atomic_interaction_cutoff() const{
	return  4.5;
}


void OrbitalsScore::indicate_required_context_graphs(utility::vector1< bool > & ) const
{

	//nothing
}








//The PICSES <2.8A crystal structure set (~3100 pdbs) was scanned with no distance restraints on the score. Anytime the score
//gave an energy, the action center coordinates were measured. The furtherest distance for polar polar residues was 8A. For
//aromatic aromatic the distance was 10A...this was true for mixed aromatic/polar. I take the distance squared because this
//is "faster". This check actually improves the speed.
bool OrbitalsScore::check_act_dist(core::conformation::Residue const & res1, core::conformation::Residue const & res2) const
{
	core::Real act_dist = res1.atom(res1.actcoord_atoms()[1]).xyz().distance_squared(res2.atom(res2.actcoord_atoms()[1]).xyz());
	if(res1.is_polar() && res2.is_polar()){
		if(act_dist < 81){
			return true;
		}else return false;
	}
	else if(act_dist < 121){
		return true;
	} else return false;
}







void
OrbitalsScore::residue_pair_energy
(
		core::conformation::Residue const & res1,
		core::conformation::Residue const & res2,
		const core::pose::Pose &,
		const core::scoring::ScoreFunction &,
		EnergyMap & emap
)const
{

	core::Real hpol_E(0.0);
	core::Real haro_E(0.0);



	if(res1.has_sc_orbitals()){
		get_E_hpol_haro(res1, res2, hpol_E, haro_E);
		emap[ orbitals_hpol ] += hpol_E;
		emap[ orbitals_haro ] += haro_E;

	}
	if(res2.has_sc_orbitals()){
		core::Real hpol_E = 0.0;
		core::Real haro_E = 0.0;

		get_E_hpol_haro(res2, res1, hpol_E, haro_E);



		emap[ orbitals_hpol ] += hpol_E;
		emap[ orbitals_haro ] += haro_E;


	}



	if(core::options::option[core::options::OptionKeys::lp::orb_orb_stats]){
		if(res1.has_sc_orbitals() && res2.has_sc_orbitals()){
			core::Real orb_orb_E(0.0);
			get_orb_orb_E(res1, res2, orb_orb_E);
			emap[ orbitals_orbitals ] += orb_orb_E;
		}

	}




}


void OrbitalsScore::get_E_hpol_haro(
		core::conformation::Residue const & res1,
		core::conformation::Residue const & res2,
		core::Real & hpol_E,
		core::Real & haro_E

)const {
//std::cout << res1.name3() << res1.seqpos() << std::endl;

	core::Real D_container(100.0);
	core::Real low_D(max_dist_squared_);
	std::string mm_atom_name(" ");
	numeric::xyzVector<core::Real> bonded_neighbor_xyz;
	core::Size H_index(0);
	core::Size atom_index(0);
	bool the_switch=false;
	bool get_score=false;
	numeric::xyzVector<core::Real> _orbital_xyz;
	numeric::xyzVector<core::Real> _H_xyz;
	haro_E=0;


	for
	(
			chemical::AtomIndices::const_iterator
			atoms_with_orb_index = res1.atoms_with_orb_index().begin(),
			atoms_with_orb_index_end = res1.atoms_with_orb_index().end();
			atoms_with_orb_index != atoms_with_orb_index_end; ++atoms_with_orb_index)
	{

		for
		(
				chemical::AtomIndices::const_iterator
				hpol_index = res2.Hpos_polar_sc().begin(),
				hpol_end = res2.Hpos_polar_sc().end();
				hpol_index != hpol_end; ++hpol_index
		){

			numeric::xyzVector<core::Real> const & orbital_xyz = res1.atom(*atoms_with_orb_index).xyz();
			numeric::xyzVector<core::Real> const & H_xyz = res2.atom(*hpol_index).xyz();
			D_container = orbital_xyz.distance_squared(H_xyz);
			//std::cout << "D_container  not in the_switch " << D_container << std::endl;
			if(D_container < low_D ){
				low_D = D_container;
				H_index=*hpol_index;
				atom_index=*atoms_with_orb_index;
				the_switch=true;

				//mm_atom_name = res1.mm_atom_name(orbital);
				//bonded_orbital_neighbor_xyz =  res1.atom(res1.bonded_neighbor(orbital)[1]).xyz();

			}

		}
	}


	D_container = 100.0;
	low_D = max_dist_squared_;



if(the_switch){
	for(	chemical::AtomIndices::const_iterator
			orbital_index = res1.bonded_neighbor(atom_index).begin(),
			orbital_index_end = res1.bonded_neighbor(atom_index).end();
			orbital_index != orbital_index_end; ++orbital_index
	){
		if(res1.atom_type( *orbital_index ).is_lone_pair()){
			numeric::xyzVector<core::Real> const & orbital_xyz =  res1.atom( *orbital_index ).xyz();
			numeric::xyzVector<core::Real> const & H_xyz =res2.atom( H_index ).xyz();
			D_container = orbital_xyz.distance_squared( H_xyz );
			if(D_container < low_D ){
				low_D = D_container;
				mm_atom_name = res1.mm_atom_name(*orbital_index);
				bonded_neighbor_xyz = res1.atom(res1.bonded_neighbor(*orbital_index)[1]).xyz();
				_orbital_xyz = orbital_xyz;
				_H_xyz=H_xyz;
				get_score=true;
			}
		}



	}



}

if(get_score){
	core::Real angle = cos_of(bonded_neighbor_xyz, _orbital_xyz, _H_xyz );
	hpol_E = lookup_table_.get_energy(lookup_table_.hpol, mm_atom_name, std::sqrt(low_D), angle);
	//std::cout << "inside get_score dist= " << low_D << " angle=" << angle << std::endl;
	//std::cout << "hpol_E=" << hpol_E << std::endl;
	if(core::options::option[core::options::OptionKeys::lp::verbose] && hpol_E < 0){
		std::cout << hpol_E <<  " act_dist " <<   res1.atom(res1.actcoord_atoms()[1]).xyz().distance_squared(res2.atom(res2.actcoord_atoms()[1]).xyz()) << std::endl;
	}
}


////////////////////
////////////////////
////////////////////
///////////////////
//////////////////
//start haro calculation
if(core::options::option[core::options::OptionKeys::lp::Haro]){

D_container =100.0;
low_D = max_dist_squared_;
mm_atom_name = " ";

the_switch=false;
get_score=false;



for
(
		chemical::AtomIndices::const_iterator
		atoms_with_orb_index = res1.atoms_with_orb_index().begin(),
		atoms_with_orb_index_end = res1.atoms_with_orb_index().end();
		atoms_with_orb_index != atoms_with_orb_index_end; ++atoms_with_orb_index)
{

	for
	(
			chemical::AtomIndices::const_iterator
			haro_index = res2.Haro_index().begin(),
			haro_end = res2.Haro_index().end();
			haro_index != haro_end; ++haro_index
	){

		numeric::xyzVector<core::Real> const & orbital_xyz = res1.atom(*atoms_with_orb_index).xyz();
		numeric::xyzVector<core::Real> const & H_xyz = res2.atom(*haro_index).xyz();
		D_container = orbital_xyz.distance_squared(H_xyz);
		//std::cout << "D_container  not in the_switch " << D_container << std::endl;
		if(D_container < low_D ){
			low_D = D_container;
			H_index=*haro_index;
			atom_index=*atoms_with_orb_index;
			the_switch=true;

			//mm_atom_name = res1.mm_atom_name(orbital);
			//bonded_orbital_neighbor_xyz =  res1.atom(res1.bonded_neighbor(orbital)[1]).xyz();

		}

	}
}


D_container = 100.0;
low_D = max_dist_squared_;



if(the_switch){
for(	chemical::AtomIndices::const_iterator
		orbital_index = res1.bonded_neighbor(atom_index).begin(),
		orbital_index_end = res1.bonded_neighbor(atom_index).end();
		orbital_index != orbital_index_end; ++orbital_index
){
	if(res1.atom_type( *orbital_index ).is_lone_pair()){
		numeric::xyzVector<core::Real> const & orbital_xyz =  res1.atom( *orbital_index ).xyz();
		numeric::xyzVector<core::Real> const & H_xyz =res2.atom( H_index ).xyz();
		D_container = orbital_xyz.distance_squared( H_xyz );
		if(D_container < low_D ){
			low_D = D_container;
			mm_atom_name = res1.mm_atom_name(*orbital_index);
			bonded_neighbor_xyz = res1.atom(res1.bonded_neighbor(*orbital_index)[1]).xyz();
			_orbital_xyz = orbital_xyz;
			_H_xyz=H_xyz;
			get_score=true;
		}


	}



}



}

if(get_score){
core::Real angle = cos_of(bonded_neighbor_xyz, _orbital_xyz, _H_xyz );
haro_E = lookup_table_.get_energy(lookup_table_.haro, mm_atom_name, std::sqrt(low_D), angle);
//std::cout << "inside get_score dist= " << low_D << " angle=" << angle << std::endl;
//std::cout << "hpol_E=" << hpol_E << std::endl;
if(core::options::option[core::options::OptionKeys::lp::verbose] && haro_E < 0){
	std::cout << haro_E <<  " act_dist " <<   res1.atom(res1.actcoord_atoms()[1]).xyz().distance_squared(res2.atom(res2.actcoord_atoms()[1]).xyz()) << std::endl;
}
}


}


}



void OrbitalsScore::get_orb_orb_E(
		core::conformation::Residue const & res1,
		core::conformation::Residue const & res2,
		core::Real & orb_orb_E)
const {
	core::Real D_container(100.0);
	core::Real low_D(max_dist_squared_);
	std::string mm_atom_name(" ");

	bool the_switch=false;
	bool get_score=false;

	core::Size orb_index_2(0);
	core::Size orb_index_1(0);

	numeric::xyzVector<core::Real> _orbital_xyz_1;
	numeric::xyzVector<core::Real> _orbital_xyz_2;

	numeric::xyzVector<core::Real> bonded_neighbor_xyz;


	for
	(
			chemical::AtomIndices::const_iterator
			res1_atoms_with_orb_index = res1.atoms_with_orb_index().begin(),
			res1_atoms_with_orb_index_end = res1.atoms_with_orb_index().end();
			res1_atoms_with_orb_index != res1_atoms_with_orb_index_end; ++res1_atoms_with_orb_index)
	{
		for
		(
				chemical::AtomIndices::const_iterator
				res2_atoms_with_orb_index = res2.atoms_with_orb_index().begin(),
				res2_atoms_with_orb_index_end = res2.atoms_with_orb_index().end();
				res2_atoms_with_orb_index != res2_atoms_with_orb_index_end; ++res2_atoms_with_orb_index)
		{
			numeric::xyzVector<core::Real> const & atom_with_orb_xyz_1 = res1.atom(*res1_atoms_with_orb_index).xyz();
			numeric::xyzVector<core::Real> const & atom_with_orb_xyz_2 = res2.atom(*res2_atoms_with_orb_index).xyz();
			D_container = atom_with_orb_xyz_1.distance_squared(atom_with_orb_xyz_2);
			//std::cout << "D_container  not in the_switch " << D_container << std::endl;
			if(D_container < low_D ){
				low_D = D_container;
				orb_index_2=*res2_atoms_with_orb_index;
				orb_index_1=*res1_atoms_with_orb_index;
				the_switch=true;

				//mm_atom_name = res1.mm_atom_name(orbital);
				//bonded_orbital_neighbor_xyz =  res1.atom(res1.bonded_neighbor(orbital)[1]).xyz();

			}

		}

	}


	D_container = 100.0;
	low_D = max_dist_squared_;

	if(the_switch){
		for(	chemical::AtomIndices::const_iterator
				orbital_index_1 = res1.bonded_neighbor(orb_index_1).begin(),
				orbital_index_end_1 = res1.bonded_neighbor(orb_index_1).end();
				orbital_index_1 != orbital_index_end_1; ++orbital_index_1
		){

			for(	chemical::AtomIndices::const_iterator
					orbital_index_2 = res2.bonded_neighbor(orb_index_2).begin(),
					orbital_index_end_2 = res2.bonded_neighbor(orb_index_2).end();
					orbital_index_2 != orbital_index_end_2; ++orbital_index_2
			)




				if(res1.atom_type( *orbital_index_1 ).is_lone_pair() && res2.atom_type(*orbital_index_2).is_lone_pair()){
					numeric::xyzVector<core::Real> const & orbital_xyz_1 =  res1.atom( *orbital_index_1 ).xyz();
					numeric::xyzVector<core::Real> const & orbital_xyz_2 =res2.atom( *orbital_index_2 ).xyz();
					D_container = orbital_xyz_1.distance_squared( orbital_xyz_2 );
					if(D_container < low_D ){
						low_D = D_container;
						mm_atom_name = res1.mm_atom_name(*orbital_index_1);
						bonded_neighbor_xyz = res1.atom(res1.bonded_neighbor(*orbital_index_1)[1]).xyz();
						_orbital_xyz_1 = orbital_xyz_1;
						_orbital_xyz_2=orbital_xyz_2;
						get_score=true;
					}


				}

		}
	}

	if(get_score){
		core::Real angle = cos_of(bonded_neighbor_xyz, _orbital_xyz_1, _orbital_xyz_2 );
		orb_orb_E = lookup_table_.get_energy(lookup_table_.orb, mm_atom_name, std::sqrt(low_D), angle);
		if(core::options::option[core::options::OptionKeys::lp::verbose] && orb_orb_E < 0){
			std::cout << orb_orb_E <<  " act_dist " <<   res1.atom(res1.actcoord_atoms()[1]).xyz().distance_squared(res2.atom(res2.actcoord_atoms()[1]).xyz()) << std::endl;
		}
	}





}





void OrbitalsScore::get_atom_atom_E_hpol_haro(
		core::conformation::Residue const & res1,
		core::conformation::Residue const & res2,
		core::Size const orbital,
		core::Real & hpol_E,
		core::Real & haro_E
) const
{


	numeric::xyzVector<core::Real> bonded_orbital_neighbor_xyz;
	std::string mm_atom_name(" ");
	core::Real D_container(100.0);
	core::Real D_squared(101.0);
	numeric::xyzVector<core::Real> _H_xyz;
	numeric::xyzVector<core::Real> _orbital_xyz;

	for
	(
			chemical::AtomIndices::const_iterator
			hpol_index = res2.Hpol_index().begin(),
			hpol_end = res2.Hpol_index().end();
			hpol_index != hpol_end; ++hpol_index
	){
		core::Size H_index(*hpol_index);
		numeric::xyzVector<core::Real> orbital_xyz = res1.atom(orbital).xyz();
		numeric::xyzVector<core::Real> H_xyz = res2.atom(H_index).xyz();
		D_container = orbital_xyz.distance_squared(H_xyz);
		if(D_container < D_squared && D_container< max_dist_squared_){
			D_squared = D_container;
			mm_atom_name = res1.mm_atom_name(orbital);
			bonded_orbital_neighbor_xyz =  res1.atom(res1.bonded_neighbor(orbital)[1]).xyz();
			_H_xyz = H_xyz;
			_orbital_xyz = orbital_xyz;
		}

	}

	if(D_squared < max_dist_squared_){
		core::Real angle = cos_of(bonded_orbital_neighbor_xyz, _orbital_xyz, _H_xyz );
		hpol_E += lookup_table_.get_energy(lookup_table_.hpol, mm_atom_name, std::sqrt(D_squared), angle);
	}



	D_container = 100.0;
	D_squared = 101.0;


	for
	(
			chemical::AtomIndices::const_iterator
			haro_index = res2.Haro_index().begin(),
			haro_end = res2.Haro_index().end();
			haro_index != haro_end; ++haro_index
	){
		core::Size H_index(*haro_index);
		numeric::xyzVector<core::Real> orbital_xyz = res1.atom(orbital).xyz();
		numeric::xyzVector<core::Real> H_xyz = res2.atom(H_index).xyz();
		D_container = orbital_xyz.distance_squared(H_xyz);
		if(D_container < D_squared && D_container< max_dist_squared_){
			D_squared = D_container;
			mm_atom_name = res1.mm_atom_name(orbital);
			bonded_orbital_neighbor_xyz =  res1.atom(res1.bonded_neighbor(orbital)[1]).xyz();
			_H_xyz = H_xyz;
			_orbital_xyz = orbital_xyz;
		}

	}


	if(D_squared < max_dist_squared_){

		core::Real angle = cos_of(bonded_orbital_neighbor_xyz, _orbital_xyz, _H_xyz );
		haro_E += lookup_table_.get_energy(lookup_table_.haro, mm_atom_name, std::sqrt(D_squared), angle);

	}
}














/*void
OrbitalsScore::eval_residue_pair_derivatives(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	ResSingleMinimizationData const &,
	ResSingleMinimizationData const &,
	ResPairMinimizationData const &,
	pose::Pose const &, // provides context
	EnergyMap const & weights,
	utility::vector1< DerivVectorPair > & r1_atom_derivs,
	utility::vector1< DerivVectorPair > & r2_atom_derivs
) const
{

		if(rsd1.has_orbitals()){
			res_res_hpol_haro_derivs_one_way(rsd1, rsd2, weights, r1_atom_derivs, r2_atom_derivs);
		}
		if(rsd2.has_orbitals()){
			res_res_hpol_haro_derivs_one_way(rsd2, rsd1, weights, r2_atom_derivs, r1_atom_derivs );
	}

}*/

void
OrbitalsScore::res_res_hpol_haro_derivs_one_way(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	EnergyMap const & weights,
	utility::vector1< DerivVectorPair > & res1_atom_derivs,
	utility::vector1< DerivVectorPair > & res2_atom_derivs
) const
{



	for
		(
				chemical::AtomIndices::const_iterator
				orb_index = res1.sc_orbitals().begin(),
				orb_index_end = res1.sc_orbitals().end();
				orb_index != orb_index_end; ++orb_index)
		{
			core::Size const orbital = *orb_index;

			core::Real D_squared =0;



	for
	(
			chemical::AtomIndices::const_iterator
			hpol_index = res2.Hpol_index().begin(),
			hpol_end = res2.Hpol_index().end();
			hpol_index != hpol_end; ++hpol_index
	){
		core::Size H_index(*hpol_index);
		numeric::xyzVector<core::Real> orbital_xyz = res1.atom(orbital).xyz();
		numeric::xyzVector<core::Real> H_xyz = res2.atom(H_index).xyz();
		D_squared = orbital_xyz.distance_squared(H_xyz);
		if(D_squared < max_dist_squared_ && D_squared != 0){
			std::string mm_atom_name = res1.mm_atom_name(orbital);
			numeric::xyzVector<core::Real> bonded_orbital_neighbor_xyz =  res1.atom(res1.bonded_neighbor(orbital)[1]).xyz();
			core::Real angle = cos_of(bonded_orbital_neighbor_xyz, orbital_xyz, H_xyz );
			numeric::xyzVector<core::Real> const & don_atom_xyz( res2.atom(res2.atom_base( H_index )).xyz() );

			//Vector r_H_A( orbital_xyz  - H_xyz );
			//Vector z_D_H( ( H_xyz - don_atom_xyz ).normalize() );

			Vector H_A_vector = orbital_xyz - H_xyz;
			Vector D_H_vector = H_xyz - don_atom_xyz;
			Vector B_A_vector = orbital_xyz - bonded_orbital_neighbor_xyz;

			Vector f2= lookup_table_.get_derivative(lookup_table_.hpol, mm_atom_name, std::sqrt(D_squared), angle, H_A_vector, D_H_vector, B_A_vector);


			f2 *= weights[orbitals_hpol];

			res2_atom_derivs[H_index].f2() += f2;
			Vector f1_H = cross( f2,orbital_xyz  );
			res2_atom_derivs[ H_index ].f1() += f1_H;

			f2 *= -1;

			res1_atom_derivs[orbital].f2() += f2;
			Vector f1_Acc = cross( f2,H_xyz );
			res1_atom_derivs[orbital].f1() +=f1_Acc;





		}

	}


	for
	(
			chemical::AtomIndices::const_iterator
			haro_index = res2.Haro_index().begin(),
			haro_end = res2.Haro_index().end();
			haro_index != haro_end; ++haro_index
	){
		core::Size H_index(*haro_index);
		numeric::xyzVector<core::Real> orbital_xyz = res1.atom(orbital).xyz();
		numeric::xyzVector<core::Real> H_xyz = res2.atom(H_index).xyz();
		D_squared = orbital_xyz.distance_squared(H_xyz);
		if(D_squared < max_dist_squared_ && D_squared != 0){
			std::string mm_atom_name = res1.mm_atom_name(orbital);
			numeric::xyzVector<core::Real> bonded_orbital_neighbor_xyz =  res1.atom(res1.bonded_neighbor(orbital)[1]).xyz();
			core::Real angle = cos_of(bonded_orbital_neighbor_xyz, orbital_xyz, H_xyz );
			numeric::xyzVector<core::Real> const & don_atom_xyz( res2.atom(res2.atom_base( H_index )).xyz() );

			Vector H_A_vector = orbital_xyz - H_xyz;
			Vector D_H_vector = H_xyz - don_atom_xyz;
			Vector B_A_vector = orbital_xyz - bonded_orbital_neighbor_xyz;

			//Vector r_H_A( orbital_xyz  - H_xyz );
			//Vector z_D_H( ( H_xyz - don_atom_xyz ).normalize() );

			Vector f2= lookup_table_.get_derivative(lookup_table_.haro, mm_atom_name, std::sqrt(D_squared), angle, H_A_vector, D_H_vector, B_A_vector);


			f2 *= weights[orbitals_hpol];

			res2_atom_derivs[H_index].f2() += f2;
			Vector f1_H = cross( f2,orbital_xyz  );
			res2_atom_derivs[ H_index ].f1() += f1_H;

			f2 *= -1;

			res1_atom_derivs[orbital].f2() += f2;
			Vector f1_Acc = cross( f2,H_xyz );
			res1_atom_derivs[orbital].f1() +=f1_Acc;


		}

	}



		}







}
















}
}
}
