// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file CompositeSequence.hh
/// @brief class definition for a sequence profile that represents each
/// position in a sequence as a probability distribution over the allowed amino
/// acids at that position.
/// @author James Thompson

#ifndef INCLUDED_core_sequence_CompositeSequence_HH
#define INCLUDED_core_sequence_CompositeSequence_HH

#include <core/sequence/CompositeSequence.fwd.hh>

#include <core/types.hh>
#include <core/sequence/Sequence.hh>
#include <core/sequence/SequenceProfile.hh>

#include <utility/file/FileName.fwd.hh>

#include <utility/vector1.hh>

namespace core {
namespace sequence {

class CompositeSequence : public Sequence {
	typedef std::string string;

public:

	/// @brief ctors
	CompositeSequence() {}

	/// @brief copy ctor
	CompositeSequence( CompositeSequence const & src )
		: Sequence()
	{
		*this = src;
	}

	core::Size n_seqs() const {
		return seqs_.size();
	}

	utility::vector1< SequenceOP > seqs() const {
		return seqs_;
	}

	SequenceOP seq( core::Size const idx ) const {
		//runtime_assert( idx <= seqs_.size() );
		return seqs_[idx];
	}

	/// @brief assignment operator.
	CompositeSequence & operator = ( CompositeSequence const & rhs ) {
		if ( this == &rhs ) return *this;

		seqs_.clear();
		for ( core::Size ii = 1; ii <= rhs.seqs().size(); ++ii ) {
			seqs_.push_back( rhs.seq(ii)->clone() );
		}

		id      ( rhs.id() );
		start   ( rhs.start() );
		gap_char( rhs.gap_char() );
		sequence( rhs.sequence() );
		return *this;
	}

	/// @brief dtor
	virtual ~CompositeSequence() {}

	/// @brief Returns an owning pointer to a new CompositeSequence object,
	/// with data that is a deep copy of the information in this object.
	virtual SequenceOP clone() const {
		SequenceOP new_seq_op( new CompositeSequence( *this ) );
		return new_seq_op;
	}

private:
	utility::vector1< SequenceOP > seqs_;
}; // class CompositeSequence

} // sequence
} // core

#endif
