// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author

#ifndef INCLUDED_core_sequence_SequenceMapping_HH
#define INCLUDED_core_sequence_SequenceMapping_HH

// Unit headers
#include <core/sequence/SequenceMapping.fwd.hh>

// Project headers
#include <core/conformation/signals/LengthEvent.fwd.hh>
#include <core/types.hh>

// Utility headers
#include <utility/pointer/ReferenceCount.hh>
// AUTO-REMOVED #include <utility/vector1.hh>
#include <utility/exit.hh>

// C++ headers
#include <iostream>
#include <string>

//Auto Headers
#include <utility/vector1_bool.hh>


namespace core {
namespace sequence {

class SequenceMapping : public utility::pointer::ReferenceCount {

public:
// constructors, destructors and assignment operator
	/// @brief ctor
	SequenceMapping() :
		size2_(0),
		seq1_(""),
		seq2_(""),
		start_seq2_(0)
	{}

	/// @brief ctor
	SequenceMapping( Size const s1, Size const s2 ) :
		size2_(s2),
		mapping_(s1,0)
	{}

	/// @brief convenience constructor from LengthEvent
	SequenceMapping( conformation::signals::LengthEvent const & event );

	/// @brief dtor
	~SequenceMapping();

	/// @brief copy constructor
	SequenceMapping( SequenceMapping const & src );

	SequenceMapping &
	operator=( SequenceMapping const & src );

public:

	/// @brief resize
	void resize( Size const s1, Size const s2 );

	///
	void reverse();

	///@brief size of target sequence
	Size size1() const;

	///@brief size of aligned sequence ???
	Size size2() const;

	//@brief the pose might have a missing N-terminal offset the sequence mapping accordingly
	//NOTE: set_offset( 5) followed by set_offset( -5 ) might not be an identity operation:
	//if residues are pushed below zero... there is no coming back.
	void set_offset( int setting );

	bool all_aligned() const;

	bool is_identity() const;
	bool is_identity_ignore_gaps() const;

	void size2( Size const s2 );

	void push_back( Size const al );

	void delete_source_residue( Size const pos1 );

	void show() const;

	void show( std::ostream & output ) const;

	void insert_source_residue( Size const pos1 );

	void insert_aligned_residue( Size const pos1, Size const pos2 );

	/// @brief same as insert_aligned_residue, but a couple of extra checks on size1 and size2.
	void insert_aligned_residue_safe( Size const pos1, Size const pos2 );

	void insert_target_residue( Size const pos );

	void delete_target_residue( Size const pos );

	void clear();

	// residue of aligned sequence at target position pos1
	Size   operator[]( Size const pos1 ) const;
	Size & operator[]( Size const pos1 );

	/// @brief Equality operator.
	bool operator==( SequenceMapping const & other ) const;

	/// @brief Construct an identity mapping, which means that for all positions,
	/// i in seq1 maps to i in seq2.
	inline
	static
	SequenceMapping
	identity( Size const size )
	{
		SequenceMapping id( size, size );
		for ( Size i=1; i<= size; ++i ) {
			id[i] = i;
		}
		runtime_assert( id.is_identity() && id.size1() == size );
		return id;
	}

	std::string & seq1() {
		return seq1_;
	}

	std::string & seq2() {
		return seq2_;
	}

	std::string const & seq1() const {
		return seq1_;
	}

	std::string const & seq2() const {
		return seq2_;
	}

	Size & start_seq2() {
		return start_seq2_;
	}

	Size const & start_seq2() const {
		return start_seq2_;
	}

	utility::vector1< core::Size > const & mapping() const {
		return mapping_;
	}

	std::string to_string() const;

private:
	Size size2_; // length of second sequence

	utility::vector1< Size > mapping_; // mapping_[ residue_i ] = residue_j

	std::string seq1_; //target=query sequence ... might be empty if not available
	std::string seq2_; //template sequence
 	Size start_seq2_; // which is the first seq_pos of seq2_
}; // class SequenceMapping

inline std::ostream & operator<< (
	std::ostream & out,
	SequenceMapping const & map
) {
	map.show( out );
	return out;
}

} // sequence
} // core

#endif
