// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file FragmentSampler.cc
/// @brief ab-initio fragment assembly protocol for proteins
/// @detailed
///	  Contains currently: Classic Abinitio
///
///
/// @author Oliver Lange

// Unit Headers
#include <protocols/NoesyAssign/DistanceScoreMover.hh>

// Package Headers
#include <protocols/NoesyAssign/CrossPeakList.hh>
#include <protocols/NoesyAssign/ResonanceList.hh>
#include <protocols/NoesyAssign/PeakAssignmentParameters.hh>
#include <core/id/Exceptions.hh>
// Project Headers
#include <core/scoring/constraints/AmbiguousNMRDistanceConstraint.hh>
#include <core/scoring/constraints/AmbiguousNMRConstraint.hh>
// Utility headers
#include <core/util/Tracer.hh>

//// C++ headers


static core::util::Tracer tr("devel.NoesyAssign.DistanceScoreMover");

using core::Real;
using namespace core;
using namespace util;
//using namespace core::options;
//using namespace OptionKeys;

namespace protocols {
namespace NoesyAssign {

DistanceScoreMover::DistanceScoreMover( CrossPeakList& cpl, pose::Pose const& pose ) :
  cross_peaks_( cpl ),
  count_decoys_( 0 ),
  nr_assignments_( cpl.count_assignments() ),
	peak_violation_counts_( cross_peaks_.size(), 0 ),
  final_dist_power_( 6.0 ) //default
{
  constraints_.reserve( nr_assignments_ );
	//	peak_constraints_.reserve( cross_peaks_.size() );

  for ( CrossPeakList::iterator it = cross_peaks_.begin(); it != cross_peaks_.end(); ++it ) {
    for ( CrossPeak::iterator ait = (*it)->begin(); ait != (*it)->end(); ++ait ) {
			try {
				constraints_.push_back( (*ait)->create_constraint( cpl.resonances(), pose ) );
			} catch ( core::id::EXCN_AtomNotFound& excn ) {
				tr.Error << "while setting up constraints in DistanceScoreMover: " << excn << std::endl;
				constraints_.push_back( NULL );
			}
    }
		try {
			//			peak_constraints_.push_back( (*it)->create_constraint( cpl.resonances(), pose ) );
		} catch ( core::id::EXCN_AtomNotFound& excn ) {
			tr.Error << "while setting up constraints in DistanceScoreMover: " << excn << std::endl;
			//			peak_constraints_.push_back( NULL );
		}
  }
}


void DistanceScoreMover::prepare_scoring( bool use_for_calibration /*default false */ ) {

	//assignment_distances_ = VectorReal( nr_assignments_, 0.0 );
	used_for_calibration_ = use_for_calibration;

	Size ct( 1 );
  for ( CrossPeakList::iterator it = cross_peaks_.begin(); it != cross_peaks_.end(); ++it, ++ct ) {
		if ( !used_for_calibration_ ) {
			for ( CrossPeak::iterator ait = (*it)->begin(); ait != (*it)->end(); ++ait ) {
				(*ait)->set_decoy_compatibility( 0.0 );
			}
		}
		peak_violation_counts_[ ct ] = 0;
		total_violation_count_ = 0;
  }
  count_decoys_ = 0;
	total_assigned_distances_ = 0;
}

void DistanceScoreMover::apply( pose::Pose& pose ) {
	PeakAssignmentParameters const& params( *PeakAssignmentParameters::get_instance() );

  count_decoys_++;
  SingleConstraints::const_iterator constraint_it( constraints_.begin() );
	//  PeakConstraints::const_iterator peak_constraint_it( peak_constraints_.begin() );
	active_peaks_ = 0;
	Size ct_peaks( 1 );
	if ( !used_for_calibration_ ) {
		tr.Debug << "DistanceScoreMover is not used in calibration mode " << std::endl;
	}
  for ( CrossPeakList::iterator it = cross_peaks_.begin(); it != cross_peaks_.end(); ++it, ++ct_peaks ) {
		//for each CrossPeak do the following:
		// ct_peaks .. number of peak, ct ... number of assignment / peak
		// a) iterate over all initial assignments and get effective distance (i.e., QD1 is evaluated with d^-6 averaging )
		// b1) average over all assignments (d^-6) (--> sum_dist )
		// b2) average over all selected assignments ( calibration phase --> sum_dist_filt )
		// c) cache distance in dist_buf to compute Dk in the end..

    Real dist_buf[ 2000 ];//some cheap memory buffer for distances
    runtime_assert( (*it)->n_assigned() < 2000 );
		if ( (*it)->n_assigned() == 0 ) continue;
    Size ct_assignments( 1 );
    Real sum_dist( 0.0 );
		Real sum_dist_filt( 0.0 ); //accumulate only distances where Vk > Vmin  --> used for calibration ( eq. (12)  ).
    for ( CrossPeak::iterator ait = (*it)->begin(); ait != (*it)->end(); ++ait, ++ct_assignments ) {
			runtime_assert( constraint_it != constraints_.end() );
			Real dist,invd6;
			if ( !( *constraint_it )) {
				dist=0;
				invd6=0;
// 				tr.Trace << "PeakAssignment " << (*it)->peak_id() << " " << " assignment: " << ct_assignments << " "
// 								 << (*ait)->resonance_id( 1 ) << " " << (*ait)->resonance_id( 2 )
// 								 << " has invalid constraint assign 0" << std::endl;
			} else {
				dist = (*constraint_it)->dist( pose );
				//				tr.Trace << " dist: " << dist << std::endl;
				Real invd = 1.0/dist;
				Real invd3 = invd*invd*invd;
				invd6 = invd3*invd3;
				//				tr.Trace << "invd6: " << invd6 << std::endl;
			}
			sum_dist+=invd6;
			//			tr.Trace << "sum_dist " << sum_dist << std::endl;
			///this resembles eq( 12 ) after filtering by peak volume --> used for calibration...
			//careful: these values are messed up when !used_for_calibration since peak_volume() can return 0 or similar ....
			sum_dist_filt += ( (*ait)->normalized_peak_volume() > params.min_volume_ )*invd6;
      dist_buf[ ct_assignments ] = dist;
			++constraint_it; //used this constraint.... we have created these constraints in prepare_scoring() in same sequence as used here
		}
		/// is upper bound violated ?
		if ( used_for_calibration_ ) {
			sum_dist=sum_dist_filt;
		}
		total_assigned_distances_ += sum_dist;
		//		tr.Trace << "sum_dist " << sum_dist << std::endl;
    if ( sum_dist > 0 )  {
			sum_dist=pow( sum_dist, -1.0/6.0 );
			++active_peaks_;
		}
		//		tr.Trace << "sum_dist " << sum_dist << std::endl;
		bool violated(  params.dcut_ > 0 && ( sum_dist - (*it)->distance_bound() ) > params.dcut_ );
		peak_violation_counts_[ ct_peaks ] += violated ? 1 : 0;
		total_violation_count_ += violated ? 1 : 0;

		if ( !used_for_calibration_ ) { ////NOTE THERE IS STILL A PROBLEM DUE TO SKIPPED CONSTRAINTS...
			//now add to Dk in Assignments  --- formulas (6)+(7) on p.214  eta --> final_dist_power
			//d_ak,bk is distance of individual PeakAssignment (computed in for-loop above --> dist_buf[] ).
			ct_assignments = 1;
			for ( CrossPeak::iterator ait = (*it)->begin(); ait != (*it)->end(); ++ait, ++ct_assignments ) {
				if ( dist_buf[ ct_assignments ] == 0 ) {
					//					tr.Trace << "Crosspeak: " << (*it)->peak_id() << " assignment " << ct_assignments << " has zero dist_buf " << std::endl;
				}
				if ( dist_buf[ ct_assignments ] ) {
					(*ait)->set_decoy_compatibility( (*ait)->decoy_compatibility() + pow( dist_buf[ ct_assignments ]/sum_dist, -final_dist_power_) );
				}
				if ( (*ait)->decoy_compatibility() == 0 ) {
					//					tr.Trace <<" oh shit no compatiblity... " << (*it)->peak_id() << std::endl;
					//					tr.Trace << ct_assignments << " " << dist_buf[ct_assignments] << " " << sum_dist << " " << dist_buf[ ct_assignments ]/sum_dist << " " << pow( dist_buf[ ct_assignments ]/sum_dist, -final_dist_power_) << std::endl;
				}
			} ///... divide by M (aka count_decoys_ ) in finalize_scoring
		}
	} //loop over peaks
}

void DistanceScoreMover::finalize_scoring() const {
	Size ct_peaks( 1 );
  for ( CrossPeakList::iterator it = cross_peaks_.begin(); it != cross_peaks_.end(); ++it, ++ct_peaks ) {
		for ( CrossPeak::iterator ait = (*it)->begin(); ait != (*it)->end(); ++ait ) {
			(*ait)->set_decoy_compatibility( (*ait)->decoy_compatibility()/count_decoys_ );
		}
	}
}

void DistanceScoreMover::eliminate_violated_constraints() const {
	PeakAssignmentParameters const& params( *PeakAssignmentParameters::get_instance() );
	Size ct_peaks( 1 );
  for ( CrossPeakList::iterator it = cross_peaks_.begin(); it != cross_peaks_.end(); ++it, ++ct_peaks ) {
		tr.Debug << "peak: " << (*it)->peak_id() <<" " << (*it)->filename() << " violations: " << peak_violation_counts_[ ct_peaks ] << std::endl;
		(*it)->set_eliminated_due_to_dist_violations( peak_violation_counts_[ ct_peaks ] > (params.nr_conformers_violatable_*count_decoys_) );
	}
}

core::Real DistanceScoreMover::compute_violation_percentage() const {
	Real percent_violated( 0 );
	Size ct_peaks( 1 );
  for ( CrossPeakList::iterator it = cross_peaks_.begin(); it != cross_peaks_.end(); ++it, ++ct_peaks ) {
		percent_violated+=peak_violation_counts_[ ct_peaks ]/count_decoys_;
	}
	return percent_violated / active_peaks_;
}





}
}
