// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file FragmentSampler.cc
/// @brief ab-initio fragment assembly protocol for proteins
/// @detailed
///	  Contains currently: Classic Abinitio
///
///
/// @author Oliver Lange

// Unit Headers
#include <protocols/NoesyAssign/NoesyModule.hh>

// Package Headers
#include <protocols/NoesyAssign/ResonanceList.hh>
#include <protocols/NoesyAssign/PeakFileFormat.hh>
//#include <protocols/NoesyAssign/DistanceScoreMover.hh>

// Project Headers
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
// #include <core/scoring/constraints/BoundConstraint.hh>
// #include <core/scoring/constraints/AmbiguousNMRDistanceConstraint.hh>

#include <core/pose/Pose.hh>

// for switching residue type set to centroid
#include <core/chemical/util.hh>
#include <core/chemical/ChemicalManager.fwd.hh>

// Utility headers
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/excn/Exceptions.hh>

#include <core/util/Tracer.hh>
#include <core/options/option_macros.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/util/prof.hh>

//// C++ headers
#include <math.h> //for isnan



OPT_2GRP_KEY( File, noesy, in, resonances )
OPT_2GRP_KEY( FileVector, noesy, in, peaks )
OPT_2GRP_KEY( Boolean, noesy, in, use_assignments )
OPT_2GRP_KEY( File, noesy, in, decoys )

OPT_2GRP_KEY( File, noesy, out, resonances )
OPT_2GRP_KEY( File, noesy, out, peaks )

OPT_1GRP_KEY( Boolean, noesy, no_decoys )
OPT_1GRP_KEY( Boolean, noesy, no_network )
// OPT_1GRP_KEY( Boolean, noesy, no_symm )
// OPT_1GRP_KEY( Boolean, noesy, no_cs )
// OPT_1GRP_KEY( Boolean, noesy, no_upper )
// OPT_1GRP_KEY( Boolean, noesy, no_remove_diagonal )
// OPT_1GRP_KEY( Boolean, noesy, no_calibrate )

bool protocols::NoesyAssign::NoesyModule::options_registered_( false );


bool protocols::NoesyAssign::NoesyModule::cmdline_options_activated() {
	return core::options::option[ core::options::OptionKeys::noesy::in::resonances ].user()
		&& core::options::option[ core::options::OptionKeys::noesy::in::peaks ].user();
}

void protocols::NoesyAssign::NoesyModule::register_options() {
  using namespace core::options;
  using namespace OptionKeys;

  PeakAssignmentParameters::register_options();
	PeakFileFormat_xeasy::register_options();
  if ( options_registered_ ) return;
  options_registered_ = true;

  //....
  NEW_OPT( noesy::in::resonances, "file with assigned chemical shifts", "" );
  NEW_OPT( noesy::in::peaks, "file with noesy peaks", "" );
  NEW_OPT( noesy::in::use_assignments, "when reading peaks the already existing assignments are not ignored", false );
  NEW_OPT( noesy::in::decoys, "silent file with decoys used for 3D structural compatibility test", "" );

  NEW_OPT( noesy::out::resonances, "the parsed resonances file with translated atom names etc.", "cs_out.dat" );
  NEW_OPT( noesy::out::peaks, "the parsed peaks file with assignments", "NOE_out.dat" );

  NEW_OPT( noesy::no_decoys, "check comp. with decoys", false );
  NEW_OPT( noesy::no_network, "skip network analysis", false );
//   NEW_OPT( noesy::no_symm, "check comp. with decoys", false );
//   NEW_OPT( noesy::no_cs, "check comp. with decoys", false );
//   NEW_OPT( noesy::no_upper, "check upper", false );
//   NEW_OPT( noesy::no_remove_diagonal, "", false );
//   NEW_OPT( noesy::no_calibrate, "don't calibrate the NOE distance bound", false );


}


static core::util::Tracer tr("devel.NoesyAssign.NoesyModule");

using core::Real;
using namespace core;
using namespace util;
//using namespace core::options;
//using namespace OptionKeys;
///// templates

#include <protocols/NoesyAssign/CrossPeakList.impl.hh>
#include <protocols/NoesyAssign/NoesyModule.impl.hh>


namespace protocols {
namespace NoesyAssign {



NoesyModule::NoesyModule( std::string const& fasta_sequence ) :
  crosspeaks_( NULL ),
  resonances_( new ResonanceList( fasta_sequence ) )
{
	read_input_files();
	runtime_assert( options_registered_ );
	skip_network_analysis_ = core::options::option[  options::OptionKeys::noesy::no_network ]();
}

void NoesyModule::reset() {
	crosspeaks_ = NULL;
	resonances_ = new ResonanceList( resonances_->sequence() );
	read_input_files();
}

void NoesyModule::read_input_files() {
 //read resonances
  utility::io::izstream input_file( core::options::option[ options::OptionKeys::noesy::in::resonances ]() );
  utility::io::ozstream output_file( core::options::option[ options::OptionKeys::noesy::out::resonances ]() );
  if ( input_file.good() ) {
    resonances_->read_from_stream( input_file );
    resonances_->write_to_stream( output_file );
  } else {
    tr.Error << "cannot read " << input_file << std::endl;
		throw utility::excn::EXCN_FileNotFound( core::options::option[ options::OptionKeys::noesy::in::resonances ]() );
  }

  //read peak lists
  crosspeaks_ = new CrossPeakList( resonances_ );
  Size nfiles( core::options::option[ options::OptionKeys::noesy::in::peaks ]().size() );
  for ( core::Size ifile = 1; ifile <= nfiles; ++ifile ) {
    std::string file( core::options::option[ options::OptionKeys::noesy::in::peaks ]()[ ifile ] );
    utility::io::izstream input_file( file );
    if ( input_file.good() ) {
      PeakFileFormat_xeasy format;
			format.set_filename( core::options::option[ options::OptionKeys::noesy::in::peaks ]()[ ifile ].base() );
      format.set_ignore_assignments( !core::options::option[ options::OptionKeys::noesy::in::use_assignments ]() );
			tr.Info << "reading " << file << "... " << std::endl;
      crosspeaks_->read_from_stream( input_file, format );
    } else {
      tr.Error << "cannot read " << file << std::endl;
    }
  }
}

void NoesyModule::write_assignments( std::string file_name ) {
	if ( file_name == "use_cmd_line" ) {
		file_name = core::options::option[ options::OptionKeys::noesy::out::peaks ]();
	}
	PeakFileFormat_xeasy format( resonances_ );
	format.set_write_atom_names( true );
	utility::io::ozstream output_file( file_name );
	crosspeaks_->write_to_stream( output_file, format );
}

void NoesyModule::assign( Size cycle ) {
  using namespace options;
  using namespace options::OptionKeys::noesy;
  core::io::silent::SilentFileData sfd;
  if ( !option[ no_decoys ]() ) {
		if ( option[ in::decoys ].user() ) sfd.read_file( option[ in::decoys ]() );
		if ( sfd.size() == 0 && option[ OptionKeys::in::file::silent ].user() ) sfd.read_file( option[ OptionKeys::in::file::silent ]()[ 1 ] );
  }
  assign( sfd.begin(), sfd.end(), cycle );
}

void NoesyModule::generate_constraint_files(
	 core::pose::Pose const& pose,
	 std::string const& cst_fa_file,
	 std::string const& cst_centroid_file,
   core::Size min_seq_separation ) const {

	PROF_START( NOESY_ASSIGN_GEN_CST );

	using namespace core::scoring::constraints;
  core::pose::Pose centroid_pose = pose;
  core::chemical::switch_to_residue_type_set( centroid_pose, core::chemical::CENTROID );

	ConstraintSetOP cstset = new ConstraintSet;
	ConstraintSetOP centroid_cstset = new ConstraintSet;

	crosspeaks_->generate_fa_and_cen_constraints( cstset, centroid_cstset, pose, centroid_pose, min_seq_separation );

	PROF_STOP( NOESY_ASSIGN_GEN_CST );

	PROF_START( NOESY_ASSIGN_WRITE_CST );

	ConstraintIO::write_constraints( cst_fa_file, *cstset, pose );
  ConstraintIO::write_constraints( cst_centroid_file, *centroid_cstset, centroid_pose );

	PROF_STOP( NOESY_ASSIGN_WRITE_CST );
}

}
}


