// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file FragmentSampler.cc
/// @brief ab-initio fragment assembly protocol for proteins
/// @detailed
///	  Contains currently: Classic Abinitio
///
///
/// @author Oliver Lange

// Unit Headers
#include <protocols/NoesyAssign/PeakAssignmentParameters.hh>

// Package Headers
//#include <protocols/NoesyAssign/Exceptions.hh>

// Project Headers

// Utility headers

//#include <ObjexxFCL/string.functions.hh>

//#include <utility/string_util.hh>
// #include <utility/excn/Exceptions.hh>
// #include <utility/vector1.fwd.hh>
// #include <utility/pointer/ReferenceCount.hh>
// #include <numeric/numeric.functions.hh>
// #include <core/util/prof.hh>
#include <core/util/Tracer.hh>
// #include <core/options/option.hh>
// #include <core/options/keys/abinitio.OptionKeys.gen.hh>
// #include <core/options/keys/run.OptionKeys.gen.hh>
//#include <core/options/keys/templates.OptionKeys.gen.hh>

// Utility headers
#include <core/options/option_macros.hh>


//// C++ headers
#include <cstdlib>
#include <string>





OPT_1GRP_KEY( Real, noesy_weights, chemshift )
OPT_1GRP_KEY( Real, noesy_weights, network_high )

OPT_1GRP_KEY( RealVector, noesy_weights, Vmin )
OPT_1GRP_KEY( RealVector, noesy_weights, symmetry )
OPT_1GRP_KEY( RealVector, noesy_weights, covalent )
OPT_1GRP_KEY( RealVector, noesy_weights, decoys )
OPT_1GRP_KEY( RealVector, noesy_weights, Smax )
OPT_1GRP_KEY( RealVector, noesy_weights, dcut )
OPT_1GRP_KEY( RealVector, noesy_weights, network_min )
OPT_1GRP_KEY( RealVector, noesy_weights, network_atom_min )
OPT_1GRP_KEY( RealVector, noesy_weights, calibration_target )
OPT_1GRP_KEY( Real, noesy_weights, elim_dist_viol )
OPT_1GRP_KEY( Real, noesy_weights, centroid_padding )

OPT_1GRP_KEY( Real, noesy_weights, cst_strength )
OPT_1GRP_KEY( Integer, noesy_weights, cycle )

OPT_2GRP_KEY( Boolean, noesy, network, include_reverse_dir )
OPT_2GRP_KEY( Boolean, noesy, network, allow_same_residue_connect )


bool protocols::NoesyAssign::PeakAssignmentParameters::options_registered_( false );
core::Size protocols::NoesyAssign::PeakAssignmentParameters::cycle_selector_( 0 );

protocols::NoesyAssign::PeakAssignmentParameters* protocols::NoesyAssign::PeakAssignmentParameters::instance_( NULL );

protocols::NoesyAssign::PeakAssignmentParameters*

protocols::NoesyAssign::PeakAssignmentParameters::get_instance() {
  if ( instance_ ) return instance_;
  instance_ = new PeakAssignmentParameters;
  instance_->set_options_from_cmdline();
  return instance_;
}

void protocols::NoesyAssign::PeakAssignmentParameters::set_cycle( core::Size cycle ) {
	cycle_selector_ = cycle;
	if ( instance_ ) delete instance_;
	instance_ = NULL;
}

void protocols::NoesyAssign::PeakAssignmentParameters::register_options() {
  using namespace core::options;
  using namespace OptionKeys;
  if ( options_registered_ ) return;

	//cycle independent
  NEW_OPT( noesy_weights::chemshift, "contribution of chem. shift overlap to peak volume", 0.5 );
	NEW_OPT( noesy_weights::network_high, "contribution of network anchoring to peak volume", 10.0 );

	//cycle dependent
	NEW_OPT7( noesy_weights::Vmin, "acceptable minimial volume contribution per assignment", 0.01, 0.001, 0.005, 0.01, 0.025, 0.05, 0.501 );
	NEW_OPT7( noesy_weights::symmetry, "contribution of symmetry compliance to peak volume", 10.0, 10.0, 10.0, 10.0, 1.0, 1.0, 1.0 );
  NEW_OPT7( noesy_weights::covalent, "contribution of local covalent compliance to peak volume", 10.0, 10.0, 10.0, 1.0, 1.0, 1.0, 1.0 );
  NEW_OPT7( noesy_weights::decoys, "exponent controlling contribution of decoy compliance to peak volume", 3, 3, 6, 6, 6, 6, 6 );
	NEW_OPT7( noesy_weights::Smax,"maximum cumulative contribution of symmetry, covalent and network to peak volume", 20, 20, 20, 20, 10, 10, 10 );
  NEW_OPT7( noesy_weights::dcut, "upper limit on acceptable distance violation for elimination of spurious NOESY cross peaks (A)", -1, 1.5, 0.9, 0.6, 0.3, 0.1, 0.1 );
	NEW_OPT( noesy_weights::elim_dist_viol, "number of decoys that can be violated by distance constraints before it is eliminated ->set to 1 to switch this feature off", 0.5 );

	NEW_OPT7( noesy_weights::network_min, "Threshold for acceptable lower limit of network-anchoring per residue", 1.0, 0.75, 0.5, 0.5, 0.5, 0.5, 0.5 );
  NEW_OPT7( noesy_weights::network_atom_min, "contribution of network anchoring to peak volume", 0.25, 0.25, 0.4, 0.4, 0.4, 0.4, 0.4 );
  NEW_OPT7( noesy_weights::calibration_target, "target for NOE calibration > 1 -- (A) (structure independent ) <1 (%) of models violated", 5, 0.15, 0.15, 0.1, 0.1, 0.1, 0.1 );

	NEW_OPT( noesy_weights::cst_strength, "curvature of the constraint potential after upper distance is violated", 4 );
	NEW_OPT( noesy_weights::cycle, "set to cycle 1-7 to select a set of cycle-dependent options", 1 );
	NEW_OPT( noesy_weights::centroid_padding, "if sidechain NOESY are mapped to CB add some padding", 0.5 );

	NEW_OPT( noesy::network::include_reverse_dir, "scan a->y->b and also b->y->a noesy connections to validate a->b", true );
	NEW_OPT( noesy::network::allow_same_residue_connect, "scan also a->y->b', a'->y->b and a'->y-b' with a' and b' in same residues as a and b, respectively", true);

  options_registered_ = true;
}


static core::util::Tracer tr("devel.NoesyAssign.parameters");


void protocols::NoesyAssign::PeakAssignmentParameters::set_options_from_cmdline() {
  using namespace core::options::OptionKeys;
  using namespace core::options;

	runtime_assert( options_registered_ );

	//set cycle from cmdline if not specified
	if ( cycle_selector_ == 0 ) cycle_selector_ = option[ noesy_weights::cycle ]();

	//cycle independent
  chemshift_overlap_weight_ = option[ noesy_weights::chemshift ](); //Gamma, eq. (4)
  dmax_ = 5.5;
  vmin_ = 0.1;
  vmax_ = 1.0;
  nmax_ = 20;
  nr_conformers_violatable_ = option[ noesy_weights::elim_dist_viol ](); // M/2
  network_reswise_high_ = 4.0;
	centroid_mapping_distance_padding_ = option[ noesy_weights::centroid_padding ]();

	//cycle dependent
  min_volume_ = (option[ noesy_weights::Vmin ]())[ cycle_selector_ ];
  symmetry_compliance_weight_ = option[ noesy_weights::symmetry ]()[ cycle_selector_ ]; //T, eq. (5)
  covalent_compliance_weight_ = option[ noesy_weights::covalent ]()[ cycle_selector_ ]; //O
  decoy_compatibility_exponent_ = option[ noesy_weights::decoys ]()[ cycle_selector_ ]; //eta, eq. (6)
  smax_ = option[ noesy_weights::Smax ]()[ cycle_selector_ ];
  dcut_ = option[ noesy_weights::dcut ]()[ cycle_selector_ ];
  network_reswise_min_ = option[ noesy_weights::network_min ]()[ cycle_selector_ ];
  network_atom_min_ = option[ noesy_weights::network_atom_min ]()[ cycle_selector_ ];
  calibration_target_ = option[ noesy_weights::calibration_target ]()[ cycle_selector_ ];

	network_include_reverse_dir_= option[ noesy::network::include_reverse_dir ];
  network_allow_same_residue_connect_ = option[ noesy::network::allow_same_residue_connect ];

	tr.Info << " ============== cycle-dependent parameter selection for Noesy peak assignment =============== " << std::endl;
	tr.Info << "   cycle: " << cycle_selector_ << std::endl;
	tr.Info << "----------------------------------------------------------------------------------------------" << std::endl;
	tr.Info << "        Vmin: " << min_volume_ << std::endl;
	tr.Info << "        T:    " << symmetry_compliance_weight_ << std::endl;
	tr.Info << "        O:    " << covalent_compliance_weight_ << std::endl;
	tr.Info << "        eta:  " << decoy_compatibility_exponent_ << std::endl;
	tr.Info << "        Smax: " << smax_ << std::endl;
	tr.Info << "        dcut: " << dcut_ << std::endl;
	tr.Info << "   Nmin(res): " << network_reswise_min_ << std::endl;
	tr.Info << "  Nmin(atom): " << network_atom_min_ << std::endl;
	tr.Info << " Calibration Target: " << calibration_target_ << ( calibration_target_ >= 1 ? " (A) average distance " : " fract. of constraints violate input structures " ) << std::endl;
	tr.Info << "==============================================================================================" << std::endl;

	tr.Info << " ============== cycle-independent parameter for Noesy peak assignment =============== " << std::endl;
	tr.Info << "      chemical shift agreement-weight: " << chemshift_overlap_weight_ << std::endl;
	tr.Info << "      elimination of peak if distance-bound violates more than: " << nr_conformers_violatable_ << std::endl;
	tr.Info << "----------------------------------------------------------------------------------------------" << std::endl;
	tr.Info << "      network-anchoring options: " << std::endl;
	tr.Info << "                network_include_reverse_dir:           " << (network_include_reverse_dir_ ? "yes ": "no " ) << std::endl;
	tr.Info << "                network_allow_same_residue_connection: " << (network_allow_same_residue_connect_ ? "yes ": "no " ) << std::endl;
	tr.Info << "==============================================================================================" << std::endl;

	cst_strength_ = option[ noesy_weights::cst_strength ]();
}


namespace protocols {
namespace NoesyAssign {
using namespace core;

}
}
