// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file FragmentSampler.cc
/// @brief ab-initio fragment assembly protocol for proteins
/// @detailed
///	  Contains currently: Classic Abinitio
///
///
/// @author Oliver Lange

// Unit Headers
#include <protocols/NoesyAssign/PeakFileFormat.hh>
#include <protocols/NoesyAssign/CrossPeak.hh>
#include <protocols/NoesyAssign/CrossPeakInfo.hh>
#include <protocols/NoesyAssign/ResonanceList.hh>

// Package Headers
#include <protocols/NoesyAssign/Exceptions.hh>

// Project Headers
#include <core/types.hh>
#include <core/chemical/AA.hh>

// Utility headers
#include <ObjexxFCL/format.hh>
#include <ObjexxFCL/string.functions.hh>

#include <utility/string_util.hh>
// #include <utility/excn/Exceptions.hh>
// #include <utility/vector1.fwd.hh>
// #include <utility/pointer/ReferenceCount.hh>
// #include <numeric/numeric.functions.hh>
// #include <core/util/prof.hh>
#include <core/util/Tracer.hh>
// #include <core/options/option.hh>
// #include <core/options/keys/abinitio.OptionKeys.gen.hh>
// #include <core/options/keys/run.OptionKeys.gen.hh>
//#include <core/options/keys/templates.OptionKeys.gen.hh>

//// C++ headers
#include <cstdlib>
#include <string>

#include <core/options/option_macros.hh>

OPT_1GRP_KEY( RealVector, noesy_weights, tolerances )

bool protocols::NoesyAssign::PeakFileFormat_xeasy::options_registered_( false );

void protocols::NoesyAssign::PeakFileFormat_xeasy::register_options() {
  using namespace core::options;
  using namespace OptionKeys;
  if ( options_registered_ ) return;
	NEW_OPT3( noesy_weights::tolerances, "if no #TOLERANCE is found use this for [indirect_H, direct_H, label ]", 0.04, 0.03, 0.3 );
	options_registered_ = true;
}


static core::util::Tracer tr("devel.NoesyAssign.io");

namespace protocols {
namespace NoesyAssign {

using namespace core;
PeakFileFormat_xeasy::PeakFileFormat_xeasy() :
	resonances_( NULL )
{
	runtime_assert( options_registered_ );
}

PeakFileFormat_xeasy::PeakFileFormat_xeasy( ResonanceListOP const& res )  :
	resonances_( res )
{
	runtime_assert( options_registered_ );
}

PeakFileFormat_xeasy::~PeakFileFormat_xeasy() {}

Size const TOL_H_INDIRECT( 1 );
Size const TOL_H_DIRECT( 2 );
Size const TOL_LABEL( 3 );

void PeakFileFormat_xeasy::write_peak( std::ostream& os, Size ct, CrossPeak const& cp ) const {
	std::ostringstream line_end;
	line_end << " #d " << cp.distance_bound();
	if ( cp.eliminated( false /*recompute*/, true /*do_not_compute*/) ) line_end << " #eliminated: " << cp.elimination_reason();

  os << ObjexxFCL::fmt::RJ( 6, ct ) << " ";
  // cp.write_to_stream( os );
  write_resonances( os, cp );
  write_strength( os, cp );
  write_assignments( os, cp, line_end.str() );
}

void PeakFileFormat_xeasy::write_header( std::ostream& os ) {
  Size const dim( col2proton_.size() );
  utility::vector1< std::string > atom_names;
  utility::vector1< core::Real > tolerances;
  //dimension 1 - label
  if ( info1_->has_label() ) {
    atom_names.push_back( ObjexxFCL::lowercased( info1_->label_atom_type() ) );
    tolerances.push_back( info1_->label_tolerance() );
  }

  if ( info2_->has_label() ) { //dimension 2 - label
    atom_names.push_back( ObjexxFCL::uppercased( info2_->label_atom_type() ) );
    tolerances.push_back( info2_->label_tolerance() );
  }

  //dimension 2
  atom_names.push_back( ObjexxFCL::uppercased( info2_->main_atom() ) );
  tolerances.push_back( info2_->proton_tolerance() );

	//dimension 1
  atom_names.push_back( ObjexxFCL::lowercased( info1_->main_atom() ) );
  tolerances.push_back( info1_->proton_tolerance() );

  os << "# Number of dimensions " << dim << std::endl;
  os << "#FORMAT xeasy4D" << std::endl;
  std::string cyana_str;
  for ( Size ct = 1; ct <= atom_names.size(); ct++ ) {
    os << "#INAME " << ct << " " << atom_names[ ct ] << std::endl;
    cyana_str += atom_names[ ct ];
  }
  os << "#CYANAFORMAT " << cyana_str << std::endl;
  os << "#TOLERANCE ";
  for ( Size ct = 1; ct <= tolerances.size(); ct++ ) {
    os << ObjexxFCL::fmt::RJ( 8, tolerances[ ct ] );
  }
  os << std::endl;
}

bool PeakFileFormat_xeasy::compatible_with_current_format( CrossPeak const& cp ) const {
  return ( *info1_ == cp.info_struct( 1 ) && *info2_ == cp.info_struct( 2 ) );
}

void PeakFileFormat_xeasy::set_format_from_peak( CrossPeak const& cp ) {
  info1_ = new CrossPeakInfo( cp.info_struct( 1 ) );
  info2_ = new CrossPeakInfo( cp.info_struct( 2 ) );
  col2proton_.clear();
  col2islabel_.clear();

	//dimension 1 - label
  if ( info1_->has_label() ) {
    col2proton_.push_back( 1 );
    col2islabel_.push_back( true );
  }

	//dimension 2 - label
  if ( info2_->has_label() ) {
    col2proton_.push_back( 2 );
    col2islabel_.push_back( true );
  }

  //dimension 2
  col2proton_.push_back( 2 );
  col2islabel_.push_back( false );

  //dimension 1
  col2proton_.push_back( 1 );
  col2islabel_.push_back( false );

}


void PeakFileFormat_xeasy::read_header( std::istream& is ) {
	using namespace ObjexxFCL;
  Size dim( 0 );
  utility::vector1< std::string > atom_names;
  utility::vector1< core::Real > tolerances;
	bool HN_column_labels( false ); //true if we find a HC or HN ( instead of h vs H )
  std::string line;
	std::string cyana_string("none");
  while ( true ) {
		char c='#';
		is.get(c);
		is.unget();
		//		std::cout << "c: " << c << std::endl;
		if ( c!='#' ) break;
		getline( is, line );
		//		std::cout << " line: " << line << std::endl;
    std::istringstream line_stream( line );
    std::string tag;
    line_stream >> tag;

    if ( line.find( "Number of dimensions" ) != std::string::npos ) {
      line_stream >> tag >> tag >> tag >> dim; //read Number of dimensions N
      atom_names.resize( dim, "" );
      tolerances.resize( dim, 0.0 );
    } else if ( tag == "#FORMAT" ) {
      std::string format;
      line_stream >> format;
      std::string expected_format = "xeasy" + utility::to_string( dim ) + "D";
      if ( format != expected_format ) {
				tr.Error << "Format expected: " << expected_format << " found in file: " << format << std::endl;
      }
    } else if ( tag == "#INAME" ) {
      Size index;
      std::string name;
      line_stream >> index >> name;
			if ( name == "HN" || name =="HC" || name =="H1" ) HN_column_labels = true;
			if ( name == "H1" ) name = "HN";
			if ( name == "C13" ) name = "C";
			if ( name == "N15" ) name = "N";
      if ( atom_names.size() < index ) {
				tr.Error << "only " << dim << "D  format; but " << index << " index found in #INAME line " << line << std::endl;
				continue;
      }
			if ( atom_names[ index ] != "" ) tr.Warning << "found index" << index << "in two different #INAME lines "<< std::endl;
      atom_names[ index ] = name;
    } else if ( tag == "#CYANAFORMAT" ) {
			line_stream >> cyana_string;
    } else if ( tag == "#TOLERANCE" ) {
      for ( Size i = 1; i <= dim; i++ ) {
				core::Real val;
				line_stream >> val;
				tolerances[ i ] = val;
      }
      break; //last line of header ...
    } else if ( tag[0] == '#' ) {
      tr.Warning << "[PeakFileFormat_xeasy]: ignore header line: " << line << std::endl;
      continue; //ignore unknown comments
    } else {
      tr.Error << "reading line as header: " << line << std::endl
							 << "expect TOLERANCE as last header entry " << std::endl;
      //ERROR
    }
  } //while
	using namespace core::options;
  using namespace OptionKeys;
	Real const default_tolerance_h( (dim < 4) ?
		option[ noesy_weights::tolerances ][ TOL_H_INDIRECT ] :
		option[ noesy_weights::tolerances ][ TOL_H_DIRECT ]
	);
	Real const default_tolerance_H( (dim < 3) ?
		option[ noesy_weights::tolerances ][ TOL_H_INDIRECT ] :
		option[ noesy_weights::tolerances ][ TOL_H_DIRECT ]
	);
	if ( dim < 2 ) {
		throw utility::excn::EXCN_BadInput(" problem reading peak file, no or incomplete header ");
	}

  // 	info2_ = new CrossPeakInfo( "N", 0.03, 0.5 );
  col2proton_.resize( dim );
  col2islabel_.resize( dim, false );
	if ( HN_column_labels && cyana_string != "none" ) {
		tr.Info << "use CYANA string to work out column order" << std::endl;
	}
  for ( Size i = 1; i<=dim; i++ ) {
		if ( HN_column_labels && cyana_string != "none" ) {
			atom_names[ i ]=cyana_string[ i-1 ];
		}
    if ( atom_names[ i ] == "h" || ( HN_column_labels && atom_names[ i ]=="H" ) ) {
      col2proton_[ i ] = 2;
			if ( tolerances[ i ]==0.0 ) tolerances[ i ]=default_tolerance_h;
      if ( !info2_ ) {
				info2_ = new CrossPeakInfo( uppercased( atom_names[ i ] ), "", tolerances[ i ], 0.0 );
      } else {
				info2_->set_proton( uppercased( atom_names[ i ] ), tolerances[ i ] );
      }
    } else if ( atom_names[ i ] == "H" || ( HN_column_labels && atom_names[ i ]=="HN" ) || ( HN_column_labels && atom_names[ i ]=="HC" )) {
      col2proton_[ i ] = 1;
			if ( HN_column_labels ) atom_names[ i ] = "h";
			if ( tolerances[ i ]==0.0 ) tolerances[ i ]=default_tolerance_H;
      if ( !info1_ ) {
				info1_ = new CrossPeakInfo( atom_names[ i ], "", tolerances[ i ], 0.0 );
      } else {
				info1_->set_proton( atom_names[ i ], tolerances[ i ] );
      }
    } else if ( atom_names[ i ] == "c" || atom_names[ i ] == "n" ) {
      col2proton_[ i ] = 2;
      col2islabel_[ i ] = true;
			if ( tolerances[ i ]==0.0 ) tolerances[ i ]=	option[ noesy_weights::tolerances ][ TOL_LABEL ];
      if ( !info2_ ) {
				info2_ = new CrossPeakInfo( "", uppercased( atom_names[ i ] ), 0.0, tolerances[ i ] );
      } else {
				info2_->set_label( uppercased( atom_names[ i ] ), tolerances[ i ] );
      }
    } else if ( atom_names[ i ] == "C" || atom_names[ i ] == "N" ) {
      col2proton_[ i ] = 1;
      col2islabel_[ i ] = true;
			if ( tolerances[ i ]==0.0 ) tolerances[ i ]=	option[ noesy_weights::tolerances ][ TOL_LABEL ];
      if ( !info1_ ) {
				info1_ = new CrossPeakInfo( "", atom_names[ i ], 0.0, tolerances[ i ] );
      } else {
				info1_->set_label( atom_names[ i ], tolerances[ i ] );
      }
    }
  }
	if ( !info2_  || !info1_ ) {
		throw utility::excn::EXCN_BadInput(" problem reading peak file, no or errorenous header ");
	}

	if ( col2proton_.size() == 3 ) { //only one label.. make sure that it is with proton 1
		output_diagnosis( tr.Debug );
		tr.Debug << " check if we need to swap columns " << std::endl;
		Size ct_col_1( 0 );
		for ( Size i=1; i<=3; i++ ) {
			if ( col2proton_[ i ] == 1 ) ct_col_1++;
		}
		if ( ct_col_1 == 1 ) {//ok we have label on 2 switch numbering around...
			for ( Size i=1; i<=3; i++ ) {
				col2proton_[ i ] = col2proton_[ i ]==1 ? 2 : 1;
			}
			CrossPeakInfoOP info = info2_;
			info2_=info1_;
			info1_=info;
		}
	} //swapping of columns
	info1_->set_filename( filename() );
	info2_->set_filename( filename() );
}

CrossPeakOP PeakFileFormat_xeasy::read_peak( std::istream& is ) const {

  CrossPeakOP cp;
  Size const ncol( col2proton_.size() );
  //CrossPeak factory
  runtime_assert( ncol >=2 && ncol <= 4 );
  if ( ncol == 2 ) {
    cp = new CrossPeak;
  } else if ( ncol == 3 ) {
    cp = new CrossPeak3D;
  } else {
    cp = new CrossPeak4D;
  }
  cp->set_info( 1, info1_ );
  cp->set_info( 2, info2_ );

  core::Size id;
  is >> id;
  if ( !is.good() ) {
    return NULL;
  }
  cp->set_peak_id( id );

  read_resonances( is, *cp );

  //eat two weird columns
  std::string tag;
  is >> tag; is >> tag;

  read_strength( is, *cp );

  //read e 0 -- two columns
  is >> tag; is >> tag;

	read_assignments( is, *cp );

  return cp;
}

void PeakFileFormat_xeasy::read_resonances( std::istream& is, CrossPeak &cp ) const {
  Size const ncol( col2proton_.size() );
  for ( Size icol=1; icol<=ncol; ++icol ) {
    Real val;
    Size iproton( col2proton_[ icol ] );
    bool is_label( col2islabel_[ icol ] );
    is >> val;
    if ( !is.good() ) {
      throw EXCN_FileFormat( "expected resonance value" );
    }
    if ( !is_label ) cp.proton( iproton ).set_freq( val );
    else {
      runtime_assert( cp.has_label( iproton ) );
      cp.label( iproton ).set_freq( val );
    }
  }
}


void PeakFileFormat_xeasy::write_resonances( std::ostream& os, CrossPeak const& cp ) const {
  //  cp.write_to_stream( os );
  Size const ncol( col2proton_.size() );
  runtime_assert( col2islabel_.size() == ncol );

  for ( Size icol=1; icol<=ncol; ++icol ) {
    Real val;
    Size iproton( col2proton_[ icol ] );
    bool is_label( col2islabel_[ icol ] );
    if ( !is_label ) val = cp.proton( iproton ).freq();
    else {
      runtime_assert( cp.has_label( iproton ) );
      val = cp.label( iproton ).freq();
    }
    os << ObjexxFCL::fmt::F( 8, 3, val ) << " ";
  }
}

void PeakFileFormat_xeasy::read_strength( std::istream& is, CrossPeak& cp ) const {
  core::Real val;
  is >> val;
  cp.set_volume( val );
  is >> val; //read 0.00E+00
}

void PeakFileFormat_xeasy::write_strength( std::ostream& os, CrossPeak const& cp ) const {
  os << ObjexxFCL::fmt::E( 10, 3, cp.volume() ) << " " << ObjexxFCL::fmt::E( 10, 3, 0.0 ) << " ";
}


void PeakFileFormat_xeasy::read_assignments( std::istream& is, CrossPeak& cp ) const {

  Size const ncol( col2proton_.size() );
  runtime_assert( col2islabel_.size() == ncol );
  core::Real weight( 0.0 );
  std::string line;
	bool first( true );
  while ( weight < 1.0-0.01 ) {
    if ( !getline( is, line ) ) {
			if ( first ) return;
			std::ostringstream errstr;
			errstr << cp;
      throw EXCN_FileFormat( "expected assignment value for " + errstr.str() );
    }
    std::istringstream line_stream( line );

    //only assign if all spins are assigned.
    Size vals[ 5 ];
    for ( Size icol=1; icol<=ncol; ++icol ) {
      Size val;
      line_stream >> val;
      if ( !is.good() ) {
				if ( first ) return;
				std::ostringstream errstr;
				errstr << cp;
				throw EXCN_FileFormat( "expected assignment value for " + errstr.str() );
      }
      if ( val == 0 ) {
				line_stream.setstate( std::ios_base::failbit );
				break;
      }
      vals[ icol ]=val;
    }
    Size reorder[ 5 ];//spin1 spin2 label1 label2
		first = false;
    if ( line_stream.good() && !ignore_assignments() ) {
      for ( Size icol=1; icol<=ncol; ++icol ) {
				Size val = vals[ icol ];
				Size iproton( col2proton_[ icol ] );
				bool is_label( col2islabel_[ icol ] );
				Size index = iproton + ( is_label ? 2 : 0 );
				runtime_assert( !is_label || cp.has_label( iproton ) );
				reorder[ index ] = val;
      }
			cp.add_full_assignment( reorder );
    }
    std::string tag;
    line_stream >> tag;
    if ( !line_stream.good() ) {
      weight = 1.0; //no VC -- no partial weights.
      break; //finish while loop
    }
    if ( tag == "#VC" ) {
      core::Real w;
      line_stream >> w;
      weight += w;
      line_stream >> tag;
    } else weight += 1.0;
    if ( tag != "#QU" ) {
      throw EXCN_FileFormat( "expected #QU " );
    }
    core::Real qu, sup;
    line_stream >> qu;
    line_stream >> tag;
    if ( line_stream.good() && tag != "#SUP" ) {
      throw EXCN_FileFormat( "expected #SUP" );
    }
    line_stream >> sup;
  }
}


void PeakFileFormat_xeasy::write_assignments( std::ostream& os, CrossPeak const& cp, std::string const& line_end ) const {
  Size const ncol( col2proton_.size() );
  runtime_assert( col2islabel_.size() == ncol );

  Size assignments_written( 0 );
  for ( CrossPeak::PeakAssignments::const_iterator it = cp.assignments().begin(); it != cp.assignments().end(); ++it ) {
    ++assignments_written;
    if ( assignments_written > 1 ) os << std::endl << "                                                        ";
		if ( assignments_written > 1 && cp.has_label( 1 ) && cp.has_label( 2 ) ) os << "         ";
    for ( Size icol=1; icol<=ncol; ++icol ) {
      Size val;
      Size iproton( col2proton_[ icol ] );
      bool is_label( col2islabel_[ icol ] );
      if ( !is_label ) val = (*it)->resonance_id( iproton );
      else {
				runtime_assert( cp.has_label( iproton ) );
				val = (*it)->label_resonance_id( iproton );
      }
			if ( resonances_ && write_atom_names_ ) {
				Resonance const& atom( (*resonances_)[ val ] );
				os << atom.atom()<< "   ";
			} else {
				os << ObjexxFCL::fmt::RJ( 6, val ) << " ";
			}
    }
		Real val( (*it)->normalized_peak_volume() );
		//if ( val > 0.00001 ) {
		os << "#VC " << ObjexxFCL::fmt::F( 5, 3, val );
			//		}
		os << " #W ";
		(*it)->dump_weights( os );
		if ( assignments_written == 1 )	os << line_end;
  }
}

void PeakFileFormat_xeasy::output_diagnosis( std::ostream& os ) const {
  os << "columns: ";
  for ( Size i = 1; i <= col2proton_.size(); i++ ) {
    os << col2proton_[ i ] << " ";
  }
  os << "\nlabels: ";
  for ( Size i = 1; i <= col2islabel_.size(); i++ ) {
    os << col2islabel_[ i ] << " ";
  }
  os << "\ninfo1: " << info1_->label_atom_type() << " " << info1_->proton_tolerance() << " " << info1_->label_tolerance();
  os << "\ninfo2: " << info2_->label_atom_type() << " " << info2_->proton_tolerance() << " " << info2_->label_tolerance();
  os << std::endl;
}

}
}
