// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.


/// @file protocols/ProteinInterfaceDesign/movers/TryRotamers.cc
/// @brief
/// @author Sarel Fleishman (sarelf@u.washington.edu)

// Unit headers
#include <protocols/ProteinInterfaceDesign/movers/RandomMutation.hh>
#include <protocols/ProteinInterfaceDesign/movers/RandomMutationCreator.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>
#include <protocols/docking/DockingProtocol.hh>
#include <core/scoring/hbonds/HBondOptions.hh>

#include <protocols/ProteinInterfaceDesign/dock_design_filters.hh>
#include <boost/foreach.hpp>
#define foreach BOOST_FOREACH
#include <core/chemical/util.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/pack/pack_rotamers.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/filters/Filter.hh>
#include <core/chemical/AA.hh>
#include <algorithm>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <protocols/moves/DataMap.hh>
#include <utility/Tag/Tag.hh>
#include <core/conformation/Conformation.hh>
#include <core/chemical/util.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/chemical/ResidueType.hh>
#include <protocols/ProteinInterfaceDesign/movers/DesignRepackMover.hh>
#include <protocols/moves/MinMover.hh>

#include <core/util/Tracer.hh>
#include <core/scoring/ScoreFunction.hh>
#include <numeric/random/random.hh>

namespace protocols {
namespace ProteinInterfaceDesign {
namespace movers {

using namespace core;
using namespace std;
using namespace protocols::moves;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.movers.RandomMutation" );
static core::util::Tracer ddG_TR( "protocols.ProteinInterfaceDesign.movers.RandomMutation.ddG" );
static numeric::random::RandomGenerator RG( 314173 );

std::string
RandomMutationCreator::keyname() const
{
	return RandomMutationCreator::mover_name();
}

protocols::moves::MoverOP
RandomMutationCreator::create_mover() const {
	return new RandomMutation;
}

std::string
RandomMutationCreator::mover_name()
{
	return "RandomMutation";
}

RandomMutation::RandomMutation() :
	DesignRepackMover( RandomMutationCreator::mover_name() ),
	scorefxn_( NULL )
{}

RandomMutation::~RandomMutation() {}

void
RandomMutation::apply( core::pose::Pose & pose ){
	using namespace core::pack::task;
	using namespace utility;
	using namespace core::chemical;

//	relax_mover_->apply( pose );
	core::pose::Pose const saved_pose( pose );
	DdgFilter const ddG( 10000, scorefxn_, 1, 3/*repeats*/ );
//	core::Real const ddG_before_mutation( ddG.compute( pose ) );
	PackerTaskCOP task = task_factory()->create_task_and_apply_taskoperations( pose );

	vector1< core::Size > being_designed;
	being_designed.clear();
	core::kinematics::MoveMapOP move_map = new core::kinematics::MoveMap;
	move_map->set_bb( false );
	move_map->set_jump( true );
	for( core::Size resi = 1; resi <= pose.total_residue(); ++resi ){
		if( task->residue_task( resi ).being_designed() )
			being_designed.push_back( resi );
		if( task->residue_task( resi ).being_packed() )
			move_map->set_chi( resi, true );
		else
			move_map->set_chi( resi, false );
	}

	core::Size const random_entry = being_designed[ (core::Size) floor( RG.uniform() * being_designed.size() )+1 ];
	typedef list< ResidueTypeCAP > ResidueTypeCAPList;
	ResidueTypeCAPList const & allowed( task->residue_task( random_entry ).allowed_residue_types() );
	utility::vector1< AA > allow_temp;
	allow_temp.clear();
	foreach( ResidueTypeCAP const t, allowed )
		allow_temp.push_back( t->aa() );

	AA const target_aa_1( allow_temp[ (core::Size) floor( RG.uniform() * allow_temp.size() ) + 1 ] );
	AA target_aa_2( target_aa_1 );
	while( target_aa_1 == target_aa_2 )
		target_aa_2 = allow_temp[ (core::Size) floor( RG.uniform() * allow_temp.size() ) + 1 ];

	utility::vector1< bool > allowed_aas_1;
	allowed_aas_1.clear();
	allowed_aas_1.assign( num_canonical_aas, false );
	allowed_aas_1[ target_aa_1 ] = true;
	PackerTaskOP mutate_residue_1 = task_factory()->create_task_and_apply_taskoperations( pose );
	utility::vector1< bool > allowed_aas_2;
	allowed_aas_2.clear();
	allowed_aas_2.assign( num_canonical_aas, false );
	allowed_aas_2[ target_aa_2 ] = true;
	PackerTaskOP mutate_residue_2 = task_factory()->create_task_and_apply_taskoperations( pose );

	for( core::Size resi = 1; resi <= pose.total_residue(); ++resi ){
		if( resi != random_entry ){
			mutate_residue_1->nonconst_residue_task( resi ).restrict_to_repacking();
			mutate_residue_2->nonconst_residue_task( resi ).restrict_to_repacking();
		}
		else{
			mutate_residue_1->nonconst_residue_task( resi ).restrict_absent_canonical_aas( allowed_aas_1 );
			mutate_residue_2->nonconst_residue_task( resi ).restrict_absent_canonical_aas( allowed_aas_2 );
		}
	}
	scorefxn_ = core::scoring::ScoreFunctionFactory::create_score_function( "score13" );
	scorefxn_->set_weight( core::scoring::fa_dun, 0.1 );
	scorefxn_->set_weight( core::scoring::envsmooth, 0 );
	core::scoring::methods::EnergyMethodOptions options( scorefxn_->energy_method_options() );
  options.hbond_options()->use_hb_env_dep( false );
	scorefxn_->set_energy_method_options( options );
	core::chemical::add_variant_type_to_pose_residue( pose, "SHOVE_BB", random_entry ); // for single residue docking
	pose.update_residue_neighbors();
	TR<<"Mutating residue "<<pose.residue( random_entry ).name3()<<random_entry<<" to ";
	core::pack::pack_rotamers( pose, *scorefxn_, mutate_residue_1 );
	TR<<pose.residue( random_entry ).name3()<<std::endl;
	(*scorefxn_)(pose);
//	protocols::moves::MinMover minimize( move_map, scorefxn_, "dfpmin_armijo_nonmonotone", 0.01/*tolerance*/, true/*nblist*/ );
//	minimize.apply( pose );
	protocols::docking::DockingProtocol dock( 1/*jump*/, true/*fa*/, true/*local refine*/, false/*view*/, scorefxn_, scorefxn_ );
	dock.apply( pose );
	protocols::moves::MinMover minimize( move_map, scorefxn_, "dfpmin_armijo_nonmonotone", 0.01/*tolerance*/, true/*nblist*/ );
	minimize.apply( pose );
	core::Real const ddG_after_1st_mutation( ddG.compute( pose ) );
//	pose = saved_pose;
//	core::pack::pack_rotamers( pose, *scorefxn_, mutate_residue_2 );
//	TR<<"And then mutating to "<<pose.residue( random_entry ).name3()<<std::endl;
//	(*scorefxn_)(pose);
//	relax_mover_->apply( pose );
//	core::Real const ddG_after_2nd_mutation( ddG.compute( pose ) );
//	ddG_TR<<"ddG before/after/delta: "<<ddG_after_1st_mutation<<" "<<ddG_after_2nd_mutation<<" "<<ddG_after_1st_mutation - ddG_after_2nd_mutation<<std::endl;

	char const & chain( pose.pdb_info()->chain( random_entry ) );
	int const & number( pose.pdb_info()->number( random_entry ) );
	std::string const res_name( pose.residue( random_entry ).name3());
	ddG_TR<<"ddG for "<<res_name<<" at "<<chain<<number<<" is "<<ddG_after_1st_mutation<<std::endl;
}

std::string
RandomMutation::get_name() const {
	return RandomMutationCreator::mover_name();
}

void
RandomMutation::parse_my_tag( utility::Tag::TagPtr const tag,
		protocols::moves::DataMap & data,
		protocols::filters::Filters_map const &filters,
		protocols::moves::Movers_map const &movers,
		core::pose::Pose const & pose)
{
	using namespace utility::Tag;
	using namespace core::scoring;

	DesignRepackMover::parse_my_tag( tag, data, filters, movers, pose );
	std::string const scorefxn = tag->getOption< std::string >( "scorefxn", "score12" );
	scorefxn_ = new ScoreFunction( *data.get< ScoreFunction * >( "scorefxns", scorefxn ) );
	std::string const mover_name = tag->getOption< std::string >( "relax_mover" );
	protocols::moves::Movers_map::const_iterator find_mover( movers.find( mover_name ) );
	bool const mover_found( find_mover != movers.end() );
	if( !mover_found ){
		TR.Error<<"Mover "<<mover_name<<" not found in RandomMutation parsing"<<std::endl;
		utility_exit();
	}
	relax_mover_ = find_mover->second;
}

} //movers
} //ProteinInterfaceDesign
} //protocols

