// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/RosettaScripts/util.cc
/// @brief Utility functions useful in RosettaScripts.
/// @author Sarel Fleishman (sarelf@u.washington.edu), Jacob Corn (jecorn@u.washington.edu), Rocco Moretti (rmoretti@u.washington.edu)

// Unit Headers

#include <protocols/RosettaScripts/util.hh>

// Project Headers

#include <core/types.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/conformation/Conformation.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBPoseMap.hh>
#include <core/pose/PDBInfo.hh>
#include <core/scoring/ScoreFunction.hh>
#include <boost/foreach.hpp>
#define foreach BOOST_FOREACH
#include <core/pack/task/operation/TaskOperation.hh>
#include <core/pack/task/TaskFactory.hh>
#include <protocols/moves/DataMap.hh>

// Utility Headers
#include <core/util/Tracer.hh>
#include <utility/string_util.hh>
#include <utility/vector1.hh>
#include <utility/Tag/Tag.hh>

// C++ headers

static core::util::Tracer TR( "protocols.RosettaScripts.util" );

namespace protocols {
namespace RosettaScripts {

using namespace core::scoring;
using namespace protocols::moves;
using namespace core;
using namespace std;
using utility::vector1;

// a convenience function to test whether the user has specified pdb numbering rather than rosetta numbering.
core::Size
get_resnum( utility::Tag::TagPtr const tag_ptr, core::pose::Pose const & pose, std::string const & prefix/*=""*/ ) {
	core::Size resnum( 0 );
	bool const pdb_num_used( tag_ptr->hasOption( prefix + "pdb_num" ) );
	if( pose.pdb_info().get() == NULL ){//no pdbinfo for this pose (e.g., silent file), resort to using just the residue number
		if( pdb_num_used ){
			TR<<"Bad tag: "<< *tag_ptr<<std::endl;
			utility_exit_with_message( "pdb_num used but no pdb_info found. Use res_num instead" );
			return( 0 );
		}
	}
	else{
		core::pose::PDBPoseMap const pose_map( pose.pdb_info()->pdb2pose() );
		if( pdb_num_used ) {
			std::string pdbnum( tag_ptr->getOption<std::string>( prefix + "pdb_num" ) );
			char const chain( pdbnum[ pdbnum.length() - 1 ] );
			std::stringstream ss( pdbnum.substr( 0, pdbnum.length() - 1 ) );
			core::Size number;
			ss >> number;
			resnum = pose_map.find( chain, number );
		}
	}
	if( !pdb_num_used )
		resnum = tag_ptr->getOption<core::Size>( prefix + "res_num" );

	runtime_assert( resnum );
	return( resnum );
}

/// @brief Extracts a residue number from a string.
/// @detail Recognizes two forms of numbering:
///   - Rosetta residue numbers (numbered sequentially from 1 to the last residue
///     in the pose). These have the form [0-9]+
///   - PDB numbers. These have the form [0-9]+[A-Z], where the trailing letter
///     is the chain ID.
/// @return the rosetta residue number for the string, or 0 upon an error
core::Size
parse_resnum(std::string const& resnum, core::pose::Pose const& pose) {

	string::const_iterator input_end = resnum.end();
	//Set number to the sequence of digits at the start of input [0-9]*
	string::const_iterator number_start = resnum.begin();
	string::const_iterator number_end = resnum.begin();
	while( number_end != input_end && *number_end >= '0' && *number_end <= '9' ) {
		++number_end;
	}
	//Set chain to the following characters
	string::const_iterator chain_start = number_end;
	string::const_iterator chain_end = number_end;
	while(  chain_end != input_end
		&& ('A' <= *chain_end && *chain_end <= 'Z' ||
			'a' <= *chain_end && *chain_end <= 'z' ||
			'_' == *chain_end ) )
	{
		++chain_end;
	}

	string number(number_start,number_end);
	string chain(chain_start,chain_end);

	//Require that the whole string match, and that the chain be a single char
	if( chain_end != input_end || chain.size() > 1 || number.size() < 1) {
		TR.Error << "Could not parse '" << resnum << "' into a residue number." << std::endl;
		return Size(0);
	}

	Size n;
	std::istringstream ss( number );
	ss >> n;
	if( chain.size() == 1 ) { // PDB Number
		TR.Trace << "Interpretting " << n << chain << " as a pdb number." << std::endl;
		pose::PDBInfoCOP info = pose.pdb_info();
		runtime_assert(info);
		return info->pdb2pose( chain[0], n );
	}
	else { // Rosetta Number
		TR.Trace << "Interpreting " << n << " as a Rosetta residue number." << std::endl;
		return n;
	}
}


/// @brief Extracts a list of residue numbers from a tag.
/// @details The tag should contain a comma-separated list of numbers, in either
///   pdb or rosetta format (@see parse_resnum for details)
vector1<Size>
get_resnum_list(utility::Tag::TagPtr const tag_ptr, string const& tag, pose::Pose const& pose)
{
	vector1< Size > resnums;
	if( ! tag_ptr->hasOption( tag ) ) {
		utility_exit_with_message( "No " + tag + " was found in tag." );
		return resnums;
	}

	set<Size> const resnums_set( get_resnum_list( tag_ptr->getOption< std::string >( tag ), pose ) );
	resnums.clear();
	resnums.insert( resnums.begin(), resnums_set.begin(), resnums_set.end() );
	sort( resnums.begin(), resnums.end() );
	unique( resnums.begin(), resnums.end() );

	return resnums;
}

set<Size>
get_resnum_list( std::string const str, core::pose::Pose const & pose )
{
	using namespace std;
	using namespace utility;
	set< Size > resid;

	resid.clear();
	vector1< string> const str_residues( utility::string_split( str , ',' ) );
	foreach( string const res, str_residues ){
		if( res == "" ) continue;
		core::Size const num( parse_resnum( res, pose ) );
		runtime_assert( num );
		resid.insert( num );
	}//foreach
	return resid;
}


/// @details This is essentially a shameless copy of Justin's PackRotamersMover::parse_task_operations. In truth
/// DesignRepackMover should disappear into Justin's better organized class, but this will wait... (SJF)
core::pack::task::TaskFactoryOP
parse_task_operations( utility::Tag::TagPtr const tag, protocols::moves::DataMap const & data )
{
	using namespace core::pack::task;
	using namespace core::pack::task::operation;

  if ( ! tag->hasOption("task_operations") ) return 0;

  TaskFactoryOP new_task_factory( new TaskFactory );
  std::string const t_o_val( tag->getOption<std::string>("task_operations") );
  typedef utility::vector1< std::string > StringVec;
  StringVec const t_o_keys( utility::string_split( t_o_val, ',' ) );
	TR<<"Adding the following task operations to mover "<<tag->getName()<<" called "<<tag->getOption<std::string>( "name", "no_name" )<<":\n";
  for ( StringVec::const_iterator t_o_key( t_o_keys.begin() ), end( t_o_keys.end() );
        t_o_key != end; ++t_o_key ) {
    if ( data.has( "task_operations", *t_o_key ) ) {
      new_task_factory->push_back( data.get< TaskOperation * >( "task_operations", *t_o_key ) );
			TR<<*t_o_key<<' ';
    } else {
      utility_exit_with_message("TaskOperation " + *t_o_key + " not found in DataMap.");
    }
  }
	TR<<std::endl;
  return new_task_factory;
}

/// @details Utility function to find a scorefunction from parser-provided data. This is essentially a shameless
/// copy of Justin's PackRotamersMover::parse_score_function.
core::scoring::ScoreFunctionOP
parse_score_function( utility::Tag::TagPtr const tag, protocols::moves::DataMap const & data )
{
	if ( ! tag->hasOption("scorefxn") ) return 0;
	std::string const scorefxn_key( tag->getOption<std::string>("scorefxn") );
	if ( ! data.has( "scorefxns", scorefxn_key ) ) {
		utility_exit_with_message("ScoreFunction " + scorefxn_key + " not found in DataMap.");
	}
	return data.get< ScoreFunction* >( "scorefxns", scorefxn_key );
}

///@details modifies an existing movemap according to tag
void
parse_movemap( utility::Tag::TagPtr const in_tag, core::pose::Pose const & pose, core::kinematics::MoveMapOP mm ){
	using utility::Tag::TagPtr;
	using namespace core::kinematics;

	if( in_tag() == NULL ) return;

	utility::vector1< TagPtr > const branch_tags( in_tag->getTags() );
	utility::vector1< TagPtr >::const_iterator tag_it;
	for( tag_it = branch_tags.begin(); tag_it!=branch_tags.end(); ++tag_it ){
		if( (*tag_it)->getName() == "MoveMap" ){
			break;
		}
	}
	if( tag_it == branch_tags.end() ) return;

	foreach( TagPtr const tag, (*tag_it)->getTags() ){
		std::string const name( tag->getName() );
		runtime_assert( name == "Jump" || name == "Chain" || name == "Span" );
		if( name == "Jump" ){
			core::Size const num( tag->getOption< core::Size >( "number" ) );
			bool const setting( tag->getOption< bool >( "setting" ) );
			if( num == 0 ) mm->set_jump( setting ); // set all jumps if number==0
			else mm->set_jump( num, setting );
		}
		if( name == "Chain" ){
			core::Size const num( tag->getOption< core::Size >( "number" ) );
			bool const chi( tag->getOption< bool >( "chi" ) );
			bool const bb( tag->getOption< bool >( "bb" ) );
			core::Size const chain_begin( pose.conformation().chain_begin( num ) );
			core::Size const chain_end( pose.conformation().chain_end( num ) );
			for( core::Size i( chain_begin ); i <= chain_end; ++i ){
				mm->set_chi( i, chi );
				mm->set_bb( i, bb );
			}
		}
		if( name == "Span" ){
			core::Size const begin( tag->getOption< core::Size >( "begin" ) );
			core::Size const end( tag->getOption< core::Size >( "end" ) );
			runtime_assert( end >= begin );
			bool const chi( tag->getOption< bool >( "chi" ) );
			bool const bb( tag->getOption< bool >( "bb" ) );
			for( core::Size i( begin ); i <= end; ++i ){
				mm->set_chi( i, chi );
				mm->set_bb( i, bb );
			}
		}
	}//foreach tag
}


} //RosettaScripts
} //protocols
