// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/core/io/pose_stream/CMPoseInputStream.hh
/// @brief
/// @author James Thompson

// libRosetta headers

#include <core/types.hh>

#include <core/chemical/ResidueTypeSet.fwd.hh>

#include <core/pose/util.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>

#include <core/io/pose_stream/PoseInputStream.hh>
#include <core/io/pose_stream/ExtendedPoseInputStream.hh>

#include <core/sequence/util.hh>
#include <core/sequence/SequenceAlignment.hh>

#include <core/util/Tracer.hh>

#include <core/options/option.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>

#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/ConstraintSet.hh>

#include <protocols/viewer/viewers.hh>
#include <protocols/comparative_modeling/ThreadingMover.hh>
#include <protocols/comparative_modeling/CMPoseInputStream.hh>

#include <utility/vector1.hh>
#include <utility/file/FileName.hh>

#include <numeric/random/random.hh>

#include <ObjexxFCL/string.functions.hh>

#ifdef BOINC
#include <protocols/boinc/boinc.hh>
#endif



#include <string>

namespace protocols {
namespace comparative_modeling {

static core::util::Tracer tr( "protocols.comparative_modeling" );
typedef std::string string;

bool CMPoseInputStream::has_another_pose() {
	return extended_input_->has_another_pose();
}

void CMPoseInputStream::reset() {
	extended_input_->reset();
}

void CMPoseInputStream::fill_pose(
	core::pose::Pose & pose,
	core::chemical::ResidueTypeSet const & residue_set
) {
	// check to make sure that we have more poses!
	if ( !has_another_pose() ) {
		utility_exit_with_message(
			"CMPoseInputStream: called fill_pose, but I have no more Poses!"
		);
	}

	extended_input_->fill_pose( pose, residue_set );
	preprocess_pose( pose );
#ifdef BOINC_GRAPHICS
	// attach boinc graphics pose observer
	boinc::Boinc::attach_graphics_current_pose_observer( pose );
#endif

#ifdef GL_GRAPHICS
	protocols::viewer::add_conformation_viewer(
		pose.conformation(), "threading_pose"
	);
#endif

	// do threadingy stuff here with a random template and alignment
	core::pose::Pose template_pose = get_random_template_();
	core::sequence::SequenceAlignment aln
		= get_random_alignment_( template_pose );
	ThreadingMover mover( aln, template_pose );
	mover.randomize_loop_coords( true );
	mover.build_loops( false );
	mover.apply( pose );

	// get a constraint file that should be compatible with this alignment
	string const cst_match( ObjexxFCL::uppercased(
		aln.sequence(2)->id().substr(0,5) )
	);
	add_random_constraints_( pose, cst_match );

	using core::pose::add_comment;
	using core::pose::add_score_line_string;
	add_score_line_string( pose, "template", aln.sequence(2)->id() );
	tr.Debug << "chose template " << aln.sequence(2)->id() << std::endl;
	add_comment( pose, "query_alignment   ", aln.sequence(1)->to_string() );
	add_comment( pose, "template_alignment", aln.sequence(2)->to_string() );
	tr.flush();
} // fill_pose

void CMPoseInputStream::input_from_cmd_line_() {
	using namespace core::options;
	using namespace core::options::OptionKeys;
	using namespace core::io::pose_stream;

	core::util::Tracer tr("protocols.comparative_modeling");

	extended_input_
		= new ExtendedPoseInputStream( sequence_, ntimes_ );
}

core::sequence::SequenceAlignment
CMPoseInputStream::get_random_alignment_(
	core::pose::Pose const & template_pose
) {
	runtime_assert( template_pose.pdb_info() );
	runtime_assert( alignments_.size() > 0 );

	using std::string;
	using utility::vector1;
	using utility::file::FileName;
	using core::sequence::SequenceAlignment;
	using ObjexxFCL::uppercased;

	// try to match the first four characters of the original pdb filename
	FileName fn( template_pose.pdb_info()->name() );
	string const base_fn( static_cast< string > (fn.base()) );
	string const match( ObjexxFCL::uppercased( base_fn.substr(0,4) ) );

	// filter for alignments compatible with template_name
	vector1< SequenceAlignment > filtered_alignments;
	typedef vector1< SequenceAlignment >::const_iterator vec_it;
	for ( vec_it it = alignments_.begin(), end = alignments_.end();
			it != end; ++it
	) {
		std::string const candidate_id( uppercased ( it->sequence(2)->id() ) );
		if ( candidate_id.find(match) != string::npos ) {
			filtered_alignments.push_back( *it );
		} else {
			tr.Debug << "Rejected alignment with template id "
				<< it->sequence(2)->id()
				<< " for template " << match << "." << std::endl;
			tr.Debug << "alignment is " << *it << std::endl;
		}
	}
	tr.Debug << "Selecting from among " << filtered_alignments.size()
		<< " alignments." << std::endl;

	runtime_assert( filtered_alignments.size() > 0 );
	return numeric::random::random_element( filtered_alignments );
} // get_random_alignment_

void
CMPoseInputStream::add_random_constraints_(
	core::pose::Pose & query_pose,
	std::string const & match
) {
	if ( cst_filenames_.size() == 0 ) return;

	using namespace core::scoring::constraints;

	using std::string;
	using utility::vector1;
	using utility::file::FileName;
	using core::sequence::SequenceAlignment;
	using ObjexxFCL::uppercased;

	vector1< std::string > filtered_csts;
	typedef vector1< string >::const_iterator vec_it;
	for ( vec_it it = cst_filenames_.begin(), end = cst_filenames_.end();
				it != end; ++it
	) {
		std::string const candidate_id( uppercased( *it ) );
		if ( candidate_id.find(match) != string::npos ) {
			filtered_csts.push_back( *it );
		} else {
			tr.Debug << "Rejected cst_file with name " << *it
				<< " (transformed to " << candidate_id << ") as it doesn't match "
				<< match << "." << std::endl;
		}
	}

	tr.Debug << "Picked " << filtered_csts.size() << " from among "
		<< cst_filenames_.size() << " candidates." << std::endl;

	if ( filtered_csts.size() == 0 ) return;

	std::string const cstfile(
		numeric::random::random_element( filtered_csts )
	);
	ConstraintSetOP cst_set = ConstraintIO::read_constraints(
		cstfile, new ConstraintSet, query_pose
	);
	query_pose.constraint_set( cst_set );
} // add_random_constraints

core::pose::Pose CMPoseInputStream::get_random_template_() {
	runtime_assert( template_poses_.size() > 0 );
	return numeric::random::random_element( template_poses_ );
}

} // comparative_modeling
} // protocols
