// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @brief
/// @author James Thompson

// include these first for building on Visual Studio
#include <protocols/jobdist/JobDistributors.hh>
#include <protocols/jobdist/Jobs.hh>

#include <core/sequence/util.hh>

#include <core/pose/util.hh>
#include <core/pose/Pose.hh>
#include <core/io/pdb/pose_io.hh>

#include <core/util/Tracer.hh>
#include <core/types.hh>

#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/constraints/util.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/ConstraintSet.hh>
#include <core/scoring/constraints/ConstraintSet.fwd.hh>
#include <core/io/silent/silent.fwd.hh>
#include <core/io/silent/SilentStructFactory.hh>

#include <core/chemical/ChemicalManager.hh>

#include <core/fragment/FragSet.hh>

#include <protocols/loops/loops_main.hh>
#include <protocols/loops/Loops.hh>
#include <protocols/loops/LoopRelaxMover.hh>

#include <protocols/viewer/viewers.hh>

#include <protocols/moves/Mover.hh>
#include <protocols/moves/Mover.fwd.hh>

#include <protocols/electron_density/util.hh>

#include <protocols/evaluation/util.hh>
#include <protocols/evaluation/PoseEvaluator.hh>

#include <protocols/comparative_modeling/util.hh>
#include <protocols/comparative_modeling/frag_refine.hh>
#include <protocols/comparative_modeling/CMPoseInputStream.hh>

#include <utility/exit.hh>
#include <numeric/random/random.hh>

// C++ headers
#include <iostream>
#include <string>

// option key includes
#include <core/options/option.hh>

#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>
#include <core/options/keys/james.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/cm.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
#include <core/options/keys/edensity.OptionKeys.gen.hh>

namespace protocols {
namespace comparative_modeling {

int
cm_main() {
	core::util::Tracer tr("protocols.comparative_modeling");

	using namespace core::options;
	using namespace core::options::OptionKeys;
	using namespace core::chemical;
	using namespace core::id;
	using namespace core::pose;
	using namespace core::io::silent;
	using namespace core::sequence;

	using namespace protocols::moves;
	using namespace protocols::loops;
	using namespace protocols::jobdist;

	using std::max;
	using core::Real;
	using core::Size;
	using std::string;
	using utility::vector1;

	// native initialization
	bool have_native( true );
	Pose native_pose;
	if ( option[ in::file::native ].user() ) {
		core::io::pdb::pose_from_pdb(
			native_pose, option[ in::file::native ]()
		);
		set_ss_from_phipsi( native_pose );
	} else {
		have_native = false;
	}

	evaluation::MetaPoseEvaluatorOP evaluator
		= new evaluation::MetaPoseEvaluator;
	evaluation::read_common_evaluator_options(*evaluator);

	// parameters for CM-looprelax
	Size const max_loop_rebuild   ( option[ cm::max_loop_rebuild ]() );
	Real const loop_rebuild_filter( option[ cm::loop_rebuild_filter ]() );
	vector1< core::sequence::SequenceAlignment > alns(
		read_aln( option[ cm::aln_format ](), option[ in::file::alignment ]()[1] )
	);

	vector1< core::pose::Pose > template_poses
		= templates_from_cmd_line();

	Size ntimes = option[ out::nstruct ]();
	std::string sequence = core::sequence::read_fasta_file(
		option[ in::file::fasta ]()[1]
	)[1]->sequence();

	CMPoseInputStream input( alns, template_poses, sequence, ntimes );
	if ( option[ constraints::cst_file ].user() ) {
		vector1< string > cst_files(
			option[ constraints::cst_file ]()
		);
		input.cst_filenames( cst_files );
	}

	// initialize fragments
	utility::vector1< core::fragment::FragSetOP > frag_libs;
	if ( option[ OptionKeys::loops::frag_files ].user() ) {
		read_loop_fragments( frag_libs );
	}

	ResidueTypeSetCAP rsd_set
		= ChemicalManager::get_instance()->residue_type_set(
		option[ in::file::residue_type_set ]()
	);
	Size const min_loop_size( option[ cm::min_loop_size ]() );
	string remodel( option[ OptionKeys::loops::remodel ]() );
	string const relax( option[ OptionKeys::loops::relax ]() );
	Size const nstruct = static_cast< Size > (
		max( static_cast< int > (1),
		static_cast< int > (option[ out::nstruct ]()) )
	);
	// end parameters

	// job distributor initialization
	utility::vector1< BasicJobOP > input_jobs;
	for ( Size ii = 1; ii <= nstruct; ++ii ) {
		BasicJobOP job = new BasicJob( "S", "cm", static_cast< int > (ii) );
		input_jobs.push_back( job );
	}
	BaseJobDistributorOP jobdist
		= new PlainSilentFileJobDistributor( input_jobs );
	jobdist->startup();

	int curr_nstruct = 0;
	BasicJobOP curr_job;
	while( input.has_another_pose() &&
		jobdist->next_job( curr_job, curr_nstruct )
	) {
		core::pose::Pose query_pose;
		input.fill_pose( query_pose, *rsd_set );
		initialize_ss( query_pose );

		// goofy alignment stuff
		std::string q_seq, t_seq, str_aln;
		runtime_assert(
			get_comment( query_pose, "query_alignment   ", q_seq ) &&
			get_comment( query_pose, "template_alignment", t_seq )
		);

		using core::sequence::Sequence;
		using core::sequence::SequenceAlignment;
		SequenceOP query( new Sequence ), templ( new Sequence );
		std::istringstream q_in( q_seq ), t_in( t_seq );
		query->read_data( q_in );
		templ->read_data( t_in );

		SequenceAlignment aln;
		aln.add_sequence( query );
		aln.add_sequence( templ );
		tr.Debug << "grabbed sequence alignment from Pose: " << std::endl
			<< aln << std::endl;

		// looprelax
		Loops my_loops = loops_from_alignment(
			query_pose.total_residue(), aln, min_loop_size
		);
		my_loops.choose_cutpoints( query_pose );
		tr.Debug << "loops to be rebuilt are: " << std::endl;
		tr.Debug << my_loops << std::endl;

		LoopRelaxMoverOP lr_mover( new LoopRelaxMover );
		lr_mover->frag_libs( frag_libs );
		lr_mover->loops( my_loops );
		lr_mover->relax( relax );
		lr_mover->remodel( remodel );
		lr_mover->cmd_line_csts( false );
		lr_mover->rebuild_filter( loop_rebuild_filter );
		lr_mover->n_rebuild_tries( max_loop_rebuild );
		lr_mover->copy_sidechains( false );

		MoverOP mover = lr_mover;
		// set pose for density scoring if a map was input
		// (potentially) dock map into density -- do this here so we only need to
		// dock the pose once we need to do a _little_ extra work to not score the
		// loops.
		if ( option[ edensity::mapfile ].user() ) {
			protocols::electron_density::SetupForDensityScoringMover pre_mover;
			pre_mover.mask( my_loops );
			pre_mover.apply( query_pose );
		}

		Pose pose = query_pose;

#ifdef BOINC_GRAPHICS
		// attach boinc graphics pose observer
		boinc::Boinc::attach_graphics_current_pose_observer( pose );

			// set native for graphics
		if ( !have_native ) native_pose = pose;
		boinc::Boinc::set_graphics_native_pose( native_pose );
#endif

#ifdef GL_GRAPHICS
		protocols::viewer::add_conformation_viewer(
			pose.conformation(), "cm_pose"
		);
#endif
		// rebuild loops if necessary
		if ( option[ cm::loop_close_level ]() == 3 ) {
			rebuild_loops_until_closed(
					pose, min_loop_size, max_loop_rebuild, option[ cm::loop_mover ]()
			);
		}
		mover->apply( pose );

		// remove annoying scores from looprelax
		clearPoseExtraScore( pose, "irms" );
		clearPoseExtraScore( pose, "cen_rms" );
		clearPoseExtraScore( pose, "cen_irms" );
		clearPoseExtraScore( pose, "cen_looprms" );
		clearPoseExtraScore( pose, "cen_loopcarms" );
		clearPoseExtraScore( pose, "corelen" );
		clearPoseExtraScore( pose, "corerms" );
		clearPoseExtraScore( pose, "final_looprelax_score" );

		if ( option[ james::debug ]() ) {
			using namespace core::scoring;
			using namespace core::scoring::constraints;
			ScoreFunctionOP sfxn(
					ScoreFunctionFactory::create_score_function("score3")
			);
			sfxn->set_weight(
					atom_pair_constraint,
					0.1
			);
			if ( option[ OptionKeys::constraints::cst_file ].user() ) {
					add_constraints_from_cmdline_to_pose( pose );

					std::string cstfile = get_cst_file_option();
					ConstraintSetOP cstset(
						ConstraintIO::read_constraints( cstfile, new ConstraintSet, pose )
					);
					pose.constraint_set( cstset );
			}

			if ( have_native ) {
					setPoseExtraScores(
						pose, "rms_pre_refine",
						core::scoring::CA_rmsd( pose, native_pose )
					);
			}

			using numeric::random::random_element;
			fragment_refine(
					pose,
					random_element< core::fragment::FragSetOP >( frag_libs ),
					sfxn
			);
			if ( have_native ) {
					setPoseExtraScores(
						pose, "rms_post_refine",
						core::scoring::CA_rmsd( pose, native_pose )
					);
			}
			//sfxn->show( std::cout, pose );
		}

		// output
		PlainSilentFileJobDistributor *jd =
			dynamic_cast< PlainSilentFileJobDistributor * >
			( jobdist() );

		core::io::silent::SilentStructOP ss
			= core::io::silent::SilentStructFactory::get_silent_struct_out();


		std::string curr_job_tag = curr_job->output_tag(curr_nstruct);
		ss->fill_struct( pose, curr_job_tag );

		// evaluation and output
		//evaluator->apply( pose, curr_job_tag, *ss );
		jd->dump_silent( curr_nstruct, *ss );
		tr.flush();
	} // while ( input.has_another_pose() )
	jobdist->shutdown();

	return 0;
} // cm_main

} // namespace comparative_modeling
} // namespace protocols
