// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file DockTaskFactory
/// @brief sets up the task factory for docking movers
/// @detailed
///		This contains the functions that set up the taskfactory for docking
///		movers depending on the command line options
/// @author Krishna Kilambi

#include <core/options/option.hh>
#include <core/pack/task/TaskFactory.hh>
#include <protocols/docking/DockTaskFactory.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/task/operation/RestrictToInterface.hh>
#include <protocols/toolbox/TaskOperations/RestrictChainToRepackingOperation.hh>
#include <core/conformation/Residue.hh> // for design() flag
#include <core/pack/task/operation/NoRepackDisulfides.hh>
#include <core/pack/task/operation/OperateOnCertainResidues.hh>
#include <core/pack/task/operation/ResLvlTaskOperations.hh> // PreventRepackingRLT
#include <core/pack/task/operation/ResFilters.hh> // ResidueLacksProperty
#include <core/pack/rotamer_set/UnboundRotamersOperation.hh>
#include <core/scoring/constraints/RotamerConstraint.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>
#include <core/conformation/Conformation.hh>

//for resfile reading
#include <core/options/keys/packing.OptionKeys.gen.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

#include <core/util/Tracer.hh>
#include <utility/vector1.hh>
#include <utility/tools/make_vector1.hh>

// option key includes
#include <core/options/keys/docking.OptionKeys.gen.hh>

// C++ headers
#include <string>

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("protocols.docking.DockTaskFactory");

using namespace core;

namespace protocols {
namespace docking {

// default constructor
DockTaskFactory::DockTaskFactory() : utility::pointer::ReferenceCount()
{
	set_default();
	register_options();
	init_from_options();
}

DockTaskFactory::DockTaskFactory( DockTaskFactory const & old_instance ) : utility::pointer::ReferenceCount( old_instance )
{
	resfile_ = old_instance.resfile_;
	norepack1_ = old_instance.norepack1_;
	norepack2_ = old_instance.norepack2_;
	design_chains_ = old_instance.design_chains_;
}
//destructor
DockTaskFactory::~DockTaskFactory() {}

void
DockTaskFactory::set_default()
{
	norepack1_ = false;
	norepack2_ = false;
	resfile_ = false;
	//design_chains_ = utility::tools::make_vector1( NULL );
	design_chains_.clear();

	// @TODO needs to change so that these options can be set through setters and
	// do not have to be called from the commandline
	// the following should default to true always for docking
//	ex1_ = true;
//	ex2aro_ = true;
//	include_input_sc_ = true;
}

void
DockTaskFactory::init_from_options()
{
	using namespace core::options;

	if ( option[ OptionKeys::docking::norepack1 ].user() )
		set_norepack1(option[ OptionKeys::docking::norepack1 ]());
	else
		set_norepack1(false);
	if ( option[ OptionKeys::docking::norepack2 ].user() )
		set_norepack2(option[ OptionKeys::docking::norepack2 ]());
	else
		set_norepack2(false);
	// why is it like this? should true be the default value?
	if ( option[ OptionKeys::packing::resfile].user() )
		resfile_ = true;

	// @TODO should be packing level, not docking level
	if ( option[ OptionKeys::docking::design_chains ].user() ) {
		utility::vector1< std::string > chains = option[ OptionKeys::docking::design_chains ]();
		for ( core::Size i = 1; i <= chains.size(); ++i ) {
			design_chains_.push_back( chains[i][0] );
		}
	}
}

void
DockTaskFactory::register_options()
{
	using namespace core::options;

	option.add_relevant( OptionKeys::docking::norepack1 );
	option.add_relevant( OptionKeys::docking::norepack2 );
	option.add_relevant( OptionKeys::packing::resfile );
}

void
DockTaskFactory::create_and_attach_task_factory(
	DockingHighResOP docker,
	core::pose::Pose & pose
) const
{
	using namespace core::pack::task;
	using namespace core::pack::task::operation;

	TaskFactoryOP tf = new TaskFactory();

//	if ( init_tf_ ) tf = new TaskFactory( *init_tf_ );

	// check to see which specific chains are being redesigned
	// set restrict_to_repacking on all other chains
	if ( design_chains_.size() > 0 ) {
		// check for movable jumps
		for ( core::Size i = 1; i <= pose.num_jump(); ++i ) {
			core::Size const cutpoint = pose.fold_tree().cutpoint_by_jump( i );
			char chain = pose.pdb_info()->chain( pose.pdb_info()->number( cutpoint ) );
			if ( find( design_chains_.begin(), design_chains_.end(), chain ) != design_chains_.end() ) {
				tf->push_back( new protocols::toolbox::TaskOperations::RestrictChainToRepackingOperation( chain ) );
			}
		}
	} else {
		tf->push_back( new RestrictToRepacking );
	}

	tf->push_back( new InitializeFromCommandline );
	tf->push_back( new IncludeCurrent );
	tf->push_back( new NoRepackDisulfides );
	if( resfile_ ) tf->push_back( new ReadResfile );

	// DockingNoRepack only works over the first rb_jump in movable_jumps
	// In a 2-body case this separates 1 & 2 based on the only cutpoint
	// In a multibody case, this separates 1 & 2 based on the first cutpoint
	if ( norepack1_ ) tf->push_back( new DockingNoRepack1( docker->movable_jumps()[1] ) );
	if ( norepack2_ ) tf->push_back( new DockingNoRepack2( docker->movable_jumps()[1] ) );

	// incorporating Ian's UnboundRotamer operation.
	// note that nothing happens if unboundrot option is inactive!
	core::pack::rotamer_set::UnboundRotamersOperationOP unboundrot = new core::pack::rotamer_set::UnboundRotamersOperation();
	unboundrot->initialize_from_command_line();
	operation::AppendRotamerSetOP unboundrot_operation = new operation::AppendRotamerSet( unboundrot );
	tf->push_back( unboundrot_operation );

	//@TODO pose needed for this, set it up somewhere else
	//	core::scoring::constraints::load_unboundrot(pose); // adds scoring bonuses for the "unbound" rotamers, if any

	// note that RestrictToInterfaceOperation is added during set_dock_mcm_protocol
	docker->set_task_factory(tf);
}


} // namespace docking
} // namespace protocols
