// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file .cc file for enzdes cacheable observer
/// @brief
/// @author Florian Richter, floric@u.washington.edu, september 09


//unit headers
#include <protocols/enzdes/EnzdesCacheableObserver.hh>

//package headers
#include <protocols/toolbox/match_enzdes_util/EnzdesCstCache.hh>
#include <protocols/enzdes/EnzdesSeqRecoveryCache.hh>
#include <protocols/enzdes/EnzdesLoopsFile.hh>

//project headers
#include <core/conformation/Conformation.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/signals/LengthEvent.hh>
#include <core/pose/Pose.hh>
#include <core/pose/datacache/CacheableObserverType.hh>
#include <core/pose/datacache/ObserverCache.hh>
#include <core/sequence/SequenceMapping.hh>

//utility headers
#include <utility/signals/Link.hh>

//Auto Headers
#include <platform/types.hh>
#include <core/types.hh>
#include <core/chemical/ResidueType.hh>
#include <core/conformation/signals/LengthEvent.hh>
#include <map>


namespace protocols {
namespace enzdes {


EnzdesCacheableObserverOP
get_enzdes_observer(
	core::pose::Pose & pose )
{
	using namespace core::pose::datacache::CacheableObserverType;
	using namespace core::pose::datacache;

	if( !pose.observer_cache().has( ENZDES_OBSERVER ) ){

		EnzdesCacheableObserverOP enz_obs = new EnzdesCacheableObserver();
		pose.observer_cache().set( ENZDES_OBSERVER, enz_obs, true );
		//std::cout << " setting new cacheable observer for pose " << std::endl;
	}
	CacheableObserverOP enz_obs = pose.observer_cache().get_ptr( ENZDES_OBSERVER );
	//return (pose.observer_cache().get_ptr( ENZDES_OBSERVER ) );
	//std::cout << " returning nonconst enzdes observer " << std::endl;
	return utility::pointer::dynamic_pointer_cast< EnzdesCacheableObserver >( enz_obs );
	//return static_cast< EnzdesCacheableObserverOP > (enz_obs);
}

EnzdesCacheableObserverCOP
get_enzdes_observer(
	core::pose::Pose const & pose )
{
	using namespace core::pose::datacache::CacheableObserverType;
	using namespace core::pose::datacache;

	//const access: if cacheable observer hasn't been set, return NULL pointer
	if( !pose.observer_cache().has( ENZDES_OBSERVER ) ) return NULL;

	CacheableObserverCOP enz_obs = pose.observer_cache().get_const_ptr( ENZDES_OBSERVER );
	//std::cout << " returning const enzdes observer with val " << enz_obs << std::endl;
	return utility::pointer::static_pointer_cast< EnzdesCacheableObserver const >( enz_obs );
	//return static_cast< EnzdesCacheableObserverCOP > (pose.observer_cache().get_const_ptr( ENZDES_OBSERVER ) );
}

EnzdesCacheableObserver::EnzdesCacheableObserver()
	: CacheableObserver(),
		cst_cache_(NULL),
		seq_recovery_cache_(NULL),
		enz_loops_file_(NULL)
{
	lig_rigid_body_confs_.clear();
}

EnzdesCacheableObserver::EnzdesCacheableObserver( EnzdesCacheableObserver const & other )
	: CacheableObserver( other ),
		cst_cache_( NULL ),
		seq_recovery_cache_(NULL),
		enz_loops_file_(other.enz_loops_file_ ),
		lig_rigid_body_confs_( other.lig_rigid_body_confs_ )
{
	if( other.cst_cache_ ) cst_cache_ = new toolbox::match_enzdes_util::EnzdesCstCache( *(other.cst_cache_) );
	if( other.seq_recovery_cache_ ) seq_recovery_cache_ = new EnzdesSeqRecoveryCache( *(other.seq_recovery_cache_) );
}

EnzdesCacheableObserver::~EnzdesCacheableObserver(){}

core::pose::datacache::CacheableObserverOP
EnzdesCacheableObserver::clone()
{
	return new EnzdesCacheableObserver( *this );
}

core::pose::datacache::CacheableObserverOP
EnzdesCacheableObserver::create()
{
	return new EnzdesCacheableObserver();
}

bool
EnzdesCacheableObserver::is_attached() const {
	return length_event_link_.valid(); }

void
EnzdesCacheableObserver::attach_impl( core::pose::Pose & pose ){

	length_event_link_ = pose.conformation().attach_length_obs( &EnzdesCacheableObserver::on_length_change, this );

}

void
EnzdesCacheableObserver::detach_impl(){
	length_event_link_.invalidate();
}

void
EnzdesCacheableObserver::on_length_change( core::conformation::signals::LengthEvent const & event ){

	if ( event.tag == core::conformation::signals::LengthEvent::INVALIDATE ) {
		//don't know what the best behaviour is in this case
		//probably nothing, because pose destruction is imminent
		return;
	}
	core::sequence::SequenceMapping smap( event );
	if( cst_cache_ ) cst_cache_->remap_resid( smap );
	if( seq_recovery_cache_ ) seq_recovery_cache_ -> remap_residues( smap );
}


void
EnzdesCacheableObserver::set_cst_cache(
	toolbox::match_enzdes_util::EnzdesCstCacheOP cst_cache )
{
	cst_cache_ = cst_cache;
}

toolbox::match_enzdes_util::EnzdesCstCacheOP
EnzdesCacheableObserver::cst_cache()
{
	return cst_cache_;
}

toolbox::match_enzdes_util::EnzdesCstCacheCOP
EnzdesCacheableObserver::cst_cache() const
{
	return cst_cache_;
}

std::map< core::Size, utility::vector1< core::conformation::ResidueCOP > > const &
EnzdesCacheableObserver::lig_rigid_body_confs() const{
	return lig_rigid_body_confs_;
}


void
EnzdesCacheableObserver::set_rigid_body_confs_for_lig(
	core::Size seqpos,
 	utility::vector1< core::conformation::ResidueCOP > const & rg_confs
)
{
	erase_rigid_body_confs_for_lig( seqpos );
	utility::vector1< core::conformation::ResidueCOP > resvec;
	for( core::Size i = 1; i <= rg_confs.size(); ++i ){
		core::conformation::ResidueOP res( rg_confs[i]->clone());
		res->seqpos( seqpos ); //security measure
		resvec.push_back( res );
	}
	lig_rigid_body_confs_.insert( std::pair< core::Size, utility::vector1< core::conformation::ResidueCOP > >( seqpos, resvec ) );

	//conf_it = lig_rigid_body_confs_.find( seqpos );
	//std::cout << "enzdes cache observer rb confs for position " << seqpos << " are being set. " << conf_it->second.size() << " rb positions were set" << std::endl;
}

void
EnzdesCacheableObserver::erase_rigid_body_confs_for_lig(
	core::Size seqpos )
{
	std::map< core::Size, utility::vector1< core::conformation::ResidueCOP > >::iterator conf_it( lig_rigid_body_confs_.find( seqpos ) );
	if( conf_it != lig_rigid_body_confs_.end() ) lig_rigid_body_confs_.erase( conf_it );
}

EnzdesSeqRecoveryCacheOP EnzdesCacheableObserver::get_seq_recovery_cache(){
	return seq_recovery_cache_;
}

EnzdesSeqRecoveryCacheCOP EnzdesCacheableObserver::get_seq_recovery_cache() const {
	return seq_recovery_cache_;
}

void EnzdesCacheableObserver::set_seq_recovery_cache(
	EnzdesSeqRecoveryCacheOP seq_recovery_cache
){
	seq_recovery_cache_ = seq_recovery_cache;
}

void
EnzdesCacheableObserver::set_enzdes_loops_file(
	EnzdesLoopsFileCOP loopfile_in
){
	enz_loops_file_ = loopfile_in;
}

EnzdesLoopsFileCOP
EnzdesCacheableObserver::enzdes_loops_file() const
{
	return enz_loops_file_;
}



} // enzdes
} //protocols
