// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/filters/SecondaryStructureFilter.cc
/// @brief filter structures by sheet topology
/// @detailed
/// @author Nobuyasu Koga ( nobuyasu@uw.edu )

// Unit Headers
#include <protocols/fldsgn/filters/SecondaryStructureFilter.hh>
#include <protocols/fldsgn/filters/SecondaryStructureFilterCreator.hh>

// Project Headers
#include <protocols/fldsgn/BluePrint.hh>
#include <core/types.hh>
#include <core/pose/Pose.hh>

// Utility headers
#include <core/util/Tracer.hh>

// Parser headers
#include <protocols/filters/Filter.hh>
#include <utility/Tag/Tag.hh>

//Auto Headers
#include <protocols/jobdist/Jobs.hh>


//// C++ headers
static core::util::Tracer tr("protocols.fldsgn.filters.SecondaryStructureFilter");

namespace protocols {
namespace fldsgn {
namespace filters {

// @brief default constructor
SecondaryStructureFilter::SecondaryStructureFilter():
	Filter( "SecondaryStructure" ),
	filtered_ss_( "" )
{}


// @brief constructor with arguments
SecondaryStructureFilter::SecondaryStructureFilter( String const & ss ):
	Filter( "SecondaryStructure" ),
	filtered_ss_( ss )
{}

// @brief copy constructor
SecondaryStructureFilter::SecondaryStructureFilter( SecondaryStructureFilter const & rval ):
	Super( rval ),
	filtered_ss_( rval.filtered_ss_ )
{}

// @brief set filtered secondary structure
void SecondaryStructureFilter::filtered_ss( String const & ss )
{
	filtered_ss_ = ss;
}

// @brief returns true if the given pose passes the filter, false otherwise.
// In this case, the test is whether the give pose is the topology we want.
bool SecondaryStructureFilter::apply( Pose const & pose ) const
{

	runtime_assert( pose.total_residue() == filtered_ss_.length() );

	for( Size i=1; i<=pose.total_residue(); i++ ){
		String sec = filtered_ss_.substr( i-1, 1 );
		if( *sec.c_str() == 'D' ){
			continue;
		}else{
			if( *sec.c_str() != pose.secstruct( i ) ){
				tr << "Filtering failed: current/filterd " << pose.secstruct() << '/' << filtered_ss_ << std::endl;
				return false;
			}
		}
	}

	tr << "Successfully " << filtered_ss_ << " was filtered. " << std::endl;
	return true;

} // apply_filter

/// @brief parse xml
void
SecondaryStructureFilter::parse_my_tag(
	TagPtr const tag,
	DataMap &,
	Filters_map const &,
	Movers_map const &,
	Pose const & )
{
	using protocols::fldsgn::BluePrint;

 	filtered_ss_ = tag->getOption<String>( "ss", "" );

	String blueprint = tag->getOption<String>( "blueprint", "" );
	if( blueprint != "" ) {
		BluePrint blue( blueprint );
		filtered_ss_ = blue.secstruct();
	}

	if( filtered_ss_ == "" ){
		tr.Error << "Error!,  option of topology is empty." << std::endl;
		runtime_assert( false );
	}
	tr << filtered_ss_ << " is filtred " << std::endl;
}

protocols::filters::FilterOP
SecondaryStructureFilterCreator::create_filter() const { return new SecondaryStructureFilter; }

std::string
SecondaryStructureFilterCreator::keyname() const { return "SecondaryStructureFilter"; }


} // filters
} // fldsgn
} // protocols
