// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// Copyright in the Rosetta software belongs to the developers and their institutions.
// For more information, see www.rosettacommons.org.

/// @file ./src/protocols/topology/BetaAlphaBetaMotif.cc
/// @brief  class for beta alpha beta motifs
/// @author Nobuyasu Koga ( nobuyasu@u.washington.edu )

// unit headers
#include <protocols/fldsgn/topology/BetaAlphaBetaMotif.hh>
#include <protocols/fldsgn/topology/SS_Info2.hh>

// project headers
#include <core/types.hh>
#include <core/util/Tracer.hh>
#include <protocols/fldsgn/topology/Sheet.hh>

// utility headers
#include <utility/exit.hh>
#include <utility/pointer/ReferenceCount.hh>

// C++ headers
#include <cassert>

// Numeric headers
#include <numeric/xyzVector.hh>
#include <numeric/conversions.hh>

#include <ObjexxFCL/format.hh>

static core::util::Tracer TR( "protocols.fldsgn.topology.BetaAlphaBetaMotif" );

using namespace core;

namespace protocols {
namespace fldsgn {
namespace topology {

/// @brief default constructor
BetaAlphaBetaMotif::BetaAlphaBetaMotif():
	strand1_( 0 ),
	strand2_( 0 ),
	helix_( 0 ),
	cross_over_( 0 ),
	left_handed_( false ),
	hs_dist_( 0.0 ),
	hs_angle1_( 0.0 ),
	hs_angle2_( 0.0 )
{}

/// @brief value constructor
BetaAlphaBetaMotif::BetaAlphaBetaMotif(
  Size const & strand1,
	Size const & strand2,
	Size const & helix,
	Size const & cross_over ) :
	strand1_( strand1 ),
	strand2_( strand2 ),
	helix_( helix ),
	cross_over_( cross_over ),
	left_handed_( false ),
	hs_dist_( 0.0 ),
	hs_angle1_( 0.0 ),
	hs_angle2_( 0.0 )
{}

/// @brief copy constructor
BetaAlphaBetaMotif::BetaAlphaBetaMotif( BetaAlphaBetaMotif const & s ):
	ReferenceCount(),
	strand1_( s.strand1_ ),
	strand2_( s.strand2_ ),
	helix_( s.helix_ ),
	cross_over_( s.cross_over_ ),
	left_handed_( s.left_handed_ ),
	hs_dist_( s.hs_dist_ ),
	hs_angle1_( s.hs_angle1_ ),
	hs_angle2_( s.hs_angle2_ )
{}

/// @brief destructor
BetaAlphaBetaMotif::~BetaAlphaBetaMotif(){}

/// @brief IO operator
std::ostream & operator<<(std::ostream & out, const BetaAlphaBetaMotif & s ) {

	if( s.is_lefthanded() ) {
		out << "# " << s.strand1() << "-" << s.helix() << "-" << s.strand2() << "." << s.cross_over() << "." << "Left_handed";
	} else {
		out << "# " << s.strand1() << "-" << s.helix() << "-" << s.strand2() << "." << s.cross_over();
	}
	out << " " << s.hs_dist() << " " << s.hs_angle1() << " " << s.hs_angle2() << std::endl;

	return out;
}


/// @brief
void
BetaAlphaBetaMotif::calc_geometry( SS_Info2_OP const & ssinfo )
{
	using core::Vector;
	using protocols::fldsgn::topology::Helix;
	using protocols::fldsgn::topology::Strand;
	using protocols::fldsgn::topology::Helices;
	using protocols::fldsgn::topology::Strands;


  using ObjexxFCL::fmt::I;
  using ObjexxFCL::fmt::F;
  using ObjexxFCL::fmt::A;

	if( ! ssinfo->bbpos_is_set() ) return;
	runtime_assert( ssinfo->bbpos_is_set() );

	Helix  const & hx ( *ssinfo->helix( helix_ ) );
	Strand const & s1 ( *ssinfo->strand( strand1_ ) );
	Strand const & s2 ( *ssinfo->strand( strand2_ ) );

	if( ! s1.is_geometry_initialized() ) return;
	if( ! s2.is_geometry_initialized() ) return;
	if( ! hx.is_geometry_initialized() ) return;

	// define middle of helix
	Vector const hmid = hx.mid_pos();

	// define beta sheet plane
	Vector const s1_mid = s1.mid_pos();
	Vector const s2_mid = s2.mid_pos();
	Vector const v1 = ( s2_mid - s1_mid ).normalized();
	Vector const v2 = ( s1.orient() + s2.orient() ).normalized();
	Vector sheet_plane = v2.cross( v1 );

	// check handness
	Vector const v3 = ( hmid - s1_mid ).normalized();
	Real d = v3.cross( v1 ).dot( s1.orient() );
	if( d > 0 ) left_handed( true );

	// distance between helix and sheet
	hs_dist_ = sheet_plane.normalized().dot( hmid - s1_mid );

//Size k(1);
//for( Size i=0; i<25; i++ ) {
//	for( Size j=0; j<25; j++ ) {
//		k++;
//		Real m = 2*i - 25.0;
//		Real n = 2*j - 25.0;
//		Vector hh = m*v1 + n*v2 + s1_mid;
//		std::cout << "ATOM " << I( 6, k ) << "  H   ALA " << "B" << I( 4, k ) << "    "
//							<< F( 8, 3, hh.x() ) << F( 8, 3, hh.y() )  << F( 8, 3, hh.z() ) << std::endl;
//	}
//}

	// calc angles between strands and helix
	Vector const s1_on_sheet = s1.orient().project_parallel( v1 ) + s1.orient().project_parallel( v2 );
	Vector const s2_on_sheet = s2.orient().project_parallel( v1 ) + s2.orient().project_parallel( v2 );
	Vector const hx_on_sheet = hx.orient().project_parallel( v1 ) + hx.orient().project_parallel( v2 );
	Real ori1 = cross( hx_on_sheet, s1_on_sheet ).dot( sheet_plane );
	Real ori2 = cross( hx_on_sheet, s2_on_sheet ).dot( sheet_plane );
	hs_angle1_ = numeric::conversions::degrees( angle_of( s1_on_sheet, hx_on_sheet ) );
	hs_angle2_ = numeric::conversions::degrees( angle_of( s2_on_sheet, hx_on_sheet ) );

	if( ori1 < 0 ) hs_angle1_ = -1*hs_angle1_;
	if( ori2 < 0 ) hs_angle2_ = -1*hs_angle2_;

	// TR.Debug << d << " " << hs_dist_ << " " << hs_angle1_ << " " << hs_angle2_ << std::endl;

}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
/// @brief default constructor
BetaAlphaBetaMotifSet::BetaAlphaBetaMotifSet() {}

/// @brief value constructor
BetaAlphaBetaMotifSet::BetaAlphaBetaMotifSet( BetaAlphaBetaMotifs const & bab_motifs ):
	bab_motifs_( bab_motifs )
{}


/// @brief value constructor
BetaAlphaBetaMotifSet::BetaAlphaBetaMotifSet( SS_Info2_OP const & ssinfo, SheetSetOP const & sheet_set )
{
	set_babmotifs( ssinfo, sheet_set );
	calc_geometry( ssinfo );
}


/// @brief copy constructor
BetaAlphaBetaMotifSet::BetaAlphaBetaMotifSet( BetaAlphaBetaMotifSet const & s ):
	ReferenceCount(),
	bab_motifs_( s.bab_motifs_ )
{}

/// @brief destructor
BetaAlphaBetaMotifSet::~BetaAlphaBetaMotifSet(){}

/// @brief add BetaAlphaBetaMotif
void
BetaAlphaBetaMotifSet::push_back( BetaAlphaBetaMotifOP const & bop )
{
	bab_motifs_.push_back( bop );
}

/// @brief
void
BetaAlphaBetaMotifSet::clear(){
	bab_motifs_.clear();
}

/// @brief
BetaAlphaBetaMotifs const &
BetaAlphaBetaMotifSet::bab_motifs() const
{
	return bab_motifs_;
}

/// @brief
BetaAlphaBetaMotifOP const &
BetaAlphaBetaMotifSet::bab_motif( Size const & i ) const
{
		runtime_assert( i <= bab_motifs_.size() );
		return bab_motifs_[ i ];
}

/// @brief
std::ostream & operator<<( std::ostream & out, const BetaAlphaBetaMotifSet & s )
{
	out << "### BetaAlphaBetaMotif Info " << std::endl;
	out << "# strand1-helix-strand2.num_crossover distance angle1 angle2 " << std::endl;
	for( BetaAlphaBetaMotifs::const_iterator iter = s.bab_motifs().begin(),
				 iter_end = s.bab_motifs().end(); iter != iter_end; ++iter ) {
		BetaAlphaBetaMotif const & bab( **iter );
		out << bab;
 	}
	return out;
}

/// @brief set bab motif
void
BetaAlphaBetaMotifSet::set_babmotifs( SS_Info2_OP const & ssinfo, SheetSetOP const & sheet_set )
{
	using protocols::fldsgn::topology::Helix;
	using protocols::fldsgn::topology::Strand;
	using protocols::fldsgn::topology::Helices;
	using protocols::fldsgn::topology::Strands;

	Helices helices = ssinfo->helices();
	Strands strands = ssinfo->strands();

	if( strands.size() < 2 ) {
		return;
	}

	for( Size ist1=1; ist1<=strands.size()-1 ; ist1++ )  {

		Size ist2 = ist1+1;
		Strand const & s1 ( *strands[ ist1 ] );
		Strand const & s2 ( *strands[ ist2 ] );

		Size const sheet_num1 ( sheet_set->which_sheet( ist1 ) );
		Size const sheet_num2 ( sheet_set->which_sheet( ist2 ) );
		if( sheet_num1 == 0 || sheet_num2 == 0 ) continue;  // in the case, ist1 or ist2 does not belong to sheet
		if( sheet_num1 != sheet_num2 ) continue;

		Size const ord1 = sheet_set->sheet( sheet_num1 )->strand_order( ist1 );
		Size const ord2 = sheet_set->sheet( sheet_num2 )->strand_order( ist2 );

		utility::vector1< Real > orients( sheet_set->sheet( sheet_num1 )->orient_strands() );
		if( orients[ ord1 ] != orients[ ord2 ] ) continue;

		Size ih( 0 );
		for( Helices::const_iterator jt=helices.begin(), jte=helices.end(); jt!=jte; ++jt ) {

			ih++;
			Helix const & hx( **jt );
			if( hx.begin() > s2.end() ) continue;

			if( hx.begin() > s1.end() && hx.end() < s2.begin() ) {
				Size cross_over = Size ( std::abs( Real(ord1) - Real(ord2) ) ) - 1;
				bab_motifs_.push_back( new BetaAlphaBetaMotif( ist1, ist2, ih, cross_over ) );
				TR.Debug << ist1 << " " << ist2 << " " << ih << " " << cross_over << std::endl;
			}
		}//helices

	}//strands


} // SheetTopology::set_bab_motifs


/// @brief
void
BetaAlphaBetaMotifSet::calc_geometry( SS_Info2_OP const & ssinfo )
{
	if( ! ssinfo->bbpos_is_set() ) return;
	runtime_assert( ssinfo->bbpos_is_set() );
	for( BetaAlphaBetaMotifs::iterator it=bab_motifs_.begin(), ite=bab_motifs_.end(); it!=ite; ++it ) {
		BetaAlphaBetaMotifOP const & babm( *it );
		babm->calc_geometry( ssinfo );
	}
}


} // namespace topology
} // namespace fldsgn
} // namespace protocols

