// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// Copyright in the Rosetta software belongs to the developers and their institutions.
// For more information, see www.rosettacommons.org.

/// @file ./src/protocols/src/fldsgn/topology/HSSTriplet.cc
/// @brief
/// @author Nobuyasu Koga ( nobuyasu@u.washington.edu )

// unit headers
#include <protocols/fldsgn/topology/HSSTriplet.hh>
#include <protocols/fldsgn/topology/SS_Info2.hh>

#include <core/util/Tracer.hh>
#include <utility/exit.hh>
#include <ostream>
#include <utility/string_util.hh>
#include <boost/lexical_cast.hpp>
#include <numeric/xyzVector.hh>

static core::util::Tracer TR( "protocols.topology.HSSTriplet" );

namespace protocols {
namespace fldsgn {
namespace topology {


/// @value constructor
HSSTriplet::HSSTriplet( String const & hss )
{
	using utility::string_split;

	std::vector< String > parts( string_split( hss, '-' ) );
	runtime_assert( parts.size() == 2 );
	helix_ = boost::lexical_cast<Size>( parts[0] );

	std::vector< String > st( string_split( parts[1], ',' ) );
	runtime_assert( st.size() == 2 );
	strand1_ = boost::lexical_cast<Size>( st[0] );
	strand2_ = boost::lexical_cast<Size>( st[1] );
}


/// @copy constructor
HSSTriplet::HSSTriplet( HSSTriplet const & hss ):
	ReferenceCount(),
	helix_( hss.helix_ ),
	strand1_( hss.strand1_ ),
	strand2_( hss.strand2_ )
{}

/// @brief IO Operator
std::ostream & operator<<(std::ostream & out, const HSSTriplet & s )
{
	out << "Helix:" << s.helix() << " Strand1:" << s.strand1() << " Strand2:" << s.strand2();
	return out;
}

/// @brief whether the CB->CA vector of the C-term residue of 1st strand is pointing inward or outward
/// 1: inward, 2: outward
core::Size
HSSTriplet::calc_inout( SS_Info2_OP const & ssinfo ) const
{
	using core::Vector;
	using protocols::fldsgn::topology::BB_Pos;

	Real neighbor_dist( 14.0 );
	Size burial( 4 );

	BB_Pos bb_pos( ssinfo->bb_pos() );

	Size resi = ssinfo->strand( strand1_)->end();
	Size start = ssinfo->strand( strand1_ )->end() + 1;
	Size end = ssinfo->strand( strand2_ )->begin() - 1;

	Vector cbca = bb_pos.CB( resi ) - bb_pos.CA( resi );

	Size neighbor( 0 );
	for( Size ii=start; ii<=end; ii++ ) {

		Vector bb = bb_pos.CA( resi ) - bb_pos.CA( ii );

		Real orient = cbca.x()*( bb_pos.CA( ii ).x() - bb_pos.CA( resi ).x() ) +
			cbca.y()*( bb_pos.CA( ii ).y() - bb_pos.CA( resi ).y() ) +
			cbca.z()*( bb_pos.CA( ii ).z() - bb_pos.CA( resi ).z() );

		if( bb.length() <= neighbor_dist && orient > 0 ) {
			neighbor ++;
		}

		if( neighbor > burial ) {
			return 1;
		}
	}

	return 2;

}


/// @brief default constructor
HSSTripletSet::HSSTripletSet()
{
	clear();
}


/// @brief value constructor
HSSTripletSet::HSSTripletSet( String const & s )
{
	using utility::string_split;
	clear();

	if( s == "" ) {
		return;
	}

	std::vector< String > hsss( string_split( s, ';' ) );
	for( std::vector< String >::const_iterator iter = hsss.begin(); iter != hsss.end() ; ++iter) {
		push_back( new HSSTriplet( *iter ) );
	}
}


/// @brief value constructor
HSSTripletSet::HSSTripletSet( HSSTriplets const & s )
{
	for( HSSTriplets::const_iterator it=s.begin(),	ite=s.end(); it!= ite; ++it ) {
		HSSTripletOP const & hss( *it );
		push_back( hss );
	}
}


/// @brief copy constructor
HSSTripletSet::HSSTripletSet( HSSTripletSet const & s ) :
	ReferenceCount(),
	hss_triplets_( s.hss_triplets_ ),
	helix2hss_( s.helix2hss_ )
{}


/// @brief destructor
HSSTripletSet::~HSSTripletSet(){}


/// @brief IO Operator
std::ostream & operator<<(std::ostream & out, const HSSTripletSet & s )
{
	out << "#### HSSTriplet Info " << std::endl;
	for( HSSTriplets::const_iterator iter = s.hss_triplets().begin(),
				 iter_end = s.hss_triplets().end(); iter != iter_end; ++iter ) {
		HSSTriplet const & hss( **iter );
		out << hss << std::endl;
	}
	return out;
}


/// @brief puch back data
void
HSSTripletSet::push_back( HSSTripletOP const & hsop )
{
	for ( std::map< Size, HSSTripletOP >::const_iterator it=helix2hss_.begin(),
					ite=helix2hss_.end(); it!=ite ; ++it ) {
		if ( it->first == hsop->helix() ) {
			TR <<  "Helix "  <<  it->first << " is already defined in HSSTriplet. " << std::endl;
			assert( false );
		}
	}

	hss_triplets_.push_back( hsop );
	helix2hss_[ hsop->helix() ] = hsop;
}

/// @brief clear data
void
HSSTripletSet::clear()
{
	hss_triplets_.clear();
	helix2hss_.clear();
}


/// @brief
HSSTripletOP const &
HSSTripletSet::hss_triplet( Size const helix )
{
	return helix2hss_[ helix ];
}


/// @brief return all data
HSSTriplets const &
HSSTripletSet::hss_triplets() const
{
	return hss_triplets_;
}


} // namespace topology
} // namespace fldsgn
} // namespace protocols
