// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// Copyright in the Rosetta software belongs to the developers and their institutions.
// For more information, see www.rosettacommons.org.

/// @file ./src/protocols/fldsgn/topology/util.cc
/// @brief utilities for fldsgn/topology
/// @author Nobuyasu Koga ( nobuyasu@u.washington.edu )

// unit header
#include <protocols/fldsgn/topology/util.hh>
#include <protocols/fldsgn/topology/StrandPairing.hh>
#include <protocols/fldsgn/topology/SS_Info2.hh>

// Project Headers
#include <core/pose/Pose.hh>
#include <core/scoring/dssp/Dssp.hh>
#include <core/scoring/dssp/StrandPairing.hh>

// utility headers
#include <utility/string_util.hh>
#include <utility/exit.hh>

// C++ headers
#include <cassert>
#include <iostream>
#include <sstream>
#include <boost/lexical_cast.hpp>
#include <core/util/Tracer.hh>


static core::util::Tracer TR( "protocols.topology.util" );

using namespace core;
typedef std::string String;
typedef utility::vector1< Size > VecSize;

namespace protocols {
namespace fldsgn {
namespace topology {

/// @brief convert StrandParingSet of dssp to fldsgn::topology::StrandPairingSet
protocols::fldsgn::topology::StrandPairingSet
calc_strand_pairing_set(
  core::pose::Pose const & pose,
	protocols::fldsgn::topology::SS_Info2_OP const & ssinfo,
	core::Size minimum_pair_length
 )
{
	using core::Size;
	typedef protocols::fldsgn::topology::StrandPairing StrandParing;
	typedef protocols::fldsgn::topology::StrandPairingOP StrandParingOP;

	core::scoring::dssp::Dssp dssp( pose );

  std::map< String, StrandPairingOP > newpairs;

  core::scoring::dssp::StrandPairingSet spairset;
  spairset = dssp.strand_pairing_set();

  for( Size ispair=1; ispair<=spairset.size(); ispair++ ) {

		core::scoring::dssp::StrandPairing sp;
    sp = spairset.strand_pairing( ispair );
    Size begin1 ( sp.begin1() );
    Size end1 ( sp.end1() );

    for( Size iaa=begin1; iaa<=end1; ++iaa ) {

      Size istrand ( ssinfo->strand_id( iaa ) );
			if( istrand == 0 ) continue;
      Size ist_begin ( ssinfo->strand( istrand )->begin() );

      Size jaa ( sp.get_pair( iaa ) );
			if ( jaa == 0 ) continue;
      Size pleats ( sp.get_pleating( iaa ) );
      Size jstrand ( ssinfo->strand_id( jaa ) );
			if( jstrand == 0 ) continue;

      Size jst_begin ( ssinfo->strand( jstrand )->begin() );
      Size jst_end ( ssinfo->strand( jstrand )->end() );
      Size jst_length ( jst_end - jst_begin );

			if( istrand == jstrand ) continue;

      if( istrand !=0 && jstrand !=0 && jaa !=0  ){

				char orient;
				Real rgstr_shift;

				if( sp.antiparallel() ){
					orient = 'A';
					rgstr_shift = Real(iaa) - Real(ist_begin) - (Real(jst_length) - (Real(jaa) - Real(jst_begin)));
				}else{
					orient = 'P';
					rgstr_shift = Real(iaa) - Real(ist_begin) - (Real(jaa) - Real(jst_begin));
				}

				std::ostringstream spairname;
				spairname << istrand << "-" << jstrand << "." << orient ;
				std::map<String, StrandPairingOP>::iterator it( newpairs.find( spairname.str() ) );
				if ( it == newpairs.end() ){
					StrandPairingOP strand_pair = new StrandPairing( istrand, jstrand, iaa, jaa, pleats, rgstr_shift, orient );
					newpairs.insert( std::map<String, StrandPairingOP>::value_type( spairname.str(), strand_pair ) );
				}else{
					(*it).second->elongate( iaa, jaa, pleats, pleats );
				}

      } // istrand !=0 && jstrand !=0 && jaa !=0

    } // iaa
  } // ispair

	StrandPairingSet spairset_new;
  std::map<String, StrandPairingOP>::iterator it( newpairs.begin() );
  while( it != newpairs.end() ){

		// skip if pair length < minimum_pair_length
		if( (*it).second->size1() < minimum_pair_length || (*it).second->size2() < minimum_pair_length ) {
			++it;
			continue;
		}

		//
		TR.Debug << "Defined strand pair : " << *((*it).second)
						 << ", 1st_strand begin: " << (*it).second->begin1() << ", end: " <<  (*it).second->end1()
						 << ", 2nd_strand begin: " << (*it).second->begin2() << ", end: " <<  (*it).second->end2() << std::endl;
    spairset_new.push_back( (*it).second );
    ++it;
  }

	spairset_new.finalize();

  return spairset_new;

} // clac_strand_pairings


} // namespace topology
} // namespace fldsgn
} // namespace protocols
