// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com)
///

// Project Headers
#include <core/pose/Pose.hh>
#include <core/conformation/Conformation.hh>
#include <core/types.hh>
#include <core/util/Tracer.hh>

// Unit Headers
#include <protocols/ligand_docking/InterfaceBuilder.hh>
#include <protocols/ligand_docking/ligand_options/Interface.hh>
#include <protocols/ligand_docking/ligand_options/interface_distance_functions.hh>
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>

// Utility headers
#include <utility/Tag/Tag.hh>
#include <utility/string_util.hh>

namespace protocols {
namespace ligand_docking {

static core::util::Tracer interface_builder_tracer("protocols.ligand_docking.ligand_options.InterfaceBuilder", core::util::t_debug);

InterfaceBuilder::InterfaceBuilder():
		ReferenceCount(),
		cutoff_(0.0),
		add_nbr_radius_(false),
		all_atom_mode_(false),
		extension_window_(0)
{}

InterfaceBuilder::InterfaceBuilder(InterfaceBuilder const & that):
		ReferenceCount(),
		cutoff_(that.cutoff_),
		add_nbr_radius_(that.add_nbr_radius_),
		all_atom_mode_(that.all_atom_mode_),
		extension_window_(that.extension_window_)
{}

InterfaceBuilder::~InterfaceBuilder() {}

//@brief parse XML (specifically in the context of the parser/scripting scheme)
void
InterfaceBuilder::parse_my_tag(
		utility::Tag::TagPtr const tag
){
	if ( ! tag->hasOption("chains") ) utility_exit_with_message("'InterfaceBuilder'  requires 'chains' tag");
	if ( ! tag->hasOption("cutoff") ) utility_exit_with_message("'InterfaceBuilder' requires 'cutoff' tag");
	if ( ! tag->hasOption("add_nbr_radius") ) utility_exit_with_message("'InterfaceBuilder' requires 'add_nbr_radius' tag");
	if ( ! tag->hasOption("all_atom_mode") ) utility_exit_with_message("'InterfaceBuilder' requires 'all_atom_mode' tag");

	std::string chains_string= tag->getOption<std::string>("chains");
	chains_= utility::string_split(chains_string, ',');

	cutoff_= tag->getOption<core::Real>("cutoff");

	if(tag->getOption<std::string>("add_nbr_radius") == "true")
		add_nbr_radius_= true;
	else if(tag->getOption<std::string>("add_nbr_radius") != "false")
		utility_exit_with_message("'add_nbr_radius' option is true or false");

	if(tag->getOption<std::string>("all_atom_mode") == "true")
		all_atom_mode_= true;
	else if(tag->getOption<std::string>("all_atom_mode") != "false")
		utility_exit_with_message("'all_atom_mode' option is true or false");

	if ( tag->hasOption("extension_window") ){
		extension_window_= tag->getOption<core::Size>("extension_window");
	}
}

ligand_options::Interface InterfaceBuilder::build(core::pose::Pose const & pose) const{
	ligand_options::Interface interface( pose.total_residue(), ligand_options::InterfaceInfo() ); // init all positions to false
	find_interface_residues(interface, pose);
	if (extension_window_ > 0) enforce_minimum_length(interface, pose);
	interface_builder_tracer.Debug<< "built interface: "<< interface<< std::endl;
	return interface;
}

void InterfaceBuilder::find_interface_residues(
		ligand_options::Interface & interface,
		core::pose::Pose const & pose
)const{
	find_ligand_residues(interface, pose);
	// set up protein residues to minimize based on distance to ligand residues we are minimizing
	utility::vector1<core::Size> ligand_residues= interface.get_interface_residues();
	find_protein_residues(interface, ligand_residues, pose);
}

void InterfaceBuilder::find_ligand_residues(
		ligand_options::Interface & interface,
		core::pose::Pose const & pose
)const{
	utility::vector1<std::string>::const_iterator chain= chains_.begin();
	for(; chain != chains_.end(); ++chain){
		core::Size chain_id= ligand_options::get_chain_id_from_chain(*chain, pose);
		core::Size residue_id= pose.conformation().chain_begin(chain_id);
		core::Size end= pose.conformation().chain_end(chain_id);
		for(; residue_id <= end; ++residue_id){
			interface[residue_id].type=  ligand_options::InterfaceInfo::is_interface;
			interface[residue_id].chain= chain_id;
		}
	}
	interface_builder_tracer.Debug << "ligands found: " << interface << std::endl;
}

// First call find_ligand_residues
void InterfaceBuilder::find_protein_residues(
		ligand_options::Interface & interface,
		utility::vector1<core::Size> const ligand_residues,
		core::pose::Pose const & pose
)const{
	for(core::Size i = 1, i_end = pose.total_residue(); i <= i_end; ++i) {
		if(pose.residue(i).is_protein() ){
			utility::vector1<core::Size>::const_iterator ligand_residue_iterator= ligand_residues.begin();
			for(; ligand_residue_iterator != ligand_residues.end(); ++ligand_residue_iterator){
				set_interface_residue(interface, i, *ligand_residue_iterator, pose);
			}
		}
	}
}

void InterfaceBuilder::set_interface_residue(
		ligand_options::Interface & interface,
		core::Size const potential_interface_residue_id,
		core::Size const ligand_interface_residue_id,
		core::pose::Pose const & pose
)const{
	core::conformation::Residue const & potential_interface_residue = pose.residue(potential_interface_residue_id);
	core::conformation::Residue const & ligand_residue = pose.residue(ligand_interface_residue_id);

	if( is_interface_residue(potential_interface_residue, ligand_residue)){
		core::Size const ligand_chain= interface[ligand_interface_residue_id].chain;
		interface[potential_interface_residue_id].type= ligand_options::InterfaceInfo::is_interface;
		interface[potential_interface_residue_id].chain= ligand_chain;
	}
}

bool InterfaceBuilder::is_interface_residue(
		core::conformation::Residue const & potential_interface_residue,
		core::conformation::Residue const & ligand_interface_residue
)const {
	core::Size const potential_interface_neighbor_atom_id= potential_interface_residue.nbr_atom();
	core::Vector const potential_interface_vector= potential_interface_residue.xyz(potential_interface_neighbor_atom_id);

	double cutoff = add_nbr_radius_ ?
			potential_interface_residue.nbr_radius() + cutoff_ :
			cutoff_;

	if (all_atom_mode_){
		return ligand_options::check_all_ligand_atoms(ligand_interface_residue, potential_interface_vector, cutoff);
	}
	else{
		return ligand_options::check_neighbor_ligand_atom(ligand_interface_residue, potential_interface_vector, cutoff);
	}
}

void InterfaceBuilder::enforce_minimum_length(
		ligand_options::Interface & interface,
		core::pose::Pose const & pose
) const{
	for(core::Size chain_id=1; chain_id <= pose.conformation().num_chains(); ++chain_id){
		core::Size start= pose.conformation().chain_begin(chain_id);
		core::Size end= pose.conformation().chain_end(chain_id);
		if(pose.residue(start).is_polymer()){ // because ligand residues don't know about terminus
			interface.enforce_minimum_length(start, end, extension_window_);
		}
	}
}

} //namespace ligand_docking
} //namespace protocols
