// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/ligand_docking/LigandDockingLoaders.cc
/// @brief  Implementation of the InterfaceBuilderLoader and MoveMapBuilderLoader classes
/// @author Gordon Lemmon
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com) -- moved here from DockDesignParser.cc

// Unit Headers
#include <protocols/ligand_docking/LigandDockingLoaders.hh>
#include <protocols/ligand_docking/LigandDockingLoaderCreators.hh>

// Project Headers
#include <protocols/ligand_docking/InterfaceBuilder.hh>
#include <protocols/ligand_docking/MoveMapBuilder.hh>

#include <core/util/Tracer.hh>

// Utility headers
#include <utility/Tag/Tag.hh>


namespace protocols {
namespace ligand_docking {

static core::util::Tracer TR( "protocols.ligand_docking.LigandDockingLoaders" );

InterfaceBuilderLoader::InterfaceBuilderLoader() {}
InterfaceBuilderLoader::~InterfaceBuilderLoader() {}

void InterfaceBuilderLoader::load_data(
	core::pose::Pose const &,
	utility::Tag::TagPtr const tag,
	moves::DataMap & data
) const
{
	using namespace utility::Tag;
	typedef utility::vector0< TagPtr > TagPtrs;

	TagPtrs const interface_builders_tags( tag->getTags() );

	for( TagPtrs::const_iterator
			tp( interface_builders_tags.begin() ), tp_e( interface_builders_tags.end() );
			tp != tp_e; ++tp ) {
		TagPtr const interface_builder_tag = *tp;
		std::string const name( interface_builder_tag->getName() );

		if ( data.has("interface_builders", name)) {
			TR << "WARNING WARNING movemap_builder of name \"" << name
				<< ") already exists. Skipping\n" << interface_builder_tag << std::endl;
			continue;
		}
		///// Add this movemap to the data map
		protocols::ligand_docking::InterfaceBuilderOP interface_builder= new protocols::ligand_docking::InterfaceBuilder();
		interface_builder->parse_my_tag( interface_builder_tag );
		data.add( "interface_builders" , name, interface_builder);
	}
	TR.flush();
}

jd2::parser::DataLoaderOP
InterfaceBuilderLoaderCreator::create_loader() const { return new InterfaceBuilderLoader; }

std::string
InterfaceBuilderLoaderCreator::keyname() const { return "INTERFACE_BUILDERS"; }

MoveMapBuilderLoader::MoveMapBuilderLoader() {}
MoveMapBuilderLoader::~MoveMapBuilderLoader() {}

void MoveMapBuilderLoader::load_data(
	core::pose::Pose const &,
	utility::Tag::TagPtr const tag,
	moves::DataMap & data
) const
{
	using namespace utility::Tag;
	typedef utility::vector0< TagPtr > TagPtrs;

	TagPtrs const movemap_builders_tags( tag->getTags() );
	for( TagPtrs::const_iterator
			tp( movemap_builders_tags.begin() ), tp_e( movemap_builders_tags.end() );
			tp != tp_e; ++tp ) {
		TagPtr const movemap_builder_tag = *tp;
		std::string const name( movemap_builder_tag->getName() );

		if ( data.has("movemap_builders", name)) {
			TR << "WARNING WARNING movemap_builder of name \"" << name
				<< ") already exists. Skipping\n" << movemap_builder_tag << std::endl;
			continue;
		}
		///// Add this movemap to the data map
		protocols::ligand_docking::MoveMapBuilderOP movemap_builder= new protocols::ligand_docking::MoveMapBuilder();
		movemap_builder->parse_my_tag( movemap_builder_tag, data );
		data.add( "movemap_builders" , name, movemap_builder);
	}
	TR.flush();
}

jd2::parser::DataLoaderOP
MoveMapBuilderLoaderCreator::create_loader() const { return new MoveMapBuilderLoader; }

std::string
MoveMapBuilderLoaderCreator::keyname() const { return "MOVEMAP_BUILDERS"; }


} //namespace jd2
} //namespace protocols
