// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/pack/task/ResfileReader.cc
/// @brief  implementation of resfile reader and its command classes
/// @author Gordon Lemmon (glemmon@gmail.com), adapted from the ResfileReader code
/// by Steven Lewis (smlewi@unc.edu) and Andrew Leaver-Fay

// Project Headers
#include <core/pose/Pose.hh>
#include <core/conformation/Conformation.hh>

// Unit Headers
#include <protocols/ligand_docking/MinimizeLigand.hh>
#include <protocols/ligand_docking/MinimizeLigandCreator.hh>
#include <protocols/ligand_docking/ResidueTorsionRestraints.hh>
#include <protocols/ligand_docking/ligand_options/chain_functions.hh>

// Utility Headers
#include <utility/exit.hh>
#include <core/util/Tracer.hh>
#include <core/types.hh>

#include <utility/Tag/Tag.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

//STL headers


namespace protocols {
namespace ligand_docking {

static core::util::Tracer MinimizeLigand_tracer("protocols.ligand_docking.MinimizeLigand", core::util::t_debug);

std::string
MinimizeLigandCreator::keyname() const
{
	return MinimizeLigandCreator::mover_name();
}

protocols::moves::MoverOP
MinimizeLigandCreator::create_mover() const {
	return new MinimizeLigand;
}

std::string
MinimizeLigandCreator::mover_name()
{
	return "MinimizeLigand";
}

MinimizeLigand::MinimizeLigand(){
	ligand_torsion_restraints_.clear();
}

MinimizeLigand::MinimizeLigand(MinimizeLigand const & that):
		protocols::moves::Mover( that ),
		chain_(that.chain_),
		degrees_(that.degrees_)
{}

MinimizeLigand::~MinimizeLigand() {}

protocols::moves::MoverOP MinimizeLigand::clone() const {
	return new MinimizeLigand( *this );
}

protocols::moves::MoverOP MinimizeLigand::fresh_instance() const {
	return new MinimizeLigand;
}

std::string MinimizeLigand::get_name() const{
	return "MinimizeLigand";
}

//@brief parse XML (specifically in the context of the parser/scripting scheme)
void
MinimizeLigand::parse_my_tag(
		utility::Tag::TagPtr const tag,
		protocols::moves::DataMap & /*datamap*/,
		protocols::filters::Filters_map const & /*filters*/,
		protocols::moves::Movers_map const & /*movers*/,
		core::pose::Pose const & /*pose*/
)
{
	if ( tag->getName() != "MinimizeLigand" ) utility_exit_with_message("This should be impossible");
	if ( ! tag->hasOption("chain") ) utility_exit_with_message("'MinimizeLigand' mover requires chain tag");
	if ( ! tag->hasOption("degrees") ) utility_exit_with_message("'MinimizeLigand' mover requires chain tag");

	chain_= tag->getOption<std::string>("chain");
	degrees_= tag->getOption<core::Real>("degrees");

	//parse_score_function( tag, datamap, filters, movers, pose );
	//parse_task_operations( tag, datamap, filters, movers, pose );
}

void
MinimizeLigand::apply( core::pose::Pose & pose ){
	core::Size chain_id= ligand_options::get_chain_id_from_chain(chain_, pose);
	core::Size begin = pose.conformation().chain_begin(chain_id);
	core::Size const end = pose.conformation().chain_end(chain_id);
	for (; begin <= end; ++begin) {
		ligand_torsion_restraints_.push_back(
			new protocols::ligand_docking::ResidueTorsionRestraints(pose, begin, degrees_));
	}
}

bool MinimizeLigand::operator==(std::string chain){
	return chain == chain_;
}

utility::vector1<protocols::ligand_docking::ResidueTorsionRestraintsOP>::iterator
MinimizeLigand::begin(){
	return ligand_torsion_restraints_.begin();
}
utility::vector1<protocols::ligand_docking::ResidueTorsionRestraintsOP>::iterator
MinimizeLigand::end(){
	return ligand_torsion_restraints_.end();
}


} //namespace ligand_docking
} //namespace protocols
