// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loophash/BackboneDB.hh
/// @brief I little in the spirit of fragments but way more memory and speed efficient. Backbones are stored as 16bit fixed comma integers. :)
/// @author Mike Tyka


#ifndef INCLUDED_protocols_loops_BackboneDB_HH
#define INCLUDED_protocols_loops_BackboneDB_HH

#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>
#include <boost/cstdint.hpp>
#include <boost/unordered_map.hpp>
#include <protocols/match/Hit.fwd.hh>
#include <protocols/match/Hit.hh>
#include <protocols/match/SixDHasher.hh>

#include <string>
#include <vector>

namespace protocols {
namespace loophash {

// surely there's a func like this somewhere ?
template <class T>
T sqr( T a){ return a*a; }

const core::Real MAXIMAL_FLOAT = 100000000.0;

typedef unsigned int BackboneIndex;
typedef boost::unordered_multimap< boost::uint64_t, core::Size , protocols::match::bin_index_hasher >   BackboneIndexMap;
typedef numeric::geometry::BoundingBox< core::Vector > BoundingBox;
const int HASH_POSITION_GRID_SIZE= 25;

short RealAngleToShort( core::Real angle );

core::Real ShortToRealAngle( short angle );

class BackboneSegment {
	public:
    BackboneSegment(){}

    BackboneSegment(
            const std::vector<core::Real> &phi,
            const std::vector<core::Real> &psi,
            const std::vector<core::Real> &omega
    ){
      phi_ = phi;
      psi_ = psi;
      omega_ = omega;
    }

    void apply_to_pose( core::pose::Pose &pose, core::Size ir, bool cut  = false ) const;
    void read_from_pose( core::pose::Pose const &pose, core::Size ir, core::Size length );

    void print() const ;

    core::Size length() const { return  phi_.size(); }
    const std::vector<core::Real> &phi()   const { return phi_; }
    const std::vector<core::Real> &psi()   const { return psi_; }
    const std::vector<core::Real> &omega() const { return omega_; }

 		core::Size get_mem_foot_print(){ return phi_.size() * 3 * sizeof( core::Real ) + sizeof( BackboneSegment ); }
	private:
    std::vector<core::Real> phi_;
    std::vector<core::Real> psi_;
    std::vector<core::Real> omega_;

};

core::Real get_rmsd( const BackboneSegment &bs1, const BackboneSegment &bs2 );





class BackboneDB {
	public:
		BackboneDB(){}

		core::Real angle( core::Size pos );

		void add_pose( const core::pose::Pose &pose, core::Size &offset );

		void get_backbone_segment(
			const core::Size start,
			const core::Size len,
			BackboneSegment &bs
		) const;

		core::Size size() const { return data_.size(); }

		void write_db_to_binary(std::string filename );

		void read_db_from_binary(std::string filename );

 		core::Size get_mem_foot_print(){ return data_.size() * sizeof( short ) + sizeof(BackboneDB); };
	private:
		std::vector < short > data_;

};

//
//// @brief A highly efficient storage for centroid poses. This is great if you have many many poses
////        with identical foldtrees and sequences and you only care about the backbone conformation
////        It's about 10x more efficient then a silent file and many many more times more efficient then a Pose.
//class BackboneStack {
// public:
//  BackboneStack( core::Size nres ): nres_(nres)
//  {
//
//  }
//
//  // @brief add a pose
//	void add( const core::pose::Pose &pose );
//
//  // @brief add an entire stack
//  void add( const BackboneStack &newbbs );
//
//  // @brief Obtain a new pose from a given index. must provide a template pose though!
//  void get_pose( const core::Size index, core::pose::Pose &template_pose );
//
//  void size(){ return index_.size(); }
//
//  void nres(){ return nres_; }
//
// private:
//  BackboneDB   db_;
//  const core::Size nres_;
//  std::vector < core::Size > index_;
//
//}

}
}

#endif

