// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loophash/LoopHashLibrary.hh
/// @brief
/// @author Mike Tyka



#ifndef INCLUDED_protocols_loops_LoopHashLibrary_HH
#define INCLUDED_protocols_loops_LoopHashLibrary_HH

#include <protocols/loophash/LoopHashLibrary.fwd.hh>
#include <protocols/loophash/LoopHashSampler.fwd.hh>
#include <protocols/loophash/LocalInserter.fwd.hh>

#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/loops/Loop.hh>
#include <protocols/loophash/LoopHashMap.hh>
#include <utility/vector1.hh>
#include <core/io/silent/SilentStruct.fwd.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <string>
#include <vector>

namespace protocols {
namespace loophash {


class LoopHashLibrary: public protocols::moves::Mover {

public:

	LoopHashLibrary( const utility::vector1< core::Size > &init_sizes = utility::vector1< core::Size >(), const core::Size num_partitions = 1, const core::Size assigned_num = 1);

	void extract_data_from_pose( core::pose::Pose& pose, core::Size nres );

	void extract_data_from_pose( core::pose::Pose& pose );

	void test_loop_sample( core::pose::Pose& pose, core::Size nres );

	void apply_random(
		core::pose::Pose& pose,
		core::Size &fir,
		core::Size &fjr,
		core::Real min_rms,
		core::Real max_rms
	);

	void get_all(
		core::pose::Pose& pose,
    std::vector< core::io::silent::SilentStructOP > &lib_structs,

    core::Size start_res = 1,
    core::Size stop_res  = 0,

    core::Real min_bbrms = 0.0,
		core::Real max_bbrms = 100000.0,
		core::Real min_rms   = 0.0,
		core::Real max_rms   = 100.0
	);

	virtual void apply( core::pose::Pose& pose );

  virtual protocols::moves::MoverOP clone() const {
		return new LoopHashLibrary( *this );
	}


	virtual std::string get_name() const {
		return "LoopHashLibrary";
	}

	virtual	protocols::moves::MoverOP	fresh_instance() const {
		return new LoopHashLibrary;
	}


	void setup_hash_maps();

  // simple accessors
	LoopHashMap & gethash( core::Size size );


	const std::vector < core::Size > & hash_sizes() const { return hash_sizes_; }

  const BackboneDB & backbone_database() const { return bbdb_; }

	void save_db();
	void load_db();

	void create_db();

	void set_create_db( bool setting = true ){ create_db_ = setting; };

  void graft_loop(
    const core::pose::Pose& src_pose,
    core::pose::Pose& tgt_pose,
    protocols::loops::Loop myloop
  );

  // setup scorefunctions for
  void set_default_score_functions();


	void mem_foot_print();

private:

  // The backbone library for this HashLibrary (the actual data)
	BackboneDB bbdb_;

  // A map of Hashmaps, each for a different loop size
	std::map    < core::Size, LoopHashMap > hash_;

  // Kindof a redundant store of sizes - remove this ?
	std::vector < core::Size > hash_sizes_;

  // Path to db
    std::string db_path_;

  // the number of partitions the database is split into
    core::Size num_partitions_;

  // which partition slice is assigned to this Library
    core::Size assigned_num_;

  // Need so we can set the db string once
    std::string assigned_string_;

  // Some basic flags
	bool do_sanity_check_;
	bool create_db_;

  // Used for grafting - ultimately to move into a seperate Mover class.

	core::scoring::ScoreFunction scorefxn_rama_cst;
	core::scoring::ScoreFunction scorefxn_cen_cst;

	core::optimization::MinimizerOptions options;
	core::optimization::MinimizerOptions options2;
};






//class LoopHashSampler {
//  public:
//
//  LoopHashSampler(
//    LoopHashLibraryOP library,
//    LocalInserterOP inserter
//  ):
//    library_(library),
//    inserter_(inserter),
//    start_res_ ( 1 ),
//    stop_res_  ( 0 ),
//    min_bbrms_ ( 0.0 ),
//		max_bbrms_ ( 100000.0 ),
//		min_rms_   ( 0.0 ),
//		max_rms_   ( 100.0 )
//  {
//
//  }
//
//  // @brief create a set of structures for a the given range of residues and other parameters
//  void build_structures(
//		core::pose::Pose& start_pose,
//    std::vector< core::io::silent::SilentStructOP > &lib_structs
//	);
//
//
//
//  private:
//    // pointer to the library used for insertion
//    LoopHashLibraryOP library_;
//
//    // pointer to the insertion functor which provides the peptide insertion facility
//    LocalInserterOP inserter_;
//
//    // parameters for insertion positions
//    core::Size start_res_;
//    core::Size stop_res_ ;
//    core::Real min_bbrms_;
//		core::Real max_bbrms_;
//		core::Real min_rms_  ;
//		core::Real max_rms_  ;
//
//};





} // namespace loops
} // namespace protocols



#endif



