// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loops/LoopHashMap.cc
/// @brief
/// @author Mike Tyka



// libRosetta headers
#include <protocols/loophash/LoopHashMap.hh>

#include <protocols/loophash/BackboneDB.hh>

#include <boost/cstdint.hpp>
#include <boost/unordered_map.hpp>
#include <core/fragment/picking/VallChunk.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/fragment/picking/VallProvider.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/pose/util.hh>
#include <core/kinematics/Jump.hh>
#include <core/kinematics/RT.hh>
#include <core/io/pose_stream/util.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/ResidueFactory.hh>
#include <core/scoring/constraints/util.hh>
#include <core/util/Tracer.hh>

#include <numeric/HomogeneousTransform.hh>
#include <protocols/match/Hit.fwd.hh>
#include <protocols/match/Hit.hh>
#include <protocols/match/SixDHasher.hh>
#include <protocols/moves/Mover.hh>
#include <utility/excn/Exceptions.hh>
#include <utility/exit.hh>
#include <utility/fixedsizearray1.hh>
#include <utility/pointer/owning_ptr.hh>
#include <numeric/angle.functions.hh>

// C++ headers
#include <cstdio>

#include <iostream>
#include <string>


//Auto Headers
#include <core/util/datacache/CacheableData.hh>
#include <core/pose/Pose.hh>


#if defined(WIN32) || defined(__CYGWIN__)
	#include <ctime>
#endif



using namespace protocols::moves;
using namespace core::scoring;
using namespace core;
using namespace core::pose;
using namespace conformation;
using namespace kinematics;
using namespace protocols::match;
using namespace core::fragment::picking;





namespace protocols {
namespace loophash {




static core::util::Tracer TR("LoopHashMap");






/// @brief This takes a pose and two residue positions and determines the rigid body transform of the Leap described by those two residues.
///        Returns true is successful or false if something went haywire and one should just ignore this loop (this can happen at the ends)
bool get_rt_over_leap( const core::pose::Pose& orig_pose, core::Size ir, core::Size jr, protocols::match::Real6 &rt_6 ){
	using namespace core;
	using namespace core::pose;
	using namespace conformation;
	using namespace kinematics;
	using namespace protocols::match;

  core::pose::Pose pose = orig_pose;

  // Create a fake foldtree with a jump from ir to jr, and the cutpoint just before jr. From that extract the
  // rigid body transfer. This is fairly hacky, but actually works v reliably and is easy to understand, without
  // having to mess deeply with the fold tree.
	FoldTree f;
	Size cutpoint= jr-1;
	f.add_edge( 1, ir, Edge::PEPTIDE );
	f.add_edge( ir, cutpoint, Edge::PEPTIDE );
	f.add_edge( cutpoint + 1, jr, Edge::PEPTIDE );
	f.add_edge( jr, pose.total_residue() , Edge::PEPTIDE );
	f.add_edge( ir, jr, 1 );  // this is the jump !!
	core::Size theroot = 1;
	if( ir == 1 ) theroot = pose.total_residue();
	if( f.reorder(theroot) == false ){
		TR.Error << "ERROR During reordering of fold tree - am ignoring this LOOP ! bailing: The root: " << theroot << " NRES " << pose.total_residue() << "   IR: " << ir << "  JR: " << jr << std::endl;
		return false; // continuing leads to a segfault - instead ignore this loop !
	}

  // Apply this new foldtree to the pose.
	pose.fold_tree( f );

	// Now extract the rigid body transform!
	Jump myjump;
	myjump = pose.jump( 1 );
	
  // Aha, now you have the RT (RigidbodyTransform)
  RT rt = myjump.rt();

	// Create a 6 value representation: (just change the data format)
	numeric::HomogeneousTransform< Real > ht( rt.get_rotation() , rt.get_translation() );
	numeric::xyzVector < Real > euler_angles =  ht.euler_angles_rad();

	rt_6[1] = rt.get_translation().x();
	rt_6[2] = rt.get_translation().y();
	rt_6[3] = rt.get_translation().z();
	rt_6[4] = euler_angles.x()*180/3.1415;
	rt_6[5] = euler_angles.y()*180/3.1415;
	rt_6[6] = euler_angles.z()*180/3.1415;

  // indicate success
	return true;
}



/// @brief This takes a pose and two residue positions and determines the rigid body transform of the Leap described by those two residues
///       THe difference between this and the get_rt_over_leap function is that this version doesnt make a copy of the pose which makes it faster.
///     However this means that the pose passed cannot be a const pose, even though the function restores the fold tree afterwards..
bool get_rt_over_leap_fast( core::pose::Pose& pose, core::Size ir, core::Size jr, protocols::match::Real6 &rt_6 ){
	using namespace core;
	using namespace core::pose;
	using namespace conformation;
	using namespace kinematics;
	using namespace protocols::match;


	FoldTree f;
	Size cutpoint= jr-1;
	f.add_edge( 1, ir, Edge::PEPTIDE );
	f.add_edge( ir, cutpoint, Edge::PEPTIDE );
	f.add_edge( cutpoint + 1, jr, Edge::PEPTIDE );
	f.add_edge( jr, pose.total_residue() , Edge::PEPTIDE );
	f.add_edge( ir, jr, 1 );  // this is the jump !!
	core::Size theroot = 1;
	if( ir == 1 ) theroot = pose.total_residue();
	if( f.reorder(theroot) == false ){
		TR.Error << "ERROR During reordering of fold tree - am ignoring this LOOP ! bailing: The root: " << theroot << " NRES " << pose.total_residue() << "   IR: " << ir << "  JR: " << jr << std::endl;
		return false; // continuing leads to a segfault - instead ignore this loop !
	}

	//TR.Info << f << std::endl;
	pose.fold_tree( f );

	// Now extract the rigid body transform!
	Jump myjump;
	myjump = pose.jump( 1 );
	RT rt = myjump.rt();

	// TR.Info << "R: " << ir << " " << jr << rt << std::endl;

	// Create a 6 value representation:

	numeric::HomogeneousTransform< Real > ht( rt.get_rotation() , rt.get_translation() );
	numeric::xyzVector < Real > euler_angles =  ht.euler_angles_rad();


	rt_6[1] = rt.get_translation().x();
	rt_6[2] = rt.get_translation().y();
	rt_6[3] = rt.get_translation().z();
	rt_6[4] = euler_angles.x()*180/3.1415;
	rt_6[5] = euler_angles.y()*180/3.1415;
	rt_6[6] = euler_angles.z()*180/3.1415;

	return true;
}



LoopHashMap::LoopHashMap( core::Size loop_size){
  loop_size_ = loop_size;
  setup(loop_size);
}

void LoopHashMap::mem_foot_print(){
	TR << "loopdb_: " << loopdb_.size() << "Size: " << loopdb_.size() * sizeof( LeapIndex ) << std::endl;
	TR << "BackboneIndexMap: " << backbone_index_map_.size() << " Size: " << backbone_index_map_.size() * (sizeof(boost::uint64_t) + sizeof(core::Size) ) << std::endl;
}

void LoopHashMap::setup( core::Size loop_size){
  loop_size_ = loop_size;

  TR.Info << "Setting up hash_: Size:  " << loop_size << std::endl;

  BoundingBox bounding_box( core::Vector( -HASH_POSITION_GRID_SIZE,
                                          -HASH_POSITION_GRID_SIZE,
                                          -HASH_POSITION_GRID_SIZE),
                            core::Vector( HASH_POSITION_GRID_SIZE,
                                          HASH_POSITION_GRID_SIZE,
                                          HASH_POSITION_GRID_SIZE ) );
  protocols::match::Size3 euler_offsets;
  euler_offsets[1] = 0;
  euler_offsets[2] = 0;
  euler_offsets[3] = 0;
  protocols::match::Real6 bin_widths;
  bin_widths[1] = 6.0*loop_size/6.0;
  bin_widths[2] = 6.0*loop_size/6.0;
  bin_widths[3] = 6.0*loop_size/6.0;
  bin_widths[4] = 15.0*loop_size/6.0;
  bin_widths[5] = 15.0*loop_size/6.0;
  bin_widths[6] = 15.0*loop_size/6.0;

  hash_ = new protocols::match::SixDCoordinateBinner( bounding_box, euler_offsets, bin_widths );
}


void LoopHashMap::add_leap( const LeapIndex &leap_index ){
  core::Size cpindex = loopdb_.size();
  loopdb_.push_back( leap_index );
  Real6 rt_6;
  rt_6[1] = leap_index.vecx;
  rt_6[2] = leap_index.vecy;
  rt_6[3] = leap_index.vecz;
  rt_6[4] = leap_index.rotx;
  rt_6[5] = leap_index.roty;
  rt_6[6] = leap_index.rotz;
	while(  rt_6[ 4 ] < 0.0 )    rt_6[ 4 ] += 360.0;
	while(  rt_6[ 4 ] >= 360.0 ) rt_6[ 4 ] -= 360.0;
	while(  rt_6[ 5 ] < 0.0 )    rt_6[ 5 ] += 360.0;
	while(  rt_6[ 5 ] >= 360.0 ) rt_6[ 5 ] -= 360.0;
	while(  rt_6[ 6 ] < 0.0 )  rt_6[ 6 ] += 180.0;
	while(  rt_6[ 6 ] > 180.0 ) rt_6[ 6 ] -= 180.0;
	if( !( rt_6[ 4 ] >= 0.0 && rt_6[ 4 ] < 360.0 ))  { std::cerr << "rt_6[4] out of bounds: " << rt_6[4] << std::endl;  utility_exit_with_message( "RANGE ERROR" ); }
	if( !( rt_6[ 5 ] >= 0.0 && rt_6[ 5 ] < 360.0 ))  { std::cerr << "rt_6[5] out of bounds: " << rt_6[5] << std::endl;  utility_exit_with_message( "RANGE ERROR" ); }
	if( !( rt_6[ 6 ] >= 0.0 && rt_6[ 6 ] <= 180.0 )) { std::cerr << "rt_6[6] out of bounds: " << rt_6[6] << std::endl;  utility_exit_with_message( "RANGE ERROR" ); }
  hash_index( rt_6, cpindex  );
}

void LoopHashMap::hash_index( protocols::match::Real6 transform, core::Size data ){
  // Hash the transform
  if( ! hash_->contains( transform ) ){
    //TR.Error << "OutofBOIUNDS! " << std::endl;
    return;
  }

  boost::uint64_t bin_index = hash_->bin_index( transform );
  // Add Address to map with hashed index
  backbone_index_map_.insert( std::make_pair( bin_index, data ) );
}


// append to a bucket of vectors in the appropriate bin
void LoopHashMap::lookup(  protocols::match::Real6 transform, std::vector < core::Size > &result ){
  // Hash the transform
  if( ! hash_->contains( transform ) ){
    //TR.Error << "OutofBOIUNDS! " << std::endl;
    return;
  }

  transform[4] = numeric::nonnegative_principal_angle_degrees(transform[4] );
  transform[5] = numeric::nonnegative_principal_angle_degrees(transform[5] );
  boost::uint64_t bin_index = hash_->bin_index( transform );

  // now get an iterator over that map entry

  std::pair<  BackboneIndexMap::iterator,
              BackboneIndexMap::iterator> range = backbone_index_map_.equal_range( bin_index );


  for( BackboneIndexMap::iterator it = range.first;
       it != range.second;
        ++it)
  {
      result.push_back( it->second );
  }
}


void LoopHashMap::write_db_to_binary(std::string filename ){
  // use basic C output  - C++ are too memory hungry to deal with these potentially v large files

  FILE *file = fopen( filename.c_str(), "w" );
  if( file == NULL ){
    utility_exit_with_message( "Cannot write loop DB file" + filename + "\n" );
  }

  for( core::Size i = 0; i < loopdb_.size(); i ++ ){
    fwrite(&loopdb_[i],sizeof(LeapIndex),1,file);
  }

  fclose( file );
}


void LoopHashMap::read_db_from_binary(std::string filename ){
  // use basic C input - C++ are too memory hungry to deal with these potentially v large files
  FILE *file = fopen( filename.c_str(), "r" );
  if( file == NULL ){
    utility_exit_with_message( "Cannot read loop DB file" + filename + "\n" );
  }

  loopdb_.clear();
  while( !feof( file ) ){
    const unsigned int bufsize = 128;
    LeapIndex bufferdata[bufsize];
    size_t readshorts =	fread(&bufferdata[0],sizeof(LeapIndex),bufsize,file);
    for( unsigned i = 0; i< readshorts; i ++ ){
      add_leap( bufferdata[i] );
    }
  }
  fclose( file );
	loopdb_.reserve( loopdb_.size() );

  // Now trigger giang rehash!
  TR.Info << "Rehashed dataset:" << backbone_index_map_.size() << "  " <<  loopdb_.size() << std::endl;
  TR.Info << "Rehashed dataset:  OUTOFBOUNDS: " <<  loopdb_.size() - backbone_index_map_.size() << std::endl;
}




} // namespace loops
} // namespace protocols
