// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loophash/MPI_LoopHashRefine_Emperor.cc
/// @brief
/// @author Mike Tyka

#define TRDEBUG TR.Debug

// MPI headers
#ifdef USEMPI
#include <mpi.h> //keep this first
#endif

#include <protocols/loophash/MPI_LoopHashRefine.hh>
#include <protocols/loophash/MPI_LoopHashRefine_Emperor.hh>
#include <protocols/loophash/WorkUnit_LoopHash.hh>
#include <protocols/wum/WorkUnit_BatchRelax.hh>
#include <protocols/wum/WorkUnitBase.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/pose/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/util.hh>
#include <core/io/pose_stream/MetaPoseInputStream.hh>
#include <core/io/pose_stream/util.hh>
#include <core/io/silent/SilentFileData.hh>
#include <core/io/silent/SilentStructFactory.hh>
#include <core/io/silent/SilentStruct.hh>
#include <core/io/silent/ProteinSilentStruct.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/lh.OptionKeys.gen.hh>
#include <core/options/option.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/util/Tracer.hh>
#include <protocols/wum/SilentStructStore.hh>
#include <ObjexxFCL/format.hh>
/// ObjexxFCL headers
#include <ObjexxFCL//string.functions.hh>
#include <ObjexxFCL/format.hh>

#include <numeric/random/random.hh>

#include <unistd.h>

using namespace ObjexxFCL;
using namespace ObjexxFCL::fmt;

namespace protocols {
namespace loophash {

using namespace protocols::wum;

static core::util::Tracer TR("MPI.LHR.Emperor");

static numeric::random::RandomGenerator RG(1248321);  // <- Magic number, do not change it (and dont try and use it anywhere else)


void
MPI_LoopHashRefine_Emperor::set_defaults(){
	using namespace core::options;
	using namespace core::options::OptionKeys;

	set_max_lib_size( option[ OptionKeys::lh::max_emperor_lib_size]() );
}


void
MPI_LoopHashRefine_Emperor::init(){
	// Are we resuming an old job ?
	if( mpi_resume() != "" ){ 
		TR << "Resuming job from IDENT:  " <<  mpi_resume() << std::endl;
		load_state( mpi_resume() );	
	};

	// Emperors have different library rules!
	set_mpi_feedback( "add_n_replace" ); 
	
	TR << "STARTLIB: " << std::endl;
	print_library();
}

void
MPI_LoopHashRefine_Emperor::go()
{
	// initialize master (this is a virtual functino call and this function is overloaded by the children of this class)
	TR << "Init Master: " << mpi_rank() << std::endl;
	init();

	TR << "Emperor Node: Waiting for data ..." << std::endl;
	while(true){
		// process any incoming messages such as incoming
		TRDEBUG << "Emperor: processing msgs.." << std::endl;
		process_incoming_msgs();

		TRDEBUG << "Emperor: process incoming" << std::endl;
		process_inbound_wus();  // lets borrow a master's routine

		TRDEBUG << "Emperor: process outbound" << std::endl;
		process_outbound_wus();// lets borrow a master's routine

		// ok, we've done all our work, now wait until we hear from our slaves/masters
		process_incoming_msgs( true );

		print_stats_auto();
	}
}


void
MPI_LoopHashRefine_Emperor::process_inbound_wus(){
	if( inbound().size() > 0 ){
		TR << "Processing inbound WUs on emperor .." << std::endl;
	}

	while( inbound().size() > 0 )
	{
		WorkUnitBaseOP  next_wu =  inbound().pop_next();
		runtime_assert( next_wu );
		WorkUnit_SilentStructStore* structure_wu = dynamic_cast<  WorkUnit_SilentStructStore * > ( (WorkUnitBase*) (&(*next_wu)) );

		if ( structure_wu == NULL ){
			TR << "Cannot save structural data for WU: " << std::endl;
			next_wu->print( TR );
			continue;
		} 

		SilentStructStore &decoys = structure_wu->decoys();
		decoys.all_sort_silent_scores();
		if ( structure_wu->get_wu_type() == "resultpack" ){
			TR << "Emperor: receivd structures: " << decoys.size() << std::endl;
			// dump structures
			if( add_structures_to_library( decoys )){
				dump_structures( decoys, false);
			}
		} else 
		if ( structure_wu->get_wu_type() == "getnewstruct" ){
			TR << "Emperor: received expired structures: " << decoys.size() << std::endl;
			// dump structures
			if( decoys.size() > 0 ){
				if( add_structures_to_library( decoys ) ){
					dump_structures( decoys, false );
					TR << "Sending a new random structure to master:" << structure_wu->last_received_from() << std::endl;	
					send_random_library_struct( structure_wu->last_received_from(), (core::Size) decoys.get_struct(0)->get_energy("ssid") );
					TR << "Done." << std::endl;
				}
			}
		} else {
			TR.Error << "ERROR: Unknown workunit received. " << std::endl;
		}

	}
	
	save_state_auto();
	print_stats();
}



void
MPI_LoopHashRefine_Emperor::process_outbound_wus(){
}




} // namespace loophash
} // namespace protocols


