// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/loops/LoopMover_Perturb_CCD.cc
/// @brief kinematic loop closure main protocols
/// @author Chu Wang
/// @author Mike Tyka

//// Unit Headers
#include <protocols/loops/loops_main.hh>
#include <protocols/loops/LoopMover.hh>
#include <protocols/loops/LoopMover_CCD.hh>
#include <protocols/loops/Loops.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <core/conformation/Residue.hh>

// Rosetta Headers
#include <core/chemical/VariantType.hh>
#include <core/chemical/util.hh>
#include <core/conformation/util.hh>
#include <core/id/TorsionID.hh>
#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/options/option.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/fragment/FragSet.hh>
#include <protocols/abinitio/FragmentMover.hh>

#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/TaskFactory.fwd.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/operation/TaskOperations.hh>
#include <core/pack/rotamer_trials.hh>
#include <core/pack/pack_rotamers.hh>
#include <protocols/moves/BackboneMover.hh>

#include <core/conformation/symmetry/util.hh>
#include <core/optimization/symmetry/SymAtomTreeMinimizer.hh>

#include <protocols/loops/util.hh>
#include <core/util/Tracer.hh> // tracer output

//Utility Headers
#include <numeric/random/random.hh>

// C++ Headers
#include <iostream>
#include <map>
#include <string>

// option key includes

#include <core/options/option.hh>
#include <core/options/keys/MonteCarlo.OptionKeys.gen.hh>
#include <core/options/keys/loops.OptionKeys.gen.hh>

//Auto Headers
#include <ObjexxFCL/format.hh>
#include <fstream>

//Auto using namespaces
namespace ObjexxFCL { namespace fmt { } } using namespace ObjexxFCL::fmt;
//Auto using namespaces end

namespace protocols {
namespace loops {

///////////////////////////////////////////////////////////////////////////////
using namespace core;

static numeric::random::RandomGenerator RG(42478);
extern core::util::Tracer tr;


//constructors
LoopMover_Perturb_CCD::LoopMover_Perturb_CCD(
	protocols::loops::Loops loops_in
) : IndependentLoopMover( loops_in )
{
	scorefxn_ = get_cen_scorefxn();

	protocols::moves::Mover::type("LoopMover_Perturb_CCD");
	set_default_settings();
}



LoopMover_Perturb_CCD::LoopMover_Perturb_CCD(
	protocols::loops::Loops loops_in,
	core::scoring::ScoreFunctionOP scorefxn
) : IndependentLoopMover( loops_in )
{
	if ( scorefxn ) {
		scorefxn_ = scorefxn;
	} else {
		scorefxn_ = get_cen_scorefxn();
	}

	protocols::moves::Mover::type("LoopMover_Perturb_CCD");
	set_default_settings();
}


LoopMover_Perturb_CCD::LoopMover_Perturb_CCD(
	protocols::loops::Loops loops_in,
	core::scoring::ScoreFunctionOP scorefxn,
	core::fragment::FragSetOP fragset
) : IndependentLoopMover( loops_in )
{
	if ( scorefxn ) {
		scorefxn_ = scorefxn;
	} else {
		scorefxn_ = get_cen_scorefxn();
	}

	protocols::moves::Mover::type("LoopMover_Perturb_CCD");
	set_default_settings();

	add_fragments( fragset );
}

//destructor
LoopMover_Perturb_CCD::~LoopMover_Perturb_CCD() {}

std::string
LoopMover_Perturb_CCD::get_name() const {
	return "LoopMover_Perturb_CCD";
}

//clone
protocols::moves::MoverOP LoopMover_Perturb_CCD::clone() const {
		return new LoopMover_Perturb_CCD(*this);
	}

LoopResult LoopMover_Perturb_CCD::model_loop(
	core::pose::Pose & pose,
  protocols::loops::Loop const & loop
) {
	using namespace scoring;
	using namespace optimization;
	using namespace options;
	using namespace core::kinematics;
	using namespace protocols::abinitio;

	bool const verbose( true );
	bool const local_debug( false );

	// store starting fold tree and cut pose
 	kinematics::FoldTree f_orig=pose.fold_tree();
	set_single_loop_fold_tree( pose, loop );

	/// prepare fragment movers
	MoveMapOP movemap = new MoveMap();
	movemap->set_bb_true_range( loop.start(), loop.stop() );

	Loops one_loop_loops;
	one_loop_loops.add_loop( loop );

	std::vector< FragmentMoverOP > fragmover;
	for ( std::vector< core::fragment::FragSetOP >::const_iterator
				it = frag_libs_.begin(), it_end = frag_libs_.end();
				it != it_end; it++ ) {
		ClassicFragmentMover *cfm = new ClassicFragmentMover( *it, movemap );
		cfm->set_check_ss( false );
		cfm->enable_end_bias_check( false );
		fragmover.push_back( cfm );
	}


	core::pose::Pose native_pose;
	if ( get_native_pose() ) {
		native_pose = *get_native_pose();
	}else{
		native_pose = pose;
	}

	Size const loop_begin( loop.start() ), loop_end( loop.stop() ), loop_cut( loop.cut() );
	Size const loop_size( loop_end - loop_begin + 1 );
	runtime_assert( !loop.is_terminal( pose ) || pose.fold_tree().is_cutpoint( loop_cut ) );

	// see if we should skip large frags. Don't if that would mean no fragment insertions:
  // mt temporary removed this.
	//// 	bool skip_long_frags( true ); // PBHACK DONT CHECKIN if false
  //// 	{
  //// 		bool found_any_frags( false );
  //// 	 frag_libs;
  //// 		for ( std::vector< core::fragment::ConstantLengthFragSetOP >::const_reverse_iterator
  //// 						it = frag_libs.rbegin(), it_end = frag_libs.rend(); it != it_end; ++it ) {
  //// 			Size const frag_size(  );
  //// 			if ( loop_size >= frag_size * 2 + 1 ) found_any_frags = true;
  //// 		}
  //// 		if ( !found_any_frags ) skip_long_frags = false;
  //// 	}

	tr << "perturb_one_loop_with_ccd: " << loop_begin << ' ' << loop_size << std::endl;

	Real const init_phi  ( -150.0 );
	Real const init_psi  (  150.0 );
	Real const init_omega(  180.0 );

	if ( !loop.is_terminal( pose ) ) {
		// set cutpoint variant for chainbreak scoring.
		chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_LOWER, loop_cut );
		chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_UPPER, loop_cut+1 );
	}

	pose::Pose start_pose;
	start_pose = pose;

	if (local_debug) {
		std::ofstream out("score.tmp_input_cen");
		out << "scoring before cen_perturb: " << (*scorefxn_)(pose) << std::endl;
		/// Now handled automatically.  scorefxn_->accumulate_residue_total_energies(pose);
		scorefxn_->show( out );
		out << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		tr << "before cen_perturb: "
							<< pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		out << pose.energies();
	}

	kinematics::MoveMap mm_one_loop;
	utility::vector1<bool> allow_sc_move( pose.total_residue(), false );
	loops_set_move_map( one_loop_loops, allow_sc_move, mm_one_loop);

	tr << "loop rmsd before initial fragment perturbation:" << loop_rmsd( pose, native_pose, one_loop_loops ) << std::endl;
	if ( loop.is_extended() ) {
		if (local_debug) {
			pose.dump_pdb("tmp_cen_preidl.pdb");
			(*scorefxn_)(pose);
			tr << "preidl: "
				 << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		}

		// avoid potential problems at the ends:
		if ( loop_begin > 1 && !pose.fold_tree().is_cutpoint( loop_begin-1 ) &&
				 pose.residue( loop_begin ).is_bonded( loop_begin-1 ) ) {
			conformation::insert_ideal_bonds_at_polymer_junction( loop_begin-1, pose.conformation() );
			pose.set_omega( loop_begin - 1, init_omega );
		}
		if ( loop_end < pose.total_residue() && !pose.fold_tree().is_cutpoint( loop_end ) &&
				 pose.residue( loop_end ).is_bonded( loop_end+1 ) ) {
			conformation::insert_ideal_bonds_at_polymer_junction( loop_end, pose.conformation() );
		}

		if (local_debug) {
			pose.dump_pdb("tmp_cen_endidl.pdb");
			(*scorefxn_)(pose);
			tr << "endidl: "
				 << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		}

		for ( Size i= loop_begin; i<= loop_end; ++i ) {
			conformation::idealize_position( i, pose.conformation() );
		}

		if (local_debug) {
			pose.dump_pdb("tmp_cen_postidl.pdb");
			(*scorefxn_)(pose);
			tr << "postidl: "
				 << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		}


		for ( Size i = loop_begin; i <= loop_end; ++ i ) {
			pose.set_phi  ( i, init_phi   );
			pose.set_psi  ( i, init_psi   );
			pose.set_omega( i, init_omega );
		}

		if (local_debug) {
			pose.dump_pdb("tmp_cen_init.pdb");
			(*scorefxn_)(pose);
			tr << "extended: "
								<< pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		}

		for ( core::Size i = loop.start(); i <= loop.stop(); ++i ) {
			for ( std::vector< FragmentMoverOP >::const_iterator
						it = fragmover.begin(),it_end = fragmover.end(); it != it_end; it++ ) {
				(*it)->apply( pose );
			}
		}


		if (local_debug) {
			pose.dump_pdb("tmp_cen_ran.pdb");
			(*scorefxn_)(pose);
			tr << "random frags: "
								<< pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
		}
	}
	tr << "loop rmsd after initial fragment perturbation:" << loop_rmsd( pose, native_pose, one_loop_loops ) << std::endl;

	// scheduler
	bool const fast = option[OptionKeys::loops::fast];
	int outer_cycles( 3 );
	int inner_cycles( fast ? std::min( Size(250), loop_size*5 ) : std::min( Size(1000), loop_size*20 ) );

	if ( option[OptionKeys::loops::debug] ) inner_cycles = 10;

	// Monte Carlo
	float const init_temp( 2.0 ), final_temp( 1.0 );
	float const gamma = std::pow( (final_temp/init_temp), (1.0f/(outer_cycles*inner_cycles)) );
	float temperature = init_temp;
	if ( local_debug ) { // hacking
		(*scorefxn_)(pose);
		tr << "before mc ctor: "
							<< pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) << std::endl;
	}

	protocols::moves::MonteCarlo mc( pose, *scorefxn_, temperature);

	// minimizer
	AtomTreeMinimizerOP minimizer;
	float const dummy_tol( 0.001 ); // linmin sets tol internally
	bool const use_nblist( false ), deriv_check( false ); // true ); // false );
	if ( core::conformation::symmetry::is_symmetric( pose ) ) {
		minimizer = dynamic_cast<AtomTreeMinimizer*> (new core::optimization::symmetry::SymAtomTreeMinimizer);
	} else {
		minimizer = new core::optimization::AtomTreeMinimizer;
	}

	MinimizerOptions options( "linmin", dummy_tol, use_nblist, deriv_check);

	mc.show_scores();

	for( int i=1; i<=outer_cycles; ++i ) {
		// recover low
		if ( local_debug) { // debug
			(*scorefxn_)( pose );
			tr << "befor rLOW: " << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) <<
				" rmsd: " << F(9,3,loop_rmsd( pose, native_pose, one_loop_loops )) << std::endl;
		}

		mc.recover_low(pose);

		if ( local_debug) { // debug
			(*scorefxn_)( pose );
			tr << "after rLOW: " << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) <<
				" rmsd: " << F(9,3,loop_rmsd( pose, native_pose, one_loop_loops )) << std::endl;
		}

		for( int j=1; j<=inner_cycles; ++j ) {
			// change temperature
			temperature *= gamma;
			mc.set_temperature( temperature );
			// use rbegin and rend to iterate from larger size to smaller size
			//for ( std::map< Size, protocols::frags::TorsionFragmentLibraryOP >::const_reverse_iterator
			//				it = frag_libs.rbegin(), it_end = frag_libs.rend(); it != it_end; ++it ) {
			for ( std::vector< FragmentMoverOP >::const_iterator
						it = fragmover.begin(),it_end = fragmover.end(); it != it_end; it++ ) {

				// skip if the loop is too short
				//Size const frag_size( it->first );

				//if ( loop_size < frag_size || ( skip_long_frags && loop_size < frag_size * 2 + 1 ) ) continue;
				// monte carlo fragment ccd minimization
				if ( verbose ) tr << "perturb out/in/frag: "
												  << i << "/" << outer_cycles << " "
												  << j << "/" << inner_cycles << std::endl;

				if ( local_debug) { // debug
					(*scorefxn_)( pose );
					tr << "befor frag: " << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) <<
						" rmsd: " << F(9,3,loop_rmsd( pose, native_pose, one_loop_loops )) << std::endl;
				}

				//insert_fragment( loop_begin, loop_end, pose, it->second );
				(*it)->apply( pose );

				if ( local_debug) { // debug
					(*scorefxn_)( pose );
					tr << "after frag: " << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) <<
						" rmsd: " << F(9,3,loop_rmsd( pose, native_pose, one_loop_loops )) << std::endl;
				}

				if ( !loop.is_terminal( pose ) ) ccd_close_loops( pose, one_loop_loops, mm_one_loop);

				if ( local_debug) { // debug
					(*scorefxn_)( pose );
					tr << "after ccd:  " << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) <<
						" rmsd: " << F(9,3,loop_rmsd( pose, native_pose, one_loop_loops )) << std::endl;
				}

				minimizer->run( pose, mm_one_loop, *scorefxn_, options );

				if ( local_debug) { // debug
					(*scorefxn_)( pose );
					tr << "after min:  " << pose.energies().total_energies().weighted_string_of( scorefxn_->weights() ) <<
						" rmsd: " << F(9,3,loop_rmsd( pose, native_pose, one_loop_loops )) << std::endl;
				}

				std::string move_type = "frag_ccd_min";
				mc.boltzmann( pose );
				mc.show_scores();
			} // for each fragment size
		} // inner_cycles
	} // outer_cycles
	pose = mc.lowest_score_pose();

	if (local_debug) {
		std::ofstream out("score.tmp_perturb_cen");
		out << "scoring after cen_perturb: " << (*scorefxn_)(pose) << std::endl;
		/// Now handled automatically.  scorefxn_->accumulate_residue_total_energies(pose);
		scorefxn_->show( out );
		out << pose.energies();
	}

	// return to original fold tree
	pose.fold_tree( f_orig );
	return Success;

}



//constructors
LoopMover_Refine_CCD::LoopMover_Refine_CCD(
	protocols::loops::Loops  loops_in
) : LoopMover( loops_in ),
		outer_cycles_(3),
		max_inner_cycles_(200),
		repack_period_(20),
		temp_initial_(1.5),
		temp_final_(0.5)
{
	read_options();
	scorefxn_ = get_fa_scorefxn();
	protocols::moves::Mover::type("LoopMover_Refine_CCD");
	set_default_settings();
}


LoopMover_Refine_CCD::LoopMover_Refine_CCD(
	protocols::loops::Loops  loops_in,
	core::scoring::ScoreFunctionOP  scorefxn
) : LoopMover( loops_in ),
		outer_cycles_(3),
		max_inner_cycles_(200),
		repack_period_(20),
		temp_initial_(1.5),
		temp_final_(0.5)
{
	read_options();
	scorefxn_ = scorefxn;
	protocols::moves::Mover::type("LoopMover_Refine_CCD");
	set_default_settings();
}

//destructor
LoopMover_Refine_CCD::~LoopMover_Refine_CCD() {}

std::string
LoopMover_Refine_CCD::get_name() const {
	return "LoopMover_Refine_CCD";
}

//clone
protocols::moves::MoverOP LoopMover_Refine_CCD::clone() const {
		return new LoopMover_Refine_CCD(*this);
}


void LoopMover_Refine_CCD::set_default_settings()
{
		redesign_loop_ = false;
		packing_isolated_to_active_loops_ = false;
}

void
LoopMover_Refine_CCD::read_options()
{
	using namespace options;
	if ( option[ OptionKeys::loops::outer_cycles ].user() )
		outer_cycles_ = option[ OptionKeys::loops::outer_cycles ]();
	if ( option[ OptionKeys::loops::max_inner_cycles ].user() )
		max_inner_cycles_ = option[ OptionKeys::loops::max_inner_cycles ]();
	if ( option[ OptionKeys::loops::repack_period ].user() )
		repack_period_ = option[ OptionKeys::loops::repack_period ]();
	if ( option[ OptionKeys::MonteCarlo::temp_initial ].user() )
		temp_initial_ = option[ OptionKeys::MonteCarlo::temp_initial ]();
	if ( option[ OptionKeys::MonteCarlo::temp_final ].user() )
		temp_final_ = option[ OptionKeys::MonteCarlo::temp_final ]();
}

void LoopMover_Refine_CCD::set_task_factory(
	core::pack::task::TaskFactoryCOP task_factory_in
)
{
	// make local, non-const copy from const input
	runtime_assert( task_factory_in );
	task_factory_ = new core::pack::task::TaskFactory( *task_factory_in );
}

core::pack::task::TaskFactoryCOP LoopMover_Refine_CCD::get_task_factory() const { return task_factory_; }

void LoopMover_Refine_CCD::apply(
	core::pose::Pose & pose
) {
	using namespace optimization;
	using namespace scoring;
	using namespace options;

	core::pose::Pose native_pose;
	if ( get_native_pose() ) native_pose = *get_native_pose();
	else native_pose = pose;

	// 'verbose' should be an option, or better yet Tracer's output filtering capabilities should be used instead
	bool const verbose( true );
	bool const local_debug( option[ OptionKeys::loops::debug ].user() );

	// set cutpoint variants for correct chainbreak scoring
	Size const nres( pose.total_residue() );   //fpd fix for symmetry
	utility::vector1< bool > is_loop( nres, false );

	for( Loops::const_iterator it=loops_.begin(), it_end=loops_.end();
			 it != it_end; ++it ) {
		for ( Size i= it->start(); i<= it->stop(); ++i ) {
			is_loop[i] = true;
		}
		Size const loop_cut(it->cut());
		//if ( loop_cut == nres ) { //c-terminal loop
		//	utility_exit_with_message("This doesnt make sense; ask Chu about it, Phil.");
		//	if ( ! pose.residue(loop_cut - 1).has_variant_type(chemical::CUTPOINT_LOWER) )
		//		chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_LOWER, loop_cut - 1);
		//	if ( ! pose.residue(loop_cut).has_variant_type(chemical::CUTPOINT_UPPER) )
		//		chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_UPPER, loop_cut );
		//} else {

	//if ( loop_cut != nres ) { //c-terminal loop  //fpd fix for symmetry
	if ( !pose.residue(loop_cut).is_upper_terminus() ) {
			if ( ! pose.residue(loop_cut).has_variant_type(chemical::CUTPOINT_LOWER) )
				chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_LOWER, loop_cut );
			if ( ! pose.residue(loop_cut+1).has_variant_type(chemical::CUTPOINT_UPPER) )
				chemical::add_variant_type_to_pose_residue( pose, chemical::CUTPOINT_UPPER, loop_cut+1 );
		}
	}

	// scheduler
	int const fast( option[ OptionKeys::loops::fast ] ); // why is this an int?
	Size const inner_cycles(
		std::min( max_inner_cycles_, fast ? loops_.loop_size() : Size(10)*loops_.loop_size() ) );

	// scorefxn
	scoring::ScoreFunctionOP scorefxn;
	if ( scorefxn_ != 0 ) scorefxn = scorefxn_->clone();
	else scorefxn = get_fa_scorefxn();

	// confirm that chainbreak weight is set
	scorefxn->set_weight( chainbreak, 1. * 10. / 3. );

	// monte carlo
	Real const gamma(
		std::pow( ( temp_final_ / temp_initial_ ), Real( 1.0f / ( outer_cycles_ * inner_cycles ) ) ) );

	Real temperature( temp_initial_ );
	// need to make sure we have scored before we ask for a tenA graph later...
	(*scorefxn)(pose);

	protocols::moves::MonteCarlo mc( pose, *scorefxn, temperature );
	// minimizer
	AtomTreeMinimizerOP minimizer;
	MinimizerOptions options( "dfpmin", 0.001, true /*use_nblist*/, false /*deriv_check*/ );
	bool const repack_neighbors( ! option[ OptionKeys::loops::fix_natsc ] );
	if ( core::conformation::symmetry::is_symmetric( pose ) ) {
		minimizer = dynamic_cast<AtomTreeMinimizer*>( new core::optimization::symmetry::SymAtomTreeMinimizer );
	} else {
		minimizer = new core::optimization::AtomTreeMinimizer;
	}

	pack::task::PackerTaskOP base_packer_task;

	// create default Packer behavior if none has been set via TaskFactory
	if ( task_factory_ == 0 ) {
		// the default Packer behavior is defined here
		using namespace core::pack::task;
		task_factory_ = new TaskFactory;
		task_factory_->push_back( new operation::InitializeFromCommandline );
		task_factory_->push_back( new operation::IncludeCurrent );
		base_packer_task = task_factory_->create_task_and_apply_taskoperations( pose );
		if ( redesign_loop_ ) {
			// allow design at loop positions
			for ( Size i=1; i<= nres; ++i ) {
				if ( !is_loop[i] ) base_packer_task->nonconst_residue_task( i ).restrict_to_repacking();
			}
		} else {
			// restrict to repacking at all positions
			base_packer_task->restrict_to_repacking();
		}
		// additional default behavior: packing restricted to active loops
		packing_isolated_to_active_loops_ = true;
	} else {
		base_packer_task = task_factory_->create_task_and_apply_taskoperations( pose );
	}
	base_packer_task->set_bump_check( true );

	// repack trial
	pack::task::PackerTaskOP this_packer_task( base_packer_task->clone() );
	utility::vector1<bool> allow_repacked( nres, false );
	if ( packing_isolated_to_active_loops_ ) {
		select_loop_residues( pose, loops_, repack_neighbors, allow_repacked, 10.0 /* neighbor_cutoff */ );
		core::conformation::symmetry::make_residue_mask_symmetric( pose, allow_repacked );
		             // does nothing if pose is not symm
		this_packer_task->restrict_to_residues( allow_repacked );
	}
	pack::pack_rotamers( pose, *scorefxn, this_packer_task );
	std::string move_type = "repack";
	mc.boltzmann( pose, move_type );
	mc.show_scores();

	if (local_debug) {
		pose.dump_pdb("tmp_fa_repack.pdb");
		std::ofstream out("score.tmp_repack_fa");
		out << "scoring for repack_fa " << (*scorefxn)(pose) << std::endl;
		scorefxn->show( out );
		out << pose.energies();
	}

	// remember all packable/repacked residues
	for ( Size index(1); index <= nres; ++index ) {
		if ( !this_packer_task->residue_task(index).being_packed() ) allow_repacked[index] = false;
		else allow_repacked[index] = true;
	}
	// set minimization degrees of freedom for all loops
	kinematics::MoveMap mm_all_loops;
	setup_movemap( pose, loops_, allow_repacked, mm_all_loops );

	// small/shear move parameters
	// should be options
	Size const nmoves = { 1 };
	std::map< char, Real > angle_max;
	angle_max.insert( std::make_pair('H', 0.0) );
	angle_max.insert( std::make_pair('E', 5.0) );
	angle_max.insert( std::make_pair('L', 6.0) );

	for (Size i=1; i<=outer_cycles_; ++i) {
		// increase CHAINBREAK weight and update monte carlo
		scorefxn->set_weight( chainbreak, Real(i)*10.0/3.0 );
		mc.score_function( *scorefxn );
		// recover low
		mc.recover_low( pose );
		// score info
		if ( verbose ) tr << "cycle: " << i << "  " << (*scorefxn)(pose) << std::endl;
		for ( Size j=1; j<=inner_cycles; ++j ) {
			temperature *= gamma;
			mc.set_temperature( temperature );
			if ( verbose ) tr << "refinement cycle (outer/inner): "
								<< i << "/" << outer_cycles_ << " "
								<< j << "/" << inner_cycles << " "
								<< std::endl;
			{// small_CCD_min_trial
				Loops::const_iterator it( loops_.one_random_loop() );
				Loops one_loop;
				one_loop.add_loop( it );
				// set up movemap properly
				kinematics::MoveMapOP mm_one_loop( new kinematics::MoveMap() );
				setup_movemap( pose, one_loop, allow_repacked, *mm_one_loop );

				if (local_debug) {
					tr << "chutmp-debug small_move-0: " << "  " << (*scorefxn)(pose) << std::endl;
					tr << "small_move-0: " << pose.energies().total_energies().weighted_string_of( scorefxn->weights() )
										<< " rmsd: " << F(9,3,loop_rmsd( pose, native_pose, loops_ )) << std::endl;
					pose.dump_pdb("small_move-0.pdb");
				}

				protocols::moves::SmallMover small_moves( mm_one_loop, temperature, nmoves );
				small_moves.apply( pose );

				if (local_debug) {
					tr << "chutmp-debug small_move-1: " << "  " << (*scorefxn)(pose) << std::endl;
					tr << "small_move-1: " << pose.energies().total_energies().weighted_string_of( scorefxn->weights() )
										<< " rmsd: " << F(9,3,loop_rmsd( pose, native_pose, loops_ )) << std::endl;
					pose.dump_pdb("small_move-1.pdb");
					std::ofstream out("score.small_move_1");
					out << "scoring of input_pose " << (*scorefxn)(pose) << std::endl;
					scorefxn->show( out );
					out << pose.energies();
				}

				if (! it->is_terminal( pose ) ) ccd_close_loops( pose, one_loop, *mm_one_loop);

				if (local_debug) {
					tr << "chutmp-debug small_move-2: " << "  " << (*scorefxn)(pose) << std::endl;
					tr << "small_move-2: " << pose.energies().total_energies().weighted_string_of( scorefxn->weights() )
										<< " rmsd: " << F(9,3,loop_rmsd( pose, native_pose, loops_ )) << std::endl;
					pose.dump_pdb("small_move-2.pdb");
					std::ofstream out("score.small_move_2");
					out << "scoring of input_pose " << (*scorefxn)(pose) << std::endl;
					scorefxn->show( out );
					out << pose.energies();
				}

				pack::rotamer_trials( pose, *scorefxn, this_packer_task );

				if (local_debug) {
					tr << "chutmp-debug small_move-3: " << "  " << (*scorefxn)(pose) << std::endl;
					tr << "small_move-3: " << pose.energies().total_energies().weighted_string_of( scorefxn->weights() )
										<< " rmsd: " << F(9,3,loop_rmsd( pose, native_pose, loops_ )) << std::endl;
					pose.dump_pdb("small_move-3.pdb");
				}

				setup_movemap( pose, loops_, allow_repacked, mm_all_loops );
				minimizer->run( pose, mm_all_loops, *scorefxn, options );

				if (local_debug) {
					tr << "chutmp-debug small_move-4: " << "  " << (*scorefxn)(pose) << std::endl;
					tr << "small_move-4: " << pose.energies().total_energies().weighted_string_of( scorefxn->weights() )
										<< " rmsd: " << F(9,3,loop_rmsd( pose, native_pose, loops_ )) << std::endl;
					pose.dump_pdb("small_move-4.pdb");
				}

				std::string move_type = "small_ccd_min";
				mc.boltzmann( pose, move_type );

				if (local_debug) {
					tr << "chutmp-debug small_move-5: " << "  " << (*scorefxn)(pose) << std::endl;
					tr << "small_move-5: " << pose.energies().total_energies().weighted_string_of( scorefxn->weights() )
										<< " rmsd: " << F(9,3,loop_rmsd( pose, native_pose, loops_ )) << std::endl;
					pose.dump_pdb("small_move-5.pdb");
				}

				mc.show_scores();
			}
			{// shear_CCD_min_trial
				Loops::const_iterator it( loops_.one_random_loop() );
				Loops one_loop;
				one_loop.add_loop( it );
				// set up movemap properly
				kinematics::MoveMapOP mm_one_loop( new kinematics::MoveMap() );
				setup_movemap( pose, one_loop, allow_repacked, *mm_one_loop );
				protocols::moves::ShearMover shear_moves( mm_one_loop, temperature, nmoves );
				shear_moves.apply( pose );
				if (! it->is_terminal( pose ) ) ccd_close_loops( pose, one_loop, *mm_one_loop);
				pack::rotamer_trials( pose, *scorefxn, this_packer_task );
				(*scorefxn)(pose); // update 10A nbr graph, silly way to do this
				setup_movemap( pose, loops_, allow_repacked, mm_all_loops );
				minimizer->run( pose, mm_all_loops, *scorefxn, options );
				std::string move_type = "shear_ccd_min";
				mc.boltzmann( pose, move_type );
				mc.show_scores();
			}
			{ //main_repack_trial
				if ( (j%repack_period_)==0 || j==inner_cycles ) {
					// repack trial

					if ( packing_isolated_to_active_loops_ ) {
						select_loop_residues( pose, loops_, repack_neighbors, allow_repacked, 10.0 /* neighbor_cutoff */ );
					}
					core::conformation::symmetry::make_residue_mask_symmetric( pose, allow_repacked );  //fpd symmetrize res mask -- does nothing if pose is not symm
					this_packer_task->restrict_to_residues( allow_repacked );
					pack::pack_rotamers( pose, *scorefxn, this_packer_task );
					std::string move_type = "repack";
					mc.boltzmann( pose, move_type );
					mc.show_scores();
				}
			}
			if ( verbose || local_debug ) tr << std::flush;
		} //inner_cycle
	} //outer_cycle

	mc.show_counters();
	pose = mc.lowest_score_pose();

}


/// @brief setup an appropriate movemap for the given loops
/// @param[in] loops The loops to model.
/// @param[in] allow_repack Indicates whether or not to allow a position to
///  repack.
/// @param[out] movemap Output movemap, all settings added here.
/// @remarks will enforce the false movemap
void LoopMover_Refine_CCD::setup_movemap(
	core::pose::Pose const & pose,
	protocols::loops::Loops const & loops,
	utility::vector1< bool > const & allow_repack,
	core::kinematics::MoveMap & movemap
)
{
	loops_set_move_map( loops, allow_repack, movemap );
	enforce_false_movemap( movemap );
	if ( core::conformation::symmetry::is_symmetric( pose ) )  {
		core::conformation::symmetry::make_symmetric_movemap( pose, movemap );
	}
}


} // namespace loops
} // namespace protocols
