// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/MatchSet.cc
/// @brief
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini

// Unit headers
#include <protocols/match/MatchSet.hh>

// Utility headers
#include <utility/LexicographicalIterator.hh>

//Auto Headers
#include <protocols/match/Hit.hh>



namespace protocols {
namespace match {

HitHasher::HitHasher() : initialized_( false ) {}
HitHasher::~HitHasher() {}

void
HitHasher::set_bounding_box(
	BoundingBox const & bb
)
{
	assert( ! initialized_ );
	bb_ = bb;
}

void
HitHasher::set_uniform_xyz_bin_width( Real bin_width )
{
	assert( ! initialized_ );
	assert( bin_width > 0 );
	for ( Size ii = 1; ii <= 3; ++ii ) { xyz_bin_widths_[ ii ] = bin_width; }
}

void
HitHasher::set_uniform_euler_angle_bin_width( Real bin_width_degrees )
{
	assert( ! initialized_ );
	assert( bin_width_degrees > 0 );
	for ( Size ii = 1; ii <= 3; ++ii ) { euler_bin_widths_[ ii ] = bin_width_degrees; }
}

void
HitHasher::set_xyz_bin_widths( Vector const & bin_widths )
{
	assert( ! initialized_ );
	for ( Size ii = 1; ii <= 3; ++ii ) {
		assert( bin_widths( ii ) > 0 );
		xyz_bin_widths_[ ii ] = bin_widths( ii );
	}

}

void
HitHasher::set_euler_bin_widths( Vector const & euler_bin_widths )
{
	assert( ! initialized_ );
	for ( Size ii = 1; ii <= 3; ++ii ) {
		assert( euler_bin_widths( ii ) > 0 );
		euler_bin_widths_[ ii ] = euler_bin_widths( ii );
	}
}

void
HitHasher::set_nhits_per_match( Size num_geometric_constraints )
{
	assert( ! initialized_ );
	n_geometric_constraints_per_match_ = num_geometric_constraints;
}

void
HitHasher::initialize()
{
	assert( ! initialized_ );

	initialized_ = true;

	hit_hashes_.resize( N_HASH_MAPS );

	/// Data for initializing the lower corner in xyz space
	Vector xyz_lower;
	Real3 half_xyzbin_widths = xyz_bin_widths_;
	for ( Size ii = 1; ii <= 3; ++ii ) { half_xyzbin_widths[ ii ] *= 0.5; }

	/// Data for initializing the lower corner in Euler space
	Size3 euler_offsets;
	utility::vector1< Size > six_twos( 6, 2 );
	utility::LexicographicalIterator lex( six_twos );

	Real6 bin_widths;
	bin_widths[ 1 ] = xyz_bin_widths_[ 1 ];   bin_widths[ 2 ] = xyz_bin_widths_[ 2 ];   bin_widths[ 3 ] = xyz_bin_widths_[ 3 ];
	bin_widths[ 4 ] = euler_bin_widths_[ 1 ]; bin_widths[ 5 ] = euler_bin_widths_[ 2 ]; bin_widths[ 6 ] = euler_bin_widths_[ 3 ];

	Size count = 1;

	//std::cout << "HitHasher initialize:";
	//for ( Size ii = 1; ii <= 6; ++ii ) std::cout << bin_widths[ ii ] << " "; std::cout << std::endl;

	while ( ! lex.at_end() ) {

		for ( Size ii = 1; ii <= 3; ++ii ) {
			xyz_lower( ii ) = bb_.lower()( ii ) + ( lex[ ii ] - 1 ) * ( half_xyzbin_widths[ ii ] );
		}
		for ( Size ii = 1, iip3 = 4; ii <= 3; ++ii, ++iip3 ) {
			euler_offsets[ ii ] = ( lex[ iip3 ] - 1 ); // 1 or 0 for offset or not.
		}

		//std::cout << "bounding box " << count << " lower: ";
		//for ( Size ii = 1; ii <= 3; ++ii ) std::cout << xyz_lower( ii ) << " ";
		//std::cout << "upper: ";
		//for ( Size ii = 1; ii <= 3; ++ii ) std::cout << bb_.upper()( ii ) << " ";
		//for ( Size ii = 1; ii <= 3; ++ii ) std::cout << euler_offsets[ ii ] << " "; std::cout << std::endl;

		BoundingBox bb( xyz_lower, bb_.upper() );

		hit_hashes_[ count ].first = new SixDCoordinateBinner( bb, euler_offsets, bin_widths );
		++lex;
		++count;
	}
}

void
HitHasher::insert_hit( Size geometric_constraint_id, Hit const * hit )
{
	runtime_assert( initialized_ );
	runtime_assert( hit );
	Real6 const & geom( hit->second() );
	Vector point( Vector( geom[ 1 ], geom[ 2 ], geom[ 3 ] ));
	if ( ! bb_.contains( point ) ) {
		utility_exit_with_message( "ERROR: Attempted to insert hit into HitHasher outside of the HitHasher bounding box!" );
	}

	for ( Size ii = 1; ii <= N_HASH_MAPS; ++ii ) {
		if ( hit_hashes_[ ii ].first->contains( geom ) ) {
			boost::uint64_t bin_index = hit_hashes_[ ii ].first->bin_index( geom );
			HitHash::iterator iter = hit_hashes_[ ii ].second.find( bin_index );
			if ( iter == hit_hashes_[ ii ].second.end() ) {

				/*if ( geometric_constraint_id != 1 ) {
					std::cout << "Weird -- this bin index " << bin_index << " should already have been inserted to hash " << ii;
					for ( Size jj = 1; jj <= 6; ++jj ) std::cout << " " << geom[ jj ]; std::cout << std::endl;
				} else {
					std::cout << "Inserting bin index: " << bin_index << " into hash " << ii << std::endl;
				}*/

				MatchSet ms( n_geometric_constraints_per_match_ );
				ms[ geometric_constraint_id ].push_back( hit );
				hit_hashes_[ ii ].second.insert( std::make_pair( bin_index, ms ));
			} else {
				iter->second[ geometric_constraint_id ].push_back( hit );
			}
		}
	}
}


/// @brief Insert a hits into a particular hash maps
void
HitHasher::insert_hit( Size which_hash_map, Size geometric_constraint_id, Hit const * hit )
{
	runtime_assert( initialized_ );
	runtime_assert( hit );
	Real6 const & geom( hit->second() );
	Vector point( Vector( geom[ 1 ], geom[ 2 ], geom[ 3 ] ));
	if ( ! bb_.contains( point ) ) {
		std::cerr << "Weird: Attempted to insert hit into HitHasher outside of the HitHasher bounding box!" << std::endl;
		std::cerr << "point: " << point.x() << " " << point.y() << " " << point.z() << std::endl;
		std::cerr << "bb: lower "<< bb_.lower().x() << " " << bb_.lower().y() << " " << bb_.lower().z() << std::endl;
		std::cerr << "bb: upper "<< bb_.upper().x() << " " << bb_.upper().y() << " " << bb_.upper().z() << std::endl;
		return;
		//utility_exit_with_message( "ERROR: Attempted to insert hit into HitHasher outside of the HitHasher bounding box!" );
	}

	if ( hit_hashes_[ which_hash_map ].first->contains( geom ) ) {
		boost::uint64_t bin_index = hit_hashes_[ which_hash_map ].first->bin_index( geom );
		HitHash::iterator iter = hit_hashes_[ which_hash_map ].second.find( bin_index );
		if ( iter == hit_hashes_[ which_hash_map ].second.end() ) {

			/*if ( geometric_constraint_id != 1 ) {
				std::cout << "Weird -- this bin index " << bin_index << " should already have been inserted to hash " << which_hash_map;
				for ( Size jj = 1; jj <= 6; ++jj ) std::cout << " " << geom[ jj ]; std::cout << std::endl;
			} else {
				std::cout << "Inserting bin index: " << bin_index << " into hash " << which_hash_map << std::endl;
			}*/

			MatchSet ms( n_geometric_constraints_per_match_ );
			ms[ geometric_constraint_id ].push_back( hit );
			hit_hashes_[ which_hash_map ].second.insert( std::make_pair( bin_index, ms ));
		} else {
			iter->second[ geometric_constraint_id ].push_back( hit );
		}
	}
}

void
HitHasher::clear_hash_map( Size which_hash_map )
{
	hit_hashes_[ which_hash_map ].second.clear();
}

HitHasher::HitHash::const_iterator
HitHasher::hit_hash_begin( Size which_hash_map ) const
{
	return hit_hashes_[ which_hash_map ].second.begin();
}

HitHasher::HitHash::const_iterator
HitHasher::hit_hash_end( Size which_hash_map ) const
{
	return hit_hashes_[ which_hash_map ].second.end();
}


MatchOutputTracker::MatchOutputTracker() {}

void
MatchOutputTracker::note_output_match( match_lite const & m )
{
	MatchHash::const_iterator iter = hash_.find( m );
	if ( iter == hash_.end() ) {
		hash_.insert( std::make_pair( m, true ) );
	}
}

bool
MatchOutputTracker::match_has_been_output( match_lite const & m ) const
{
	return hash_.find( m ) != hash_.end();
}



}
}
