// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/match/output/MatchOutputter.cc
/// @brief  Implementation of class to write output matches that pass filters
/// This class does not "look ahead" to future matches to decide whether the current match
/// to process should be output, however, filters are able to keep a history of what they've
/// output so far.
/// @author Alex Zanghellini (zanghell@u.washington.edu)
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com), porting to mini

// Unit headers
#include <protocols/match/output/MatchOutputter.hh>

// Package headers
#include <protocols/match/output/MatchProcessor.hh>
#include <protocols/match/output/MatchFilter.hh>
#include <protocols/match/output/OutputWriter.hh>

// Utility headers
#include <utility/pointer/ReferenceCount.hh>

// C++ headers
#include <iostream>

namespace protocols {
namespace match {
namespace output {

MatchOutputter::MatchOutputter()
	: MatchProcessor()
{}

MatchOutputter::~MatchOutputter() {}

void
MatchOutputter::begin_processing()
{
	MatchProcessor::begin_processing();
}

void
MatchOutputter::end_processing()
{
	MatchProcessor::end_processing();
}

void
MatchOutputter::process_match(
	match const & m
)
{
	note_match_processed();
	for ( std::list< MatchFilterOP >::const_iterator
			iter = filters_.begin(), iter_end = filters_.end();
			iter != iter_end; ++iter ) {
		if (  ! (*iter)->passes_filter( m ) ) {
			note_filter_fail( (*iter)->filter_name() );
			return;
		}
	}

	for ( std::list< StateAccumulatingMatchFilterOP >::const_iterator
			iter = filters_with_state_.begin(), iter_end = filters_with_state_.end();
			iter != iter_end; ++iter ) {
		(*iter)->note_match_accepted( m );
	}

	if ( writer_ ) {
		writer_->record_match( m );
	}
}


void
MatchOutputter::process_match(
	match_dspos1 const & m
)
{
	note_match_processed();
	for ( std::list< MatchFilterOP >::const_iterator
			iter = filters_.begin(), iter_end = filters_.end();
			iter != iter_end; ++iter ) {
		if (  ! (*iter)->passes_filter( m ) ) {
			note_filter_fail( (*iter)->filter_name() );
			return;
		}
	}

	for ( std::list< StateAccumulatingMatchFilterOP >::const_iterator
			iter = filters_with_state_.begin(), iter_end = filters_with_state_.end();
			iter != iter_end; ++iter ) {
		(*iter)->note_match_accepted( m );
	}

	if ( writer_ ) {
		writer_->record_match( m );
	}
}

void
MatchOutputter::add_filter( MatchFilterOP filter )
{
	filters_.push_back( filter );

	/// Is this a state-accululating filter?
	StateAccumulatingMatchFilterOP filter_with_state(
		dynamic_cast< StateAccumulatingMatchFilter * > ( filter.get() ) );
	if ( filter_with_state ) {
		filters_with_state_.push_back( filter_with_state );
	}
}


void
MatchOutputter::reset_filters()
{
	for ( std::list< StateAccumulatingMatchFilterOP >::const_iterator
			iter = filters_with_state_.begin(), iter_end = filters_with_state_.end();
			iter != iter_end; ++iter ) {
		(*iter)->reset();
	}
}

void
MatchOutputter::clear_filters()
{
	filters_.clear();
}

}
}
}
