// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file TrialMover
/// @brief performs a move and accepts it according to Monte Carlo accept/reject criterion.
/// @author Monica Berrondo

#include <protocols/moves/DME_FilterMover.hh>

// Rosetta Headers
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
// AUTO-REMOVED #include <core/util/basic.hh>
#include <core/util/Tracer.hh>

// AUTO-REMOVED #include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/Mover.hh>

#include <core/graph/PointGraph.hh>
#include <core/graph/find_neighbors.hh>

// Random number generator
// AUTO-REMOVED #include <numeric/random/random.hh>

static core::util::Tracer TR( "protocols.DME_FilterMover" );

//
#include <string>

//Auto Headers
#include <core/graph/PointGraphData.hh>
#include <core/graph/UpperEdgeGraph.hh>


namespace protocols {
namespace moves {

using namespace core;


/// @details  Setup a pointgraph for later use in dma calcs
graph::PointGraphOP
setup_dme_point_graph( pose::Pose const & ref_pose, Real const threshold )
{
	graph::PointGraphOP pg( new graph::PointGraph );
	graph::residue_point_graph_from_pose( ref_pose, *pg );
	graph::find_neighbors( pg, threshold );
	return pg;
}

/// @details  Calculate the dme using a pointgraph
Real
point_graph_dme( graph::PointGraph const & pg, pose::Pose const & pose )
{
	Size total(0);
	Real dme(0.0);
	for ( Size i=1; i<= pose.total_residue(); ++i ) {
		conformation::Residue const & i_rsd( pose.residue(i) );
		for ( graph::PointGraph::UpperEdgeListConstIter
						i_iter     = pg.get_vertex( i ).const_upper_edge_list_begin(),
						i_end_iter = pg.get_vertex( i ).const_upper_edge_list_end();
					i_iter != i_end_iter; ++i_iter ) {
			Size const j = i_iter->upper_vertex();
			Real const reference_distance( std::sqrt( i_iter->data().dsq() ) );
			Real const pose_distance( i_rsd.nbr_atom_xyz().distance( pose.residue(j).nbr_atom_xyz() ) );
			dme += ( reference_distance - pose_distance ) * ( reference_distance - pose_distance );
			++total;
		}
	}
	return std::sqrt( dme / total );
}

/// @details  Keep trying to make a move with my_mover until the dme is less than our threshold or
/// max_tries is exceeded
/// @note  At the expense of a few additional pose copies we could save the pose with the best dme and use
/// that if we exceed max_tries

void
DME_FilterMover::apply( pose::Pose & pose )
{
	// setup a pointgraph for future dme calculations
	graph::PointGraphOP pg( setup_dme_point_graph( pose, 10.0 ) );

	// for undoing moves with dme's that are too big
	pose::Pose const start_pose( pose );

	Size ntries( 0 );
	while ( true ) { // keep looping until we succeed
		++ntries;

		my_mover_->apply( pose );

		Real const dme( point_graph_dme( *pg, pose ) );

		TR.Trace << "apply: " << type() << " ntries= " << ntries << " dme= " << dme << std::endl;

		if ( ntries >= max_tries_ || dme < dme_threshold_ ) break;

		pose = start_pose;
	}
}

std::string
DME_FilterMover::get_name() const {
	return "DME_FilterMover";
}

}  // namespace moves
}  // namespace protocols
