// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/protocols/moves/GenericMonteCarloMover.cc
/// @brief perform a given mover and sample structures by MonteCarlo
/// @detailed The score evaluation of pose during MC after applying mover is done by
/// either FilterOP that can do report_sm() or ScoreFunctionOP.
/// By setting sample_type_ to high, you can also sample the pose that have higher score.
/// @author Nobuyasu Koga ( nobuyasu@uw.edu )


// Unit Headers
#include <protocols/moves/GenericMonteCarloMover.hh>
#include <protocols/moves/GenericMonteCarloMoverCreator.hh>

// Package Headers

// Project Headers
#include <core/pose/Pose.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/util/Tracer.hh>
#include <protocols/filters/Filter.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/MoverStatus.hh>

// Parser headers
#include <protocols/moves/DataMap.hh>
// Auto-header: duplicate removed #include <protocols/moves/Mover.hh>
// Auto-header: duplicate removed #include <protocols/filters/Filter.hh>
#include <utility/Tag/Tag.hh>

// Utility headers
#include <ObjexxFCL/format.hh>
#include <numeric/random/random.hh>

//// C++ headers

static core::util::Tracer TR("protocols.moves.GenericMonteCarloMover");
static numeric::random::RandomGenerator mc_RG(61452); // <- Magic number, do not change it!!!

using namespace core;

namespace protocols {
namespace moves {

using namespace ObjexxFCL::fmt;

std::string
GenericMonteCarloMoverCreator::keyname() const
{
	return GenericMonteCarloMoverCreator::mover_name();
}

protocols::moves::MoverOP
GenericMonteCarloMoverCreator::create_mover() const {
	return new GenericMonteCarloMover;
}

std::string
GenericMonteCarloMoverCreator::mover_name()
{
	return "GenericMonteCarlo";
}

/// @brief default constructor
GenericMonteCarloMover::GenericMonteCarloMover():
	Mover("GenericMonteCarlo"),
	maxtrials_( 10 ),
	mover_( NULL ),
	filter_( NULL ),
	scorefxn_( NULL ),
	temperature_( 0.0 ),
	sample_type_( "low" ),
	drift_( true ),
	last_accepted_pose_( NULL ),
	lowest_score_pose_( NULL )
{
	initialize();
}

/// @brief value constructor
GenericMonteCarloMover::GenericMonteCarloMover(
	Size const maxtrials,
  MoverOP const & mover,
	FilterOP const & filter,
	Real const temperature,
	String const sample_type,
  bool const drift ) :
	Super("GenericMonteCarlo"),
	maxtrials_( maxtrials ),
	mover_( mover ),
	filter_( filter ),
	scorefxn_( NULL ),
	temperature_( temperature ),
	sample_type_( sample_type ),
	drift_( drift ),
	last_accepted_pose_( NULL ),
	lowest_score_pose_( NULL )
{
	initialize();
}

/// @brief value constructor
GenericMonteCarloMover::GenericMonteCarloMover(
	Size const maxtrials,
  MoverOP const & mover,
	ScoreFunctionOP const & sfxn,
	Real const temperature,
	String const sample_type,
  bool const drift ) :
	Super("GenericMonteCarlo"),
	maxtrials_( maxtrials ),
	mover_( mover ),
	filter_( NULL ),
	scorefxn_( sfxn ),
	temperature_( temperature ),
	sample_type_( sample_type ),
	drift_( drift )
{
	initialize();
}

/// @brief copy constructor
GenericMonteCarloMover::GenericMonteCarloMover( GenericMonteCarloMover const & rval ):
	Super( rval ),
	maxtrials_( rval.maxtrials_ ),
	mover_( rval.mover_ ),
	temperature_( rval.temperature_ ),
	sample_type_( rval.sample_type_ ),
	drift_( rval.drift_ )
{
	if( rval.filter_ ){
		filter_ =  rval.filter_->clone();
	}
	if( rval.scorefxn_ ){
		scorefxn_ =  rval.scorefxn_->clone();
	}
	initialize();
}

/// @brief destructor
GenericMonteCarloMover::~GenericMonteCarloMover(){}

/// @brief clone this object
GenericMonteCarloMover::MoverOP
GenericMonteCarloMover::clone() const
{
	return new GenericMonteCarloMover( *this );
}

/// @brief create this type of object
GenericMonteCarloMover::MoverOP
GenericMonteCarloMover::fresh_instance() const
{
	return new GenericMonteCarloMover();
}

/// @brief initialize
void
GenericMonteCarloMover::initialize()
{
	if( sample_type_ == "high" ){
		flip_sign_ = -1;
	}else if( sample_type_ == "low" ){
		flip_sign_ = 1;
	}else{
		TR << "WARNING: the sample type, " << sample_type_ << ", is not defined." << std::endl;
		runtime_assert( false );
	}
	trial_counter_ = 0;
	accept_counter_ = 0;
	energy_gap_counter_ = 0.0;
}

/// @brief return the last accepted pose
GenericMonteCarloMover::PoseOP
GenericMonteCarloMover::last_accepted_pose() const
{
	return last_accepted_pose_;
}

/// @brief return the last accepted score
GenericMonteCarloMover::Real
GenericMonteCarloMover::last_accepted_score() const
{
	return last_accepted_score_;
}

/// @brief return the lowest score pose
GenericMonteCarloMover::PoseOP
GenericMonteCarloMover::lowest_score_pose() const
{
	return lowest_score_pose_;
}

/// @brief return the lowest score
GenericMonteCarloMover::Real
GenericMonteCarloMover::lowest_score() const
{
	return lowest_score_;
}

/// @brief return the lowest score
GenericMonteCarloMover::Real
GenericMonteCarloMover::current_score() const
{
	return current_score_;
}

/// @brief return mc_accepted
MCA
GenericMonteCarloMover::mc_accpeted() const
{
	return mc_accepted_;
}

/// @brief set max trials of monte carlo iterations
void
GenericMonteCarloMover::set_maxtrials( Size const ntrial )
{
	maxtrials_ = ntrial;
}

/// @brief set mover
void
GenericMonteCarloMover::set_mover( MoverOP const & mover )
{
	mover_ = mover;
}

/// @brief set filter
/// Pose is evaluated by FilterOP which can do report_sm() or ScoreFunctionOP during MC trials
/// You can choose either way FilterOP or ScoreFunction.
void
GenericMonteCarloMover::set_filter( FilterOP const & filter )
{
	filter_ = filter;
	scorefxn_ = NULL;
}

/// @brief set scorefxn
/// Pose is evaluated by FilterOP which can do report_sm() or ScoreFunctionOP during MC trials
/// You can choose either way FilterOP or ScoreFunction.
void
GenericMonteCarloMover::set_scorefxn( ScoreFunctionOP const & sfxn )
{
	scorefxn_ = sfxn;
	filter_ = NULL;
}

/// @brief set temperatrue
void
GenericMonteCarloMover::set_temperature( Real const temp )
{
	temperature_ = temp;
}

/// @brief set sample type, high or low
/// when sample_type == high, sample pose which have higher value of scorey
/// when sample_type == low, sample pose which have lower value of score
void
GenericMonteCarloMover::set_sampletype( String const & type )
{
	if( sample_type_ != "high" && sample_type_ != "low" ){
		TR << "WARNING !! the sample type, " << type << ", is not defined." << std::endl;
		runtime_assert( false );
	}
	sample_type_ = type;
}

/// @brief if drift=false, the pose is set back to the initial pose
/// Of course, this is not MC sampling.
void
GenericMonteCarloMover::set_drift( bool const drift ){
	drift_ = drift;
}

/// @brief show scores of last_accepted_score and lowest_score
void
GenericMonteCarloMover::show_scores( std::ostream & out ) const
{
	if( sample_type_ == "high" ){
		out << "Higher score sampled: trial=" << I( 5, trial_counter_ ) << ", score(current/last_accepted/best)=";
	}else{
		out << "Lower score sampled: trial=" << I( 5, trial_counter_ ) << ", score(current/last_accepted/best)=";
	}
	out << F( 9, 3, flip_sign_*current_score() ) << '/'
      << F( 9, 3, flip_sign_*last_accepted_score() )<< '/'
			<< F( 9, 3, flip_sign_*lowest_score() ) << std::endl;
	//show_counters( out );
}

/// @brief show counters
void
GenericMonteCarloMover::show_counters( std::ostream & out ) const
{
	String const & mover( mover_->get_name() );
	String evaluation;
	if( filter_ ){
		evaluation = filter_->name();
	}else if( scorefxn_ ){
		evaluation = "ScoreXXX"; // should we have name for ScoreFunction ?
	}
	int  const ntrials( trial_counter_ );
	int  const accepts( accept_counter_ );
	Real const energy_gap( energy_gap_counter_ );
	if( accepts > 0 ){
		out << "mover=" << LJ( 16, mover ) << " Score_eval=" << LJ( 16, evaluation ) <<
			" trials= " << I( 5, ntrials ) << "; " <<
			" accepts= " << F( 6, 3, Real( accepts )/ntrials ) << "; " <<
			" energy_gap/trial= " << F( 8, 3, Real( flip_sign_ * energy_gap ) / ntrials ) << std::endl;
	}else{
		out << "mover=" << A( 16, mover ) << " Score_eval=" << A( 16, evaluation )
				<< " trials= " << I( 6, ntrials ) <<	" NO ACCEPTS." << std::endl;
	}
}

/// @Breif return the simulation state to the lowest energy structure we've seen
void
GenericMonteCarloMover::recover_low( Pose & pose )
{
	if( lowest_score_pose_ ) {
		pose = *lowest_score_pose_;
		*last_accepted_pose_ = *lowest_score_pose_;
	}else{
		// Case of that lowest_score_pose_ was never updated in MonteCarlo
		TR << " No lowest_score_pose_ exists, probably because all movers or filters failed. " << std::endl;
	}
}


/// @brief reset this GenericMonteCarloMover
void
GenericMonteCarloMover::reset( Pose & pose )
{
	lowest_score_ = scoring( pose );
	lowest_score_pose_ = new Pose( pose );
	last_accepted_pose_ = new Pose( pose );
	last_accepted_score_ = lowest_score_;

	trial_counter_ = 0;
	accept_counter_ = 0;
	energy_gap_counter_ = 0;
	TR << "Initialization done " << std::endl;
}

/// @brief score pose based on filter or scorefxn
GenericMonteCarloMover::Real
GenericMonteCarloMover::scoring( Pose & pose )
{
	Real score( 0.0 );
	if( filter_ ){
		score = flip_sign_ * filter_->report_sm( pose );
	}else if( scorefxn_ ){
		score = flip_sign_ * (*scorefxn_)( pose );
	}else{
		TR.Error << "Error, both FilterOP and ScorefunctionOP are defined !!! " << std::endl;
		runtime_assert( false );
	}
	return score;
}

// @brief
bool
GenericMonteCarloMover::boltzmann( Pose & pose )
{
	++trial_counter_;
	Real score = scoring( pose );
	current_score_ = score; // for debugging
	show_scores( TR );
	if ( score > last_accepted_score() ) {
		if( temperature_ < 1e-8 ){
			mc_accepted_ = MCA_rejected; // rejected
		}else{
			Real const boltz_factor = ( last_accepted_score() - score ) / temperature_;
			Real const probability = std::exp( std::min (40.0, std::max(-40.0,boltz_factor)) );
			if ( mc_RG.uniform() >= probability ) {
				mc_accepted_ = MCA_rejected; // rejected
			}else{
				mc_accepted_ = MCA_accepted_thermally; // accepted thermally
			}
		}
	}else{
		mc_accepted_ = MCA_accepted_score_beat_last; // accepted: energy is lower than last_accepted
	}

	if( mc_accepted_ >= 1 ){ // accepted
		++accept_counter_;
		energy_gap_counter_ += score - last_accepted_score();
		*last_accepted_pose_ = pose;
		last_accepted_score_ = score;
		if ( score < lowest_score() ){  // energy is lower than last_accepted
			lowest_score_ = score;
			*lowest_score_pose_ = pose;
			mc_accepted_ = MCA_accepted_score_beat_low; //3;
		}
		return true;
	}else{ //rejected
		return false;
	}
} // boltzmann

/// @Brief
void
GenericMonteCarloMover::apply( Pose & pose )
{
	using protocols::moves::FAIL_DO_NOT_RETRY;
	using protocols::moves::FAIL_BAD_INPUT;
	using protocols::moves::FAIL_RETRY;

	if( !mover_ ){
		TR << "Mover is empty ! " << std::endl;
		return;
	}
	if( !filter_ && !scorefxn_ ){
		TR << "Both ScorefunctionOP and FilterOP are empty ! " << std::endl;
		return;
	}

	PoseOP initial_pose = new Pose( pose );
	bool initialized( false );
	MoverStatus ms( FAIL_RETRY );
	for( Size i=1; i<=maxtrials_; i++ ){
		TR<<"Trial number: "<<i<<std::endl;
		Pose store_pose( pose );
		// Mover apply
		mover_->apply( pose );
		ms = mover_->get_last_move_status();
		if( ms == FAIL_RETRY ){
			TR << "Mover failed. The mover, " << mover_->get_name() << ", is performed again. " << std::endl;
			pose = store_pose;
			continue;
		}else if( ms == FAIL_DO_NOT_RETRY || ms == FAIL_BAD_INPUT ){
			TR << "Mover failed. Exit from GenericMonteCarloMover." << std::endl;
			break;
		}
		// MonteCarlo
		if( ! initialized ){
			reset( pose );
			initialized = true;
		}else{
			if( ! boltzmann( pose ) ){ // evaluate pose by scorefxn_ or filter_.report_sm()
				pose = *last_accepted_pose();
			}
		}
		if( !drift_ ){
			pose = (*initial_pose); // set back pose to initial one, of course this is not way of Monte Carlo
		}
	} // i<=maxtrials_

	// Recover pose that have the lowest score
	recover_low( pose );

}// apply

std::string
GenericMonteCarloMover::get_name() const {
	return GenericMonteCarloMoverCreator::mover_name();
}

/// @brief parse xml file
void
GenericMonteCarloMover::parse_my_tag( TagPtr const tag, DataMap & data, Filters_map const &filters, Movers_map const &movers, Pose const & )
{

	maxtrials_ = tag->getOption< core::Size >( "trials", 10 );
	temperature_ = tag->getOption< Real >( "temperature", 0.0 );

	String const  mover_name( tag->getOption< String >( "mover_name" ));
	String const filter_name( tag->getOption< String >( "filter_name", "true_filter" ) );
 	Movers_map::const_iterator  find_mover ( movers.find( mover_name ));
	Filters_map::const_iterator find_filter( filters.find( filter_name ));
	if( find_mover == movers.end() ) {
		TR << "WARNING !! mover not found in map. skipping:\n" << tag << std::endl;
		runtime_assert( find_mover != movers.end() );
	}
	if( find_filter == filters.end() ) {
		TR << "WARNING !! filter not found in map. skipping:\n" << tag << std::endl;
		runtime_assert( find_filter == filters.end() );
	}
	mover_ = find_mover->second;
	filter_ = find_filter->second->clone();

	String const sfxn ( tag->getOption< String >( "scorefxn_name", "" ) );
	if( sfxn != "" ){
		scorefxn_ = new ScoreFunction( *data.get< ScoreFunction * >( "scorefxns", sfxn ));
		if( filter_name != "true_filter" ){
			TR << "Score evaluation during MC is done by" << sfxn << ", ";
			TR << filter_name << " is ignored." << std::endl;
			filter_ = NULL;
		}
	}else{
		scorefxn_ = NULL;
	}

	if( filter_name == "true_filter" && !scorefxn_ ){
		TR.Error << "You need to set filter_name or scorefxn_name for MC criteria." << std::endl;
		runtime_assert( false );
	}

	if( filter_ ){
		TR << "Apply mover of " << mover_name << ", and evaluate score by " << filter_name
			 << " at Temperature=" << temperature_ << ", ntrails= " << maxtrials_ << std::endl;
	}else{
		TR << "Apply mover of " << mover_name << ", and evaluate score by " << sfxn
			 << " at Temperature=" << temperature_ << ", ntrails= " << maxtrials_ << std::endl;
	}

	drift_ = tag->getOption< bool >( "drift", 1 );
	sample_type_ = tag->getOption< String >( "sample_type", "low" );


	if( !drift_ ){
		TR << "Pose is set back to initial pose every after appling mover and score evalution." << std::endl;
	}

	if( sample_type_ == "high" ){
		TR << "Pose that have higher score is sampled." << sfxn << std::endl;
	}

	//selected_pose_ = data.get< Pose * >( "pose", "selected_pose" );
	initialize();

}

} // ns moves
} // ns protocols
