// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/protocols/moves/GenericMonteCarloMover.hh
/// @brief perform a given mover and sample structures by MonteCarlo
/// @detailed The "score" evaluation of pose during MC after applying mover is done by
/// ither FilterOP that can do report_sm() or ScoreFunctionOP you gave.
/// By setting sample_type_ to high, you can also sample the pose that have higher score.
/// @author Nobuyasu Koga ( nobuyasu@uw.edu )

#ifndef INCLUDED_protocols_moves_GenericMonteCarloMover_HH
#define INCLUDED_protocols_moves_GenericMonteCarloMover_HH

// Unit Headers
#include <protocols/moves/GenericMonteCarloMover.fwd.hh>
#include <protocols/moves/MonteCarloStatus.hh>

// Project Headers
#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>
#include <protocols/filters/Filter.fwd.hh>
#include <protocols/moves/Mover.hh>

// Utility headers
// AUTO-REMOVED #include <utility/vector1.hh>

// Parser headers
#include <protocols/moves/DataMap.fwd.hh>
#include <protocols/moves/Mover.fwd.hh>
// Auto-header: duplicate removed #include <protocols/filters/Filter.fwd.hh>
#include <utility/Tag/Tag.fwd.hh>

//// C++ headers
#include <string>

namespace protocols {
namespace moves {

class GenericMonteCarloMover : public protocols::moves::Mover {
public:


	typedef protocols::moves::Mover Super;
	typedef std::string String;
	typedef core::Size Size;
	typedef core::Real Real;
	typedef core::pose::Pose Pose;
	typedef core::pose::PoseOP PoseOP;
	typedef core::scoring::ScoreFunction ScoreFunction;
	typedef core::scoring::ScoreFunctionOP ScoreFunctionOP;
	typedef protocols::filters::FilterOP FilterOP;
	typedef protocols::moves::MoverOP MoverOP;

	typedef utility::Tag::TagPtr TagPtr;
	typedef protocols::moves::DataMap DataMap;
	typedef protocols::filters::Filters_map Filters_map;
	typedef protocols::moves::Movers_map Movers_map;


public:	// constructor/destructor

	/// @brief default constructor
	GenericMonteCarloMover();

	/// @brief value constructor
	GenericMonteCarloMover(
		Size const maxtrials,
		MoverOP const & mover,
		FilterOP const & filter,
		Real const temperature = 0.0,
		String const sample_type = "min",
		bool const drift = true
	);

	/// @brief value constructor
	GenericMonteCarloMover(
		Size const maxtrials,
		MoverOP const & mover,
		ScoreFunctionOP const & sfxn,
		Real const temperature = 0.0,
		String const sample_type = "min",
		bool const drift = true
	);

	/// @brief copy constructor
	GenericMonteCarloMover( GenericMonteCarloMover const & rval );

	/// @brief destructor
	~GenericMonteCarloMover();

	/// @brief create copy constructor
	virtual MoverOP clone() const;

	/// @brief create this type of objectt
	virtual MoverOP fresh_instance() const;


private: // used in constructor


	/// @brief initialize object used in constructor
	void initialize();


public: // vitural main method


	/// @brief apply GenericMonteCarloMover (Mover)
	virtual void apply( Pose & pose );
	virtual std::string get_name() const;


public:


	/// @brief reset MC iterations
	void reset( Pose & pose );

	/// @breif return the simulation state to the lowest energy structure we've seen
	void recover_low( Pose & pose );


private:


	/// @brief evalute pose by ScoreFunctionOP or FilterOP
	Real scoring( Pose & pose );

	/// @brief core of MC
	bool boltzmann( Pose & pose );


public: // accessor


	/// @brief return the last accepted pose
	PoseOP last_accepted_pose() const;

	/// @brief return the last accepted score
	Real last_accepted_score() const;

	/// @brief return the lowest score pose
	PoseOP lowest_score_pose() const;

	/// @brief return the lowest score
	Real lowest_score() const;

	/// @brief return the lowest score
	Real current_score() const;

	/// @brief return mc_accepted
	MCA mc_accpeted() const;


public: // mutators


	/// @brief set max trials of MC trials
	void set_maxtrials( Size const ntrial );

	/// @brief set mover
	void set_mover( MoverOP const & mover );

	/// @brief Pose is evaluated by FilterOP which can do report_sm() during MC trials
	void set_filter( FilterOP const & filter );

	/// @brief Pose is evaluated by ScoreFunctionOP during MC trials
	void set_scorefxn( ScoreFunctionOP const & sfxn );

	/// @brief set temperature
	void set_temperature( Real const temp );

	/// @brief set sample type, max or min
	/// when sample_type == max, sample pose which have higher score
	/// when sample_type == min, sample pose which have lower score
	void set_sampletype( String const & type );

	/// @brief if drift=false, the pose is set back to the initial pose at each MC trial
	/// Of course, this is not MC sampling.
	void set_drift( bool const drift );


public: // report MC trials


	/// @brief show scores of last_accepted_score and "best_score" ( = flip_sign_ * lowest_score )
	void show_scores( std::ostream & out ) const;

	/// @brief show counters of ntrial and acceptance ratio
	void show_counters( std::ostream & out ) const;


public:// parser


	virtual void parse_my_tag(
		TagPtr const tag,
		DataMap & data,
		Filters_map const & filters,
		Movers_map const & movers,
		Pose const &
	);


private: // data


	/// @brief max number of MC trials
	Size maxtrials_;

	/// @brief mover
	MoverOP mover_;

	/// @brief Pose is evaluated by FilterOP which can do report_sm() during MC trials
	FilterOP filter_;

	/// @brief Pose is evaluated by ScoreFunctionOP during MC trials
	ScoreFunctionOP scorefxn_;

	/// @brief acceptance criterion temperature
	Real temperature_;

	/// @brief set sample type, max or min
	/// when sample_type == max, sample pose which have higher score
	/// when sample_type == min, sample pose which have lower score
	String sample_type_;

	/// @brief if drift=false, the pose is set back to the initial pose at each MC trial
	/// Of course, this is not MC sampling.
	bool drift_;

	/// @brief current score
	Real current_score_;

	/// @brief accepted structure
	Real last_accepted_score_;

	/// @brief lowest energy structure we've seen
	Real lowest_score_;

	/// @brief accepted structure
	PoseOP last_accepted_pose_;

	/// @brief lowest energy structure we've seen
	PoseOP lowest_score_pose_;

	/// @brief result of the last call to boltzmann
	MCA mc_accepted_;

	/// @brief to change the sing of calculated "score"
	Real flip_sign_;

	/// @brief diagnostics
	int trial_counter_;
	int accept_counter_;
	Real energy_gap_counter_;

};

} // namespace moves
} // namespace protocols

#endif

