// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/moves/PyMolMover.hh
/// @brief  Send infromation to PyMol. Contain classes PyMolMover, PyMolObserver and helper classes.
/// @author Sergey Lyskov

#ifndef INCLUDED_protocols_moves_PyMolMover_HH
#define INCLUDED_protocols_moves_PyMolMover_HH

#include <protocols/moves/PyMolMover.fwd.hh>


// Unit Headers
#include <protocols/moves/Mover.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/pose/signals/GeneralEvent.hh>
#include <core/scoring/ScoreType.hh>


#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <string>

namespace protocols {
namespace moves {


/// @brief PyMolMover helper class. Handle low level UDP transactions stuff.
///        This is a port of original Python version of UDP socket client written writen for PyRosetta
class UDPSocketClient
{
public:
	UDPSocketClient();
	~UDPSocketClient();

	void sendMessage(std::string msg);

private:

	void sendRAWMessage(int globalPacketID, int packetI, int packetCount, char * msg_begin, char *msg_end);

	/// @brief last know mximum size of suspenseful sended UDP packet.
	///        ~64k for local connection and ~10k for inet connection
	unsigned int max_packet_size_;  // last know mximum size of suspenseful sended UDP packet.

	/// @brief unique id of this socket client
	union UUID {
		char bytes_[16];
		short unsigned int shorts_[8];
	};

	/// @brief Almost real UUID, but for simplicity we just use random sequence
	UUID uuid_;

	/// @brief counter for number of packet already sent
	int sentCount_;

	/// @brief socket address and handle
	sockaddr_in socket_addr_;
	int socket_h_;
};


class PyMolMover : public protocols::moves::Mover
{
public:
	PyMolMover() : update_energy_(false), energy_type_(core::scoring::total_score),
	               keep_history_(false), update_interval_(0), last_packet_sent_time_(0), name_() {};


	virtual std::string get_name() const;
	virtual void apply( Pose & );


	/// @brief Actually our mover does not change the Pose object, so we have additional const version...
	virtual void apply( Pose const & );


	/// @brief Send specified energy to PyMOL.
	void send_energy(Pose, core::scoring::ScoreType stype = core::scoring::total_score);


	bool update_energy() { return update_energy_; }
	void update_energy(bool f) { update_energy_ = f; }


	core::scoring::ScoreType energy_type(void) { return energy_type_; }
	void energy_type(core::scoring::ScoreType t) { energy_type_ = t; }

	/// @brief Set the keep history flag. If set to True - PyMol will keep track of all frames that
	///        was sent.
	void keep_history(bool kh) { keep_history_ = kh; }


	/// @brief	Return current keep_history flag.
	bool keep_history(void) { return keep_history_; }


	/// @brief	Set current minimum update interval 't' in seconds. Mover will not send anyinformation to
	///         PyMOL before atleast 't' secons has passes since last packet was sent.
	///         default value is 0, - no packets are skipped.
	void update_interval(core::Real t) { update_interval_ = t; };


	/// @brief  Return current update interval.
	core::Real update_interval() { return update_interval_; };

private:
	std::string get_PyMol_model_name(Pose const & pose) const;

	bool is_it_time();

	/// @brief Actual link object
	UDPSocketClient link_;

	bool update_energy_;

	core::scoring::ScoreType energy_type_;

	/// @brief Should PyMol keep history of all models that was sent? - Default is false.
    bool keep_history_;

    /// @brief  Update interval in seconds.
    core::Real update_interval_;

    /// @brief  Time stamp from where last packet was sent.
    core::Real last_packet_sent_time_;

    /// @brief  Name of model in pymol.
    std::string name_;
};



class PyMolObserver : public utility::pointer::ReferenceCount
{
public:
	PyMolObserver() {};
	virtual ~PyMolObserver() {};

	virtual void generalEvent( core::pose::signals::GeneralEvent const & ev) {
		pymol_.apply( *ev.pose );
	};

	PyMolMover & pymol() { return pymol_; };

private:
	PyMolMover pymol_;

//	friend PyMolObserverOP AddPyMolObserver(core::pose::Pose &p, bool keep_history);
};


/// @brief Helper function that create PyMolObserver Object and add it to the give Pose.
///        This is the most likely the only function that you need to call...
PyMolObserverOP AddPyMolObserver(core::pose::Pose &p, bool keep_history=false, core::Real update_interval=0);


} // moves
} // protocols


#endif // INCLUDED_protocols_moves_PyMolMover_HH
