// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file relax_protocols
/// @brief protocols that are specific to relax
/// @detailed
/// @author Mike Tyka, Monica Berrondo

// #include <protocols/jobdist/JobDistributors.hh> // keep first
// #include <protocols/jobdist/Jobs.hh>

#include <protocols/relax/RelaxProtocolBase.hh>

#include <core/pose/Pose.hh>
#include <core/io/pdb/pose_io.hh>

#include <core/util/datacache/BasicDataCache.hh> //pba
#include <core/pose/datacache/CacheableDataType.hh> //pba

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/scoring/constraints/util.hh>
#include <core/scoring/electron_density/util.hh>
#include <protocols/moves/symmetry/SetupForSymmetryMover.hh>
#include <protocols/moves/SuperimposeMover.hh>
#include <protocols/moves/ConstraintSetMover.hh>


#include <core/options/option.hh>
#include <core/options/keys/relax.OptionKeys.gen.hh>
#include <core/options/keys/edensity.OptionKeys.gen.hh>
#include <core/options/keys/constraints.OptionKeys.gen.hh>
#include <core/options/keys/symmetry.OptionKeys.gen.hh>
#include <core/options/keys/in.OptionKeys.gen.hh>

#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/util.hh>

#include <protocols/moves/MoverContainer.hh>
#include <protocols/electron_density/util.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

#ifdef BOINC_GRAPHICS
#include <protocols/boinc/boinc.hh>
#endif

#include <core/util/Tracer.hh>

//Auto Headers
#include <protocols/relax/util.hh>

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer tr("protocols.relax");

using namespace core;
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace protocols {
namespace relax {
////////////////////////////////////////////////////////////////////////////////////////////////////

int
Relax_main( bool ) {
	using namespace protocols::jobdist;
	using namespace core::options;
	using namespace core::options::OptionKeys;

	protocols::moves::MoverOP protocol = generate_relax_from_cmd();
	protocols::jd2::set_native_in_mover( *protocol );

	// add constraints from cmd line
	if ( option[ OptionKeys::constraints::cst_fa_file ].user() ) {
			protocols::moves::SequenceMoverOP seqmov = new protocols::moves::SequenceMover;
			protocols::moves::ConstraintSetMoverOP loadCsts( new protocols::moves::ConstraintSetMover );
			loadCsts->constraint_file( core::scoring::constraints::get_cst_fa_file_option() );
			seqmov->add_mover( loadCsts );
			seqmov->add_mover( protocol );
			protocol = seqmov;
	}

	// set pose for density scoring if a map was input
	// (potentially) dock map into density
	if ( option[ OptionKeys::edensity::mapfile ].user() ) {
		protocols::moves::SequenceMoverOP seqmov = new protocols::moves::SequenceMover;
		seqmov->add_mover( new protocols::electron_density::SetupForDensityScoringMover );
		seqmov->add_mover( protocol );
		protocol = seqmov;
	}

	// setup symmetry mover ... this should happen _before_ SetupForDensityScoringMover
	//   to avoid adding extra VRTs
	if ( option[ OptionKeys::symmetry::symmetry_definition ].user() )  {
			protocols::moves::SequenceMoverOP seqmov = new protocols::moves::SequenceMover;
	    seqmov->add_mover( new protocols::moves::symmetry::SetupForSymmetryMover );
	    seqmov->add_mover( protocol );
	    protocol = seqmov;
	}

	// superimpose input model to the native structure
	if ( option[ OptionKeys::relax::superimpose_to_native ].user() ) {
		if ( option[ OptionKeys::in::file::native ].user() ) {
				core::pose::Pose native;
				core::io::pdb::pose_from_pdb( native, option[ core::options::OptionKeys::in::file::native ] );
				protocols::moves::SequenceMoverOP seqmov = new protocols::moves::SequenceMover;
				protocols::moves::SuperimposeMover *sm  =  new protocols::moves::SuperimposeMover;
				sm->set_reference_pose( native );
				seqmov->add_mover( sm );
				seqmov->add_mover( protocol );
				protocol = seqmov;
		}
	}

	protocols::jd2::JobDistributor::get_instance()->go( protocol );

	return 0;
} 



}
}
