// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/toolbox/TaskOperations/LayerDesignOperation.cc
/// @brief Design residues with selected amino acids depending on the enviroment: layer.
/// The layer of each residue is assigned as core, boundary, or surface, which are defined by
/// accessible surface of mainchain + CB. If resfile is read before calling this operation,
/// this operation is not applied for the residues defined by PIKAA.
/// @author Nobuyasu Koga ( nobuyasu@uw.edu )

//  The following are using amino acids for each layer
/// @CORE
//    Loop: AFILPVWY
//  Strand:  FIL VWY
//   Helix: AFIL VWY ( P only at the beginning of helix )
//
/// @BOUDNARY
//    Loop: ADEGIKLNPQRSTVY
//  Strand:  DE IKLN QRSTVY
//   Helix: ADE IKLN QRSTVY ( P only at the beginning of helix )
//   HelixCapping: DNST
//
/// @SURFACE
//    Loop: DEGHKNPQRST
//  Strand: DE HKN QRST
//   Helix: DE HKN QRST  ( P only at the beginning of helix )
//   HelixCapping: DNST


// Unit Headers
#include <protocols/toolbox/TaskOperations/LayerDesignOperation.hh>
#include <protocols/toolbox/TaskOperations/LayerDesignOperationCreator.hh>

// Project Headers
#include <core/id/AtomID_Map.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/pose/Pose.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/PackerTask_.hh>
#include <core/scoring/dssp/Dssp.hh>
#include <core/scoring/sasa.hh>
#include <core/options/option.hh>
#include <core/util/Tracer.hh>
#include <protocols/flxbb/SelectResiduesByLayer.hh>


// Utility Headers
#include <utility/string_util.hh>
#include <utility/Tag/Tag.hh>
#include <utility/vector1.hh>
#include <string>

using core::util::T;
using core::util::Error;
using core::util::Warning;


//Auto Headers
#include <ObjexxFCL/format.hh>

using namespace core::options;
using namespace core::options::OptionKeys;
using namespace core;

static core::util::Tracer TR("protocols.toolbox.TaskOperations.LayerDesignOperation");

namespace protocols{
namespace toolbox{
namespace TaskOperations{

core::pack::task::operation::TaskOperationOP
LayerDesignOperationCreator::create_task_operation() const
{
	return new LayerDesignOperation;
}


/// @brief default constructor
LayerDesignOperation::LayerDesignOperation():
	TaskOperation(),
	dsgn_core_( true ),	dsgn_boundary_( true ),	dsgn_surface_( true ),
	add_helix_capping_( true ),
	use_original_( false ),
	pore_radius_( 2.0 ), burial_( 20.0 ), surface_( 40.0 )
{
	TR << " pore_radius : " <<  pore_radius_ << std::endl;
	TR << " burial : " << burial_ << std::endl;
	TR << " surface : " << surface_ << std::endl;
}


/// @brief value constructor
LayerDesignOperation::LayerDesignOperation( bool dsgn_core, bool dsgn_boundary, bool dsgn_surface ):
	TaskOperation(),
	dsgn_core_( dsgn_core ), dsgn_boundary_( dsgn_boundary ),	dsgn_surface_( dsgn_surface ),
	use_original_( false ),
	pore_radius_( 2.0 ), burial_( 20.0 ), surface_( 40.0 )
{
	TR << " pore_radius : " <<  pore_radius_ << std::endl;
	TR << " burial : " << burial_ << std::endl;
	TR << " surface : " << surface_ << std::endl;
}

/// @brief destructor
LayerDesignOperation::~LayerDesignOperation() {}

/// @brief clone
core::pack::task::operation::TaskOperationOP
LayerDesignOperation::clone() const {
	return new LayerDesignOperation( *this );
}

/// @brief apply
void
LayerDesignOperation::apply( Pose const & pose, PackerTask & task ) const
{
	using protocols::flxbb::SelectResiduesByLayer;

	// define the layer each residue belong to
	SelectResiduesByLayer srbl(	dsgn_core_,	dsgn_boundary_,	dsgn_surface_ );
	srbl.compute( pose );

	// calc dssp
	core::scoring::dssp::Dssp dssp( pose );
	dssp.dssp_reduced();

	// find the position of residues of helix capping and intial residue of helix
	bool flag( false );
	utility::vector1< bool > helix_capping( pose.total_residue(), false );
	utility::vector1< bool > initial_helix( pose.total_residue(), false );
	for( Size i=1; i<=pose.total_residue(); ++i ){
		char ss( dssp.get_dssp_secstruct( i ) );
		if( ss == 'H' && flag == false && i != 1 ){
			initial_helix[ i ] = true;
			helix_capping[ i-1 ] = true;
			flag = true;
		}
		if( ss != 'H' && flag == true ){
			flag = false;
		}
	}

	// terminal residues set to be allaa
	utility::vector1<bool> restrict_to_aa( 20, true );
	task.nonconst_residue_task( 1 ).restrict_absent_canonical_aas( restrict_to_aa );
	task.nonconst_residue_task( pose.total_residue() ).restrict_absent_canonical_aas( restrict_to_aa );

	for( Size i=2; i<=pose.total_residue()-1; ++i ) {

		char ss( dssp.get_dssp_secstruct( i ) );
		TR << "Resnum=" << i << " ,SS=" << ss << " "
       << "Sasa=" << ObjexxFCL::fmt::F( 6, 2, srbl.rsd_sasa( i ) );

		// skip the residue if this position is defined as PIKAA by resfile
		if( task.residue_task( i ).command_string().find( "PIKAA" ) != std::string::npos ){
			TR << " ,Resfile info is used." << std::endl;
			continue;
		}

		if( srbl.layer( i ) == "core" && dsgn_core_ ) {

			utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, false );
			restrict_to_aa[chemical::aa_from_name( "TRP" )] = true;
			restrict_to_aa[chemical::aa_from_name( "PHE" )] = true;
			restrict_to_aa[chemical::aa_from_name( "ILE" )] = true;
			restrict_to_aa[chemical::aa_from_name( "LEU" )] = true;
			restrict_to_aa[chemical::aa_from_name( "VAL" )] = true;
			restrict_to_aa[chemical::aa_from_name( "TYR" )] = true;
			restrict_to_aa[chemical::aa_from_name( "MET" )] = true;

			if( ss == 'E' ) {

			} else if( ss == 'H' ) {
				restrict_to_aa[chemical::aa_from_name( "ALA" )] = true;
				if( initial_helix[ i ] ) {
					restrict_to_aa[chemical::aa_from_name( "PRO" )] = true;
				}
			} else {
				restrict_to_aa[chemical::aa_from_name( "ALA" )] = true;
				restrict_to_aa[chemical::aa_from_name( "PRO" )] = true;
			}
			task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
			TR << " " << srbl.layer( i ) << std::endl;

		} else if ( srbl.layer( i ) == "boundary" && dsgn_boundary_ ) {

			if( helix_capping[ i ] == true && add_helix_capping_ ) {
				utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, false );
				restrict_to_aa[chemical::aa_from_name( "ASP" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ASN" )] = true;
				restrict_to_aa[chemical::aa_from_name( "THR" )] = true;
				restrict_to_aa[chemical::aa_from_name( "SER" )] = true;
				task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
				TR << " ,Helix Capping " << std::endl;
			} else {

				utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, true );
				restrict_to_aa[chemical::aa_from_name( "CYS" )] = false;
				restrict_to_aa[chemical::aa_from_name( "PHE" )] = false;
				restrict_to_aa[chemical::aa_from_name( "TRP" )] = false;
				restrict_to_aa[chemical::aa_from_name( "MET" )] = false;
				restrict_to_aa[chemical::aa_from_name( "HIS" )] = false;

				if( ss == 'E' ){
					restrict_to_aa[chemical::aa_from_name( "GLY" )] = false;
					restrict_to_aa[chemical::aa_from_name( "ALA" )] = false;
					restrict_to_aa[chemical::aa_from_name( "PRO" )] = false;
				} else if( ss == 'H' ) {
					if( ! initial_helix[ i ] ) {
						restrict_to_aa[chemical::aa_from_name( "PRO" )] = false;
					}
					restrict_to_aa[chemical::aa_from_name( "GLY" )] = false;
				}
				task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
				TR << " " << srbl.layer( i ) << std::endl;
			}

		} else if ( srbl.layer( i ) == "surface" && dsgn_surface_  ) {

			if( helix_capping[ i ] == true && add_helix_capping_ ) {
				utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, false );
				restrict_to_aa[chemical::aa_from_name( "ASP" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ASN" )] = true;
				restrict_to_aa[chemical::aa_from_name( "THR" )] = true;
				restrict_to_aa[chemical::aa_from_name( "SER" )] = true;
				task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
				TR << " ,Helix Capping " << std::endl;
			} else {

				utility::vector1<bool> restrict_to_aa( chemical::num_canonical_aas, false );
				restrict_to_aa[chemical::aa_from_name( "GLU" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ARG" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ASP" )] = true;
				restrict_to_aa[chemical::aa_from_name( "LYS" )] = true;
				restrict_to_aa[chemical::aa_from_name( "HIS" )] = true;
				restrict_to_aa[chemical::aa_from_name( "ASN" )] = true;
				restrict_to_aa[chemical::aa_from_name( "GLN" )] = true;
				restrict_to_aa[chemical::aa_from_name( "SER" )] = true;
				restrict_to_aa[chemical::aa_from_name( "THR" )] = true;

				if( ss == 'E' ) {
				} else if ( ss == 'H' ) {
					if( initial_helix[ i ] == true ){
						restrict_to_aa[chemical::aa_from_name( "PRO" )] = true;
					}
				} else {
					restrict_to_aa[chemical::aa_from_name( "GLY" )] = true;
				}

				task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
				TR << " " << srbl.layer( i ) << std::endl;
			}

		} else {

			if( use_original_ ){
				task.nonconst_residue_task( i ).restrict_to_repacking();
				TR << " ,Original sequence used" << std::endl;
			}else{
				utility::vector1<bool> restrict_to_aa( 20, false );
				restrict_to_aa[chemical::aa_from_name( "ALA" )] = true;
				task.nonconst_residue_task( i ).restrict_absent_canonical_aas( restrict_to_aa );
				TR << " ,No design" << std::endl;
			}

		}

	} // for( i )
} // apply

void
LayerDesignOperation::parse_tag( TagPtr tag )
{

	String design_layers = tag->getOption< String >( "layer" );
	utility::vector1< String > layers( utility::string_split( design_layers, '_' ) );

	dsgn_core_ = false;
	dsgn_surface_ = false;
	dsgn_boundary_ = false;
	for ( utility::vector1< String >::const_iterator iter = layers.begin(); iter != layers.end() ; ++iter) {
		String layer(*iter);

		if ( layer == "core" ) {
			dsgn_core_ = true;
		} else if ( layer == "surface" ) {
			dsgn_surface_ = true;
		} else if ( layer == "boundary" ) {
			dsgn_boundary_ = true;
		} else {
			TR << "Error!, wrong specification of layer_mode " << layer << std::endl;
			TR << "Every layers are designed. " << std::endl;
			dsgn_core_ = true;
			dsgn_surface_ = true;
			dsgn_boundary_ = true;
		}
	}

	use_original_ = tag->getOption< bool >( "use_original_non_designed_layer", 0 );
	pore_radius_ = tag->getOption< Real >( "pore_radius", 2.0 );
	burial_ = tag->getOption< Real >( "burial", 20.0 );
	surface_ = tag->getOption< Real >( "surface", 40.0 );

}



} // TaskOperations
} // toolbox
} // protocols

