// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

//////////////////////////////////////////////////////////////////////////////////////////////////
/// @file rdwizard/bindings/authorize.cc
/// @brief gksu/kdesu wrapper for OS X and Windows Vista
///
/// @detailed
/// This is an OS X and Windows wrapper for gksu/gksudo/kdesu.  This should compile to a simple
/// program that, when invoked with at least some argv[1], attempts to request administrator
/// execution priveleges using the target system's standard API (Authenticate Dialog on OS X,
/// UAC on Windows), and, if successful, executes argv[1] with arguments argv[2,...].
///
/// Exits with status -1 if insufficient arguments were provided; -2 if unable to gain admin
/// priveleges; -3 if a problem occurs synchronizing with the child process; and with the exit
/// status of the called binary otherwise (ideally, 0.)
///
/// @author Jon Latane (jonlatane@gmail.com)
//////////////////////////////////////////////////////////////////////////////////////////////////

#define BADARGS -1
#define NOADMIN -2
#define BADSUBPROCESS -3

#ifdef OSX
#include <Security/Authorization.h>
#include <Security/AuthorizationTags.h>
#include <stdio.h>
#include <sys/wait.h>

int main( int argc, char* argv[] )
{
	// Check arguments
	if( argc < 2 )
		return BADARGS;

	// Initialize our AuthorizationRef
	AuthorizationRef authRef;
	OSStatus status = AuthorizationCreate( NULL,
																				 kAuthorizationEmptyEnvironment,
																				 kAuthorizationFlagDefaults,
																				 &authRef );

	if( status != errAuthorizationSuccess )
		return NOADMIN;

	// Structures containing rights we'll be requesting
	AuthorizationItem items = { kAuthorizationRightExecute, 0, NULL, 0 };
	AuthorizationRights rights = { 1, &items };
	AuthorizationFlags flags = kAuthorizationFlagDefaults |
	                           kAuthorizationFlagInteractionAllowed |
	                           kAuthorizationFlagPreAuthorize |
	                           kAuthorizationFlagExtendRights;

	status = AuthorizationCopyRights( authRef, &rights, NULL, flags, NULL );

	if( status != errAuthorizationSuccess )
		return NOADMIN;

	status = AuthorizationExecuteWithPrivileges( authRef,
	                                             argv[1],
	                                             kAuthorizationFlagDefaults,
	                                             &argv[2],
																							 NULL );
	if( status != errAuthorizationSuccess )
		return NOADMIN;

	// Wait for the child process to complete and get its exit status.
	int exitInfo;
	int pid = wait( &exitInfo );

	// Verify that child process exited properly
	if ( pid == -1 || !WIFEXITED(exitInfo) )
		return BADSUBPROCESS;

	// Free up the authorization
	AuthorizationFree( authRef, kAuthorizationFlagDefaults );

	// We've managed to run successfully.  Return the same exit status as the child process.
	return WEXITSTATUS( exitInfo );
}

#elif defined WINDOWS
int main( int argc, char* argv[] )
{
  return NOADMIN;
}
#else
int main( int argc, char* argv[] )
{
  return NOADMIN;
}
#endif
