// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   test/core/scoring/methods/P_AA_Energy.cxxtest.hh
/// @brief  test suite for core::scoring::P_AA_Energy.cc
/// @author Ron Jacak

// Test headers
#include <cxxtest/TestSuite.h>

// Unit headers
#include <core/scoring/methods/UnfoldedStateEnergy.hh>

#include "platform/types.hh"

// Package Headers
#include <test/core/init_util.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/EnergyMap.hh>

#include <core/chemical/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/options/option.hh>
#include <core/chemical/ResidueTypeSet.hh>

//Auto Headers
#include <core/chemical/VariantType.hh>
#include <core/id/DOF_ID.hh>
#include <core/id/NamedStubID.hh>
#include <core/pose/signals/ConformationEvent.hh>
#include <core/pose/signals/DestructionEvent.hh>
#include <core/pose/signals/EnergyEvent.hh>
#include <core/scoring/ScoreFunctionInfo.fwd.hh>
#include <core/scoring/types.hh>
#include <core/scoring/constraints/Constraints.fwd.hh>
#include <core/util/Tracer.fwd.hh>
#include <utility/keys/Key2Tuple.hh>
#include <ObjexxFCL/FArray.fwd.hh>



// --------------- Test Class --------------- //

// using declarations
using namespace core;
using namespace core::pose;
using namespace core::scoring;
using namespace core::scoring::methods;

class UnfoldedStateEnergyTests : public CxxTest::TestSuite {

	public:

	PoseOP the_pose;
	Real TOLERATED_ERROR;

	// --------------- Fixtures --------------- //

	// Define a test fixture (some initial state that several tests share)
	// In CxxTest, setUp()/tearDown() are executed around each test case. If you need a fixture on the test
	// suite level, i.e. something that gets constructed once before all the tests in the test suite are run,
	// suites have to be dynamically created. See CxxTest sample directory for example.

	// Shared initialization goes here.
	void setUp() {

		TOLERATED_ERROR = 0.001;
		core_init();

		the_pose = new Pose;
		core::chemical::ResidueTypeSetCAP rsd_set = chemical::ChemicalManager::get_instance()->residue_type_set( "fa_standard" );
		core::chemical::make_pose_from_sequence( *the_pose, "DFGLK", *rsd_set );

	}

	// Shared finalization goes here.
	void tearDown() {
		the_pose = 0;
	}


	// --------------- Test Cases --------------- //

	// Test four different ways of using the UnfoldedStateEnergy energy method class. The difference lies in how the class
	// is constructed: with the default constructor, with an empty EMap, an full EMap or with a vector.

	void test_residue_energy_default_file_weights() {
		// AA fa_atr fa_rep fa_sol fa_intra_rep pro_close fa_pair hbond_sr_bb hbond_lr_bb hbond_bb_sc hbond_sc rama omega fa_dun p_aa_pp
		// WEIGHT 0.8 0.44 0.65 0.004 1.0 0.49  0.585 1.17 1.17 1.1 0.2 0.5 0.56 0.32
		// ALA -2.5587 0.57372 1.56747 0.22524 0.00490 0.00000 -0.5280 -0.0270 -0.0057 0.00000 -0.2123 0.14908 0.00000 -0.1565
		// ARG -3.1829 0.61819 2.22200 2.10321 0.00502 -0.0866 -0.4856 -0.0168 -0.0206 -0.0125 -0.2571 0.17437 2.32912 -0.0972
		// ASN -2.8712 0.63352 2.40500 1.66277 0.00779 -0.0763 -0.3850 -0.0174 -0.0311 -0.0057 0.03004 0.18038 1.17435 -0.2480
		// ASP -3.0032 0.67573 2.45898 1.49946 0.00499 -0.1044 -0.4009 -0.0166 -0.0416 -0.0125 -0.0134 0.16177 0.87325 -0.1915
		// CYS -2.5378 0.62453 1.70147 0.39955 0.00601 0.00000 -0.3914 -0.0316 -0.0069 0.00000 0.02147 0.24108 0.54994 0.12551
		// GLN -3.1503 0.67242 2.23849 1.55477 0.00559 -0.0530 -0.4965 -0.0110 -0.0183 -0.0037 -0.2326 0.16813 1.72092 -0.0651
		// GLU -3.0684 0.62899 2.17359 1.55916 0.00278 -0.1429 -0.4797 -0.0117 -0.0147 -0.0094 -0.4136 0.14356 1.72345 -0.1438
		// GLY -1.6906 0.48787 1.20086 0.01814 0.00333 0.00000 -0.3093 -0.0139 -0.0093 0.00000 0.65941 0.14735 0.00000 -1.0637
		// HIS -3.5341 0.69800 2.35095 0.99401 0.00405 -0.0199 -0.4122 -0.0172 -0.0229 -0.0064 -0.0335 0.21329 1.36355 -0.0844
		// ILE -2.9308 0.61438 1.57883 4.73755 0.00701 0.00000 -0.4375 -0.0224 -0.0037 0.00000 -0.1999 0.16784 0.48838 -0.3580
		// LEU -3.2405 0.65432 1.77459 2.05196 0.00486 0.00000 -0.5425 -0.0343 -0.0046 0.00000 -0.1254 0.17174 0.61234 -0.1728
		// LYS -2.8124 0.65897 1.81361 1.07980 0.00474 -0.1160 -0.4479 -0.0148 -0.0091 -0.0012 -0.2868 0.15867 1.53033 -0.0155
		// MET -3.2688 0.67819 1.86497 1.11619 0.00630 0.00000 -0.5336 -0.0273 -0.0050 0.00000 -0.1043 0.17163 1.91699 -0.0656
		// PHE -3.6792 0.72906 1.73395 6.68027 0.00525 0.00000 -0.4257 -0.0216 -0.0055 0.00000 -0.1227 0.21814 1.03248 -0.1014

		UnfoldedStateEnergyOP unweighted_unf_energy = new UnfoldedStateEnergy( scoring::UNFOLDED_SCORE12 );

		EnergyMap emap;
		emap.zero();

		unweighted_unf_energy->residue_energy( the_pose->residue(2), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], -1.11401, TOLERATED_ERROR ); // value checked by hand by P. Douglas Renfrew (01/08/2010)

		emap.zero();
		unweighted_unf_energy->residue_energy( the_pose->residue(3), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], -0.69677, TOLERATED_ERROR ); // value checked by hand by P. Douglas Renfrew (01/08/2010)

		unweighted_unf_energy = 0;

	}

	void test_residue_energy_no_weights() {

		EnergyMap emap; // same emap used for initalization and holding scores
		emap.zero();

		UnfoldedStateEnergyOP unweighted_unf_energy = new UnfoldedStateEnergy( scoring::UNFOLDED_SCORE12, emap );

		unweighted_unf_energy->residue_energy( the_pose->residue(2), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.0, TOLERATED_ERROR );

		emap.zero();
		unweighted_unf_energy->residue_energy( the_pose->residue(3), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.0, TOLERATED_ERROR );

		unweighted_unf_energy = 0;
	}

	void test_residue_energy_emap_weights() {

		EnergyMap emap;
		emap[ fa_atr ] = 0.8;
		emap[ fa_rep ] = 0.454238;
		emap[ fa_sol ] = 1.04457;
		emap[ fa_intra_rep ] = 0.223082;
		emap[ pro_close ] = 0.374325;
		emap[ fa_pair ] = 0.743825;
		emap[ hbond_sr_bb ] = 0.585;
		emap[ hbond_lr_bb ] = 1.17;
		emap[ hbond_bb_sc ] = 2.58097;
		emap[ hbond_sc ] = 0.477475;
		emap[ dslf_ss_dst ] = 1;
		emap[ dslf_cs_ang ] = 1;
		emap[ dslf_ss_dih ] = 1;
		emap[ dslf_ca_dih ] = 1;
		emap[ rama ] = 0.2;
		emap[ omega ] = 0.151806;
		emap[ fa_dun ] = 0.358969;
		emap[ p_aa_pp ] = 1.01463;
		emap[ unfolded ] = 0.61846;
		emap[ ref ] = 0.0;

		UnfoldedStateEnergyOP weighted_unf_energy_emap = new UnfoldedStateEnergy( scoring::UNFOLDED_SCORE12, emap );

		// atr  rep  sol  intra_rep  pro  pair  hb_sr_bb  hb_lr_bb  hb_bb_sc  hb_sc  rama  omega  dun  paapp
		// GLY -1.6933 0.48918 1.20216 0.01788 0.00784 0.00000 0.00000 0.00000 -0.0085 0.00000 0.61547 0.22644 0.00000 -1.0623
		// LEU -3.1780 0.71276 1.75359 2.00079 0.01865 0.00000 0.00000 0.00000 -0.0044 0.00000 -0.1056 0.20147 0.60841 -0.1721
		// values below verified using Excel spreadsheet for correctness
		emap.zero();
		weighted_unf_energy_emap->residue_energy( the_pose->residue(4), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], -0.3058, TOLERATED_ERROR );

		emap.zero();
		weighted_unf_energy_emap->residue_energy( the_pose->residue(5), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.2971, TOLERATED_ERROR );

		weighted_unf_energy_emap = 0;
	}

	// prefix'ing the name of a test with the letter 'x' disables that test (without having to comment it out)
	void xtest_residue_energy_vector_weights() {

		UnfoldedStateEnergyOP weighted_unf_energy_vector;

		Real weights[] = { 0.8, 0.454238, 1.04457, 0, 0.223082, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0.374325, 0, 0, 0, 0, 0, 0.743825, 0,
											 0, 0, 0, 0.585, 1.17, 2.58097, 0, 0, 0.477475, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
											 1, 0, 0, 0, 0, 0, 0, 0, 0.2, 0.151806,
											 0.358969,1.01463, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
											 0.61846, 0 };

		const size_t size = sizeof(weights) / sizeof (weights[0]);
		utility::vector1< Real > v(weights, weights + size);

		weighted_unf_energy_vector = new UnfoldedStateEnergy( scoring::UNFOLDED_SCORE12, v );

		EnergyMap emap;
		emap.zero();
		weighted_unf_energy_vector->residue_energy( the_pose->residue(1), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.8143, TOLERATED_ERROR );

		emap.zero();
		weighted_unf_energy_vector->residue_energy( the_pose->residue(4), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.1083, TOLERATED_ERROR );

		emap.zero();
		weighted_unf_energy_vector->residue_energy( the_pose->residue(5), *the_pose, emap );
		TS_ASSERT_DELTA( emap[ unfolded ], 0.5965, TOLERATED_ERROR );

		weighted_unf_energy_vector = 0;
	}

};


