// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   protocols/moves/SmallMover.cxxtest.hh
/// @brief  test suite for protocols::moves::SmallMover.cc
/// @author Monica Berrondo


// Test headers
#include <cxxtest/TestSuite.h>

#include <test/core/init_util.hh>
#include <test/util/pose_funcs.hh>

#include <core/chemical/ResidueTypeSet.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/conformation/Residue.hh>
#include <core/pose/Pose.hh>
#include <core/types.hh>

// Unit headers
#include <protocols/moves/BackboneMover.hh>

// Package headers
#include <core/io/pdb/pose_io.hh>
#include <core/kinematics/MoveMap.hh>

#include <core/util/Tracer.hh>

//Auto Headers
#include <core/chemical/AtomType.hh>
#include <core/chemical/AtomTypeSet.hh>
#include <core/chemical/VariantType.hh>
#include <core/id/AtomID_Map.hh>
#include <core/id/AtomID_Mask.hh>
#include <core/id/DOF_ID_Map.hh>
#include <core/id/DOF_ID_Mask.hh>
#include <core/id/NamedStubID.hh>
#include <core/io/pdb/file_data.hh>
#include <core/pose/signals/ConformationEvent.hh>
#include <core/pose/signals/DestructionEvent.hh>
#include <core/pose/signals/EnergyEvent.hh>
#include <core/scoring/ScoreFunctionInfo.fwd.hh>
#include <core/scoring/constraints/Constraints.fwd.hh>
#include <core/util/OStream.hh>
#include <protocols/moves/MoverStatistics.hh>
#include <utility/keys/Key2Tuple.hh>
#include <ObjexxFCL/format.hh>
#include <utility>

static core::util::Tracer TR("protocols.moves.SmallMover.cxxtest");

// using declarations
using namespace core;
using namespace core::pose;
using namespace protocols::moves;

///////////////////////////////////////////////////////////////////////////
/// @name SmallMoverTest
/// @brief: test a single small move using all default values
/// @detailed Default values used are 0, 5, 6 (H,E,L)
///						nmoves=1, all residues movable
///  					The values for the new phi and psi torsion angles
///						were checked against values calculated by hand using
///						4 significant figures for the calculations.
///						Residue number 77 is the residue that is picked
///						The new phi angle is -73.1381 (assuming only L is being used)
///						The new psi angle is -26.9266
///
/// @author Monica Berrondo October 09 2007
///////////////////////////////////////////////////////////////////////////
class SmallMoverTest : public CxxTest::TestSuite {

public:
	chemical::ResidueTypeSetCAP residue_set;

	PoseOP the_pose;
	SmallMoverTest() {}

	void setUp() {
		core_init();
		residue_set = chemical::ChemicalManager::get_instance()->residue_type_set( chemical::FA_STANDARD );

		the_pose = new Pose;
		io::pdb::pose_from_pdb( *the_pose, "protocols/moves/test_in.pdb" );

		core::init_random_generators(1000, numeric::random::_RND_TestRun_, "mt19937");
	}

	void tearDown() {
		the_pose = 0;
	}

	void test_OneSmallMover() {
		SmallMover mover; // create a default small mover
		core::Real correct_phi ( -57.0946 );
		core::Real correct_psi ( -57.1872 );
		// make the move
		mover.apply( *the_pose );

		core::Real phi = mover.new_phi();
		core::Real psi = mover.new_psi();

		// compare to the correct answer
		float const TOLERATED_ERROR = 0.0001;

		TS_ASSERT_DELTA( phi, correct_phi, TOLERATED_ERROR );
		TS_ASSERT_DELTA( psi, correct_psi, TOLERATED_ERROR );
		TR << "test_OneSmallMover completed!! " << std::endl;
	}

	void test_SmallMover() {
		// this is assuming that the residue is of type L so that the max_angle is 6.0
		const int size = 6; ///< size of array to store numbers
		const int N = 1000; ///< number of trials
		const int resnum = 77; ///< residue number to change phis and psis
		// create array and fill it
		std::vector<int> phi_V(size), psi_V(size);
		for ( unsigned int i=0; i<phi_V.size(); ++i )  {
			phi_V[i] = 0;
			psi_V[i] = 0;
		}

		// set up the mover to only change on residue and loop 100 times
		SmallMover mover; // create a default small mover
		core::kinematics::MoveMapOP movemap( new core::kinematics::MoveMap );
		movemap->set_bb( false );
		movemap->set_bb( resnum, true );
		mover.movemap( movemap );

		// make an initial move and get the phi and psis
		mover.apply( *the_pose );
		core::Real phi = mover.new_phi();
		core::Real psi = mover.new_psi();
		// store the phi and psi as the original ones to go back to
		core::Real orig_phi = phi;
		core::Real orig_psi = psi;

		// make the move 100 times
		for ( int i=0; i<N; ++i ) {
			mover.apply( *the_pose );
			phi = mover.new_phi();
			psi = mover.new_psi();
			int phi_ind = int( phi - orig_phi + int(size/2) );
			int psi_ind = int( psi - orig_psi + int(size/2) );
			phi_V[phi_ind] += 1;
			psi_V[psi_ind] += 1;

			the_pose->set_phi( resnum, orig_phi );
			the_pose->set_psi( resnum, orig_psi );
		}

		for ( unsigned int i=0; i<phi_V.size(); ++i )  {
			TR << "phi of " << i << " : " << phi_V[i];
			TR << " / psi of " << i << " : " << psi_V[i] << std::endl;
		}

		int min_c_phi = N;
		int max_c_phi = 0;
		int min_c_psi = N;
		int max_c_psi = 0;
		for(unsigned int i=0; i<phi_V.size(); i++) {
			if( min_c_phi > phi_V[i] ) min_c_phi = phi_V[i];
			if( max_c_phi < phi_V[i] ) max_c_phi = phi_V[i];
			if( min_c_psi > psi_V[i] ) min_c_psi = psi_V[i];
			if( max_c_psi < psi_V[i] ) max_c_psi = psi_V[i];
		}
		double phi_rate = double(min_c_phi)/max_c_phi;
		TR << min_c_phi << " ";
		TR << max_c_phi << " ";
		TR << phi_rate << " ";
		TS_ASSERT_LESS_THAN(0.6, phi_rate);
		double psi_rate = double(min_c_psi)/max_c_psi;
		TR << min_c_psi << " ";
		TR << max_c_psi << " ";
		TR << psi_rate << " ";
		TS_ASSERT_LESS_THAN(0.6, psi_rate);
		TR << "test_SmallMover completed!! " << std::endl;
	}

	void test_comprehensiveSmallMover() {
		core::pose::Pose initial_pose;
		initial_pose = *the_pose;
		SmallMover mover; // create a default small mover
		// make the move
		mover.apply( *the_pose );

		// make sure that only the residues before the move location are the same and only the residue
		// after the move point have changed
		// I'm not sure this is the right way to do this...  Monica Berrondo
		for ( unsigned int i=1; i<=the_pose->total_residue(); ++i ) {
			// if the residue number is less than or equal to that which gets changed, the coordinates should be the same,
			// within some tolerance (0.005).
			if ( i <= 27 ) {
				TS_ASSERT_DELTA( the_pose->residue(i).atom("CA").xyz()[0], initial_pose.residue(i).atom("CA").xyz()[0], 0.005 );
				TS_ASSERT_DELTA( the_pose->residue(i).atom("CA").xyz()[1], initial_pose.residue(i).atom("CA").xyz()[1], 0.005 );
				TS_ASSERT_DELTA( the_pose->residue(i).atom("CA").xyz()[2], initial_pose.residue(i).atom("CA").xyz()[2], 0.005 );
			// if the residue number is greater than that which gets changed, the coordinates should differ
			} else {
				TS_ASSERT_DIFFERS( the_pose->residue(i).atom("CA").xyz()[0], initial_pose.residue(i).atom("CA").xyz()[0] );
				TS_ASSERT_DIFFERS( the_pose->residue(i).atom("CA").xyz()[1], initial_pose.residue(i).atom("CA").xyz()[1] );
				TS_ASSERT_DIFFERS( the_pose->residue(i).atom("CA").xyz()[2], initial_pose.residue(i).atom("CA").xyz()[2] );
			}
		}
		TR << "test_comprehensiveSmallMover completed!! " << std::endl;
	}
};

