// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.


//Headers are generally organized by either what they do or where they come from.  This organization is first core library headers, then protocols library, then utility stuff.


// Project Headers
#include <core/pose/Pose.hh>
#include <core/io/pdb/pose_io.hh>

#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskOperation.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/kinematics/FoldTree.hh>
#include <core/kinematics/MoveMap.hh>

// Mover headers
#include <protocols/moves/MinMover.hh>
#include <protocols/moves/MoverContainer.hh>
#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/moves/RotamerTrialsMover.hh>
#include <protocols/moves/TrialMover.hh>
#include <protocols/moves/ReturnSidechainMover.hh>
#include <protocols/moves/TaskAwareMinMover.hh>
#include <protocols/moves/MonteCarlo.hh>
#include <protocols/moves/RepeatMover.hh>
#include <protocols/moves/BackboneMover.hh>
#include <protocols/loops/CcdLoopClosureMover.hh>

#include <protocols/loops/LoopClass.hh>

#include <protocols/geometry/RB_geometry.hh>
#include <protocols/docking/DockingProtocol.hh>

#include <protocols/toolbox/TaskOperations/RestrictToInterfaceOperation.hh>

// Utility Headers
#include <core/init.hh>
#include <core/options/util.hh>
#include <core/options/option.hh>
#include <core/util/tracer.hh>
#include <utility/exit.hh>

// C++ headers
#include <string>
#include <sstream>

//The original author used a lot of using declarations here.  This is a stylistic choice.
// Namespaces
using namespace core;
using namespace conformation;
using namespace chemical;
using namespace scoring;
using namespace pose;
using namespace protocols;
using namespace protocols::moves;
using namespace protocols::toolbox;
using namespace core::pack::task;
using namespace core::options;
using namespace core::options::OptionKeys;
using namespace core::id;
using core::util::T;
using core::util::Error;
using core::util::Warning;
using utility::file::FileName;

// Here is a sketch of the basic flow of the program...
//
// Pertubation Phase
//   +-Monte Carlo Mover---------------------------------------+
//   | +-Random Mover-( 1 / 2 / 1 / 1 )--------------------+ | |
//   | | +-Docking Mover-----------------------------------+ | |
//   | | | small rigid body movements between the peptide	 | | |
//   | | | and protein for conformational diversity        | | |
//   | | +-------------------------------------------------+ | |
//   | | +-Loop Modeling-----------------------------------+ | |
//   | | | perturb and close one of loops to generate      | | |
//   | | | conformational diversity                        | | |
//   | | +-------------------------------------------------+ | |
//   | | +-Termini Modeling--------------------------------+ | |
//   | | | move termini with small/shear moves to generate | | |
//   | | | conformational diversity                        | | |
//   | | +-------------------------------------------------+ | |
//   | | +-Peptide Modeling--------------------------------+ | |
//   | | | move peptide with small/shear moves to generate | | |
//   | | | conformational diversity                        | | |
//   | | +-------------------------------------------------+ | |
//   | +-----------------------------------------------------+ |
//   | +-Rotamer Trials Mover--------------------------------+ |
//   | | quick sidechain packing to find optimal rotamers    | |
//   | | before the next cycle                               | |
//   | +-----------------------------------------------------+ |
//   +---------------------------------------------------------+
//
// Design Minimization Phase
//   | +-Pack Rotamers Mover---------------------------------+ |
//   | | repack and design rotamers to explore sequence 	   | |
//   | | space  	                                           | |
//   | +-----------------------------------------------------+ |
//   | +-Minimization Mover----------------------------------+ |
//   | | energy minimize the current conformation            | |
//   | +-----------------------------------------------------+ |

//
// We begin by making one of four large pertubations to the structure. For the HP1 Chromo domain we will be
// modeling two loops and one termini (termini: 23-26, loop1: 45-54, loop2: 59-64). Loop modeling will be
// done using CCD. Termini pertubations will be done
// using small and shear moves. Peptide modeling will be done with small and shear moves. Docking will be done
// with small random ridged body purtubations of the peptide with respect to the protein. We will cycle
// through the pertubation phase repeatedly, then run the design minimization phase.

// tracer - used to replace cout
static util::Tracer TR("apps.demonstration");

// application specific options
namespace dddm {
// pert options
RealOptionKey const pert_mc_temp( "dddm::pert_mc_temp" );
RealOptionKey const pert_dock_rot_mag( "dddm::pert_dock_rot_mag" );
RealOptionKey const pert_dock_trans_mag( "dddm::pert_dock_trans_mag" );
RealOptionKey const pert_loop_temp( "dddm::pert_loop_temp" );
RealOptionKey const pert_pep_small_temp( "dddm::pert_pep_small_temp" );
RealOptionKey const pert_pep_small_H( "dddm::pert_pep_small_H" );
RealOptionKey const pert_pep_small_L( "dddm::pert_pep_small_L" );
RealOptionKey const pert_pep_small_E( "dddm::pert_pep_small_E" );
RealOptionKey const pert_pep_shear_temp( "dddm::pert_pep_shear_temp" );
RealOptionKey const pert_pep_shear_H( "dddm::pert_pep_shear_H" );
RealOptionKey const pert_pep_shear_L( "dddm::pert_pep_shear_L" );
RealOptionKey const pert_pep_shear_E( "dddm::pert_pep_shear_E" );
RealOptionKey const pert_ter_small_temp( "dddm::pert_ter_small_temp" );
RealOptionKey const pert_ter_small_H( "dddm::pert_ter_small_H" );
RealOptionKey const pert_ter_small_L( "dddm::pert_ter_small_L" );
RealOptionKey const pert_ter_small_E( "dddm::pert_ter_small_E" );
RealOptionKey const pert_ter_shear_temp( "dddm::pert_ter_shear_temp" );
RealOptionKey const pert_ter_shear_H( "dddm::pert_ter_shear_H" );
RealOptionKey const pert_ter_shear_L( "dddm::pert_ter_shear_L" );
RealOptionKey const pert_ter_shear_E( "dddm::pert_ter_shear_E" );
RealOptionKey const pert_pep_num_rep( "dddm::pert_pep_num_rep" );
RealOptionKey const pert_ter_num_rep( "dddm::pert_pep_num_rep" );
IntegerOptionKey const pert_num( "dddm::pert_num" );


// design options
RealOptionKey const desn_mc_temp( "dddm::desn_mc_temp" );

// common options
FileOptionKey const loops_file( "dddm:loops_file" );

}

void
setup_pert_foldtree( core::pose::Pose & pose, protocols::loops::Loops & myloops );


int
main( int argc, char* argv[] )
{
	/*********************************************************************************************************************
	  ____                                         ____  _       	  __  __
	 / ___|___  _ __ ___  _	__ ___ 	 ___  _	__		/ ___||	|_ _   _ / _|/ _|
	| |   /	_ \| '_	` _ \| '_ ` _ \	/ _ \| '_ \		\___ \|	__| | |	| |_| |_
	| |__| (_) | | | | | | | | | | | (_) | | | |	 ___) |	|_| |_|	|  _|  _|
	 \____\___/|_| |_| |_|_| |_| |_|\___/|_| |_|	|____/ \__|\__,_|_| |_|

	**********************************************************************************************************************/

	// add application specific options to options system
	// There are far more options here than you will realistically need for a program of this complexity - but this gives you an idea of how to fine-grain option-control everything
	option.add( dddm::pert_mc_temp, "The temperature to use for the pertubation phase of the DDDM protocol. Defaults to 0.8." ).def( 0.8 );
	option.add( dddm::pert_dock_rot_mag, "The rotation magnitude for the ridged body pertubation in the pertubation phase of the DDDM protocol. Defaults to 0.8." ).def( 1 );
	option.add( dddm::pert_dock_trans_mag, "The translation magnitude for the ridged body pertubation in the pertubation phase of the DDDM protocol. Defaults to 0.8." ).def( 0.5 );
	option.add( dddm::pert_loop_temp, "The temperature used to control the MC objec tin the kinematic mover" ).def( 0.4 );
	option.add( dddm::pert_pep_small_temp, "" ).def( 0.8 );
	option.add( dddm::pert_pep_shear_temp, "" ).def( 0.8 );
	option.add( dddm::pert_ter_small_temp, "" ).def( 0.8 );
	option.add( dddm::pert_ter_shear_temp, "" ).def( 0.8 );
	option.add( dddm::pert_pep_small_H, "" ).def( 2.0 );
	option.add( dddm::pert_pep_small_L, "" ).def( 2.0 );
	option.add( dddm::pert_pep_small_E, "" ).def( 2.0 );
	option.add( dddm::pert_pep_shear_H, "" ).def( 2.0 );
	option.add( dddm::pert_pep_shear_L, "" ).def( 2.0 );
	option.add( dddm::pert_pep_shear_E, "" ).def( 2.0 );
	option.add( dddm::pert_ter_small_H, "" ).def( 2.0 );
	option.add( dddm::pert_ter_small_L, "" ).def( 2.0 );
	option.add( dddm::pert_ter_small_E, "" ).def( 2.0 );
	option.add( dddm::pert_ter_shear_H, "" ).def( 2.0 );
	option.add( dddm::pert_ter_shear_L, "" ).def( 2.0 );
	option.add( dddm::pert_ter_shear_E, "" ).def( 2.0 );
	option.add( dddm::pert_pep_num_rep, "Number of small and shear iterations for the peptide" ).def( 100.0 );
	option.add( dddm::pert_ter_num_rep, "Number of small and shear iterations for the terminus" ).def( 100.0 );
	option.add( dddm::pert_num, "Number of iterations of perturbation loop per design" ).def(10);

	option.add( dddm::desn_mc_temp, "The temperature to use for the design/minimization phase of the DDDM protocol. Defaults to 0.8." ).def( 0.8 );
	option.add( dddm::loops_file, "File that holds the info about the loops in loop file format" );

	// init command line options
	//you MUST HAVE THIS CALL near the top of your main function, or your code will crash when you first access the command line options
	core::init(argc, argv);

	// create score function
	scoring::ScoreFunctionOP score_fxn( ScoreFunctionFactory::create_score_function( scoring::STANDARD_WTS, scoring::SCORE12_PATCH ) );

  // read pdb file
	pose::Pose pose;
	io::pdb::pose_from_pdb( pose, core::options::start_file() );

	// create Loops from file
	protocols::loops::Loops myloops;
	myloops.read_loop_file( option[ dddm::loops_file ].value() );
	myloops.auto_choose_cutpoints( pose );

	// get a fold tree suitable for docking and loop modeling (local helper function)
	setup_pert_foldtree( pose, myloops );

	/*********************************************************************************************************************
	 ____  	       	_      	       _       	   _   _                ____  _
	|  _ \ ___ _ __| |_ _  	_ _ __|	|__   __ _| |_(_) ___  _ __		 |  _ \| |__   __ _ ___  ___
	| |_) /	_ \ '__| __| | | | '__|	'_ \ / _` | __|	|/ _ \|	'_ \	 | |_) |	_ \ / _` / __|/ _ \
	|  __/ 	__/ |  | |_| |_| | |  |	|_) | (_| | |_|	| (_) |	| | |	 |  __/| | | | (_| \__ \  __/
	|_|   \___|_|  	\__|\__,_|_|  |_.__/ \__,_|\__|_|\___/|_| |_|	 |_|   |_| |_|\__,_|___/\___|

	**********************************************************************************************************************/

	// create a monte carlo object for the pertubation phase
	moves::MonteCarloOP pert_mc( new moves::MonteCarlo( pose, *score_fxn, option[ dddm::pert_mc_temp ].value() ) );

	/*********************************************************
	  ___	       	 _   _ 	       	   ___ 	    _
	 |   \ ___  __| |_(_)_ _  __ _  / __|	___| |_	_  _ _ __
	 | |)	/ _ \/ _| / / |	' \/ _`	| \__ \/ -_)  _| || | '_ \
	 |___/\___/\__|_\_\_|_||_\__,	| |___/\___|\__|\_,_| .__/
	     	       	       	   |___/       	       	    |_|
	**********************************************************/

	// create a rigid body mover to move the peptide around in the pocket
	moves::RigidBodyPerturbMoverOP pert_dock_rbpm( new moves::RigidBodyPerturbMover() );
	pert_dock_rbpm->rot_magnitude( option[ dddm::pert_dock_rot_mag].value() );
	pert_dock_rbpm->trans_magnitude( option[ dddm::pert_dock_trans_mag].value() );

	/*********************************************************
	  _  	       	       	 ___   	  _
	 | | 	 ___  ___ _ __ 	/ __| ___| |_ _	 _ _ __
	 | |__/ _ \/ _ \ '_ \	\__ \/ -_)  _| || | '_ \
	 |____\___/\___/ .__/	|___/\___|\__|\_,_| .__/
	     	       	 |_|   	       	       	  |_|
	**********************************************************/

	// create CCD-style loop closure
	//movemaps
	kinematics::MoveMapOP pert_loop1_mm( new kinematics::MoveMap() ), pert_loop2_mm( new kinematics::MoveMap() );
	for( core::Size i(myloops[1].start()); i<=myloops[1].stop(); ++i ){
		pert_loop1_mm->set_bb(i, true); //backbone mobile
		pert_loop1_mm->set( TorsionID(i, BB, omega_torsion), false ); //fixes omega angle
	}
	for( core::Size i(myloops[2].start()); i<=myloops[2].stop(); ++i ){
		pert_loop2_mm->set_bb(i, true); //backbone mobile
		pert_loop2_mm->set( TorsionID(i, BB, omega_torsion), false ); //fixes omega angle
	}

	//smallmovers
	core::Size const nmoves(1);
	MoverOP pert_loop1_small = new SmallMover(pert_loop1_mm, 0.8, nmoves);
	MoverOP pert_loop2_small = new SmallMover(pert_loop2_mm, 0.8, nmoves);

	//CCD movers
	MoverOP pert_loop1_ccd = new protocols::loops::CcdLoopClosureMover(myloops[1], pert_loop1_mm);
	MoverOP pert_loop2_ccd = new protocols::loops::CcdLoopClosureMover(myloops[2], pert_loop2_mm);

	// create sequence movers
	moves::SequenceMoverOP pert_loop1_ccd_sequence( new moves::SequenceMover() );
	pert_loop1_ccd_sequence->add_mover( pert_loop1_small );
	pert_loop1_ccd_sequence->add_mover( pert_loop1_ccd );

	moves::SequenceMoverOP pert_loop2_ccd_sequence( new moves::SequenceMover() );
	pert_loop2_ccd_sequence->add_mover( pert_loop2_small );
	pert_loop2_ccd_sequence->add_mover( pert_loop2_ccd );

	/*********************************************************
	  ___	       	 _   _ 	  _    	  ___  	   _
	 | _ \___ _ __| |_(_)__| |___	 / __| ___| |_ _  _ _ __
	 |  _/ -_) '_	\  _| /	_` / -_) \__ \/	-_)  _|	|| | '_	\
	 |_| \___| .__/\__|_\__,_\___| |___/\___|\__|\_,_| .__/
	     	   |_| 	       	       	       	       	   |_|
	**********************************************************/

	// get peptide start and end positions
	Size pep_start( pose.conformation().chain_begin( 2 ) ); Size pep_end( pose.total_residue() );

	// create movemap for peptide
	kinematics::MoveMapOP pert_pep_mm( new kinematics::MoveMap() );
	pert_pep_mm->set_bb_true_range(pep_start, pep_end);

	// create small and shear movers
	moves::SmallMoverOP pert_pep_small( new moves::SmallMover( pert_pep_mm, option[ dddm::pert_pep_small_temp ].value(), 1 ) );
	pert_pep_small->angle_max( 'H', option[ dddm::pert_pep_small_H ].value() );
	pert_pep_small->angle_max( 'L', option[ dddm::pert_pep_small_L ].value() );
	pert_pep_small->angle_max( 'E', option[ dddm::pert_pep_small_E ].value() );

	moves::ShearMoverOP pert_pep_shear( new moves::ShearMover( pert_pep_mm, option[ dddm::pert_pep_shear_temp ].value(), 1 ) );
	pert_pep_shear->angle_max( 'H', option[ dddm::pert_pep_shear_H ].value() );
	pert_pep_shear->angle_max( 'L', option[ dddm::pert_pep_shear_L ].value() );
	pert_pep_shear->angle_max( 'E', option[ dddm::pert_pep_shear_E ].value() );

	// create sequence mover
	moves::RandomMoverOP pert_pep_random( new moves::RandomMover() );
	pert_pep_random->add_mover( pert_pep_small, 1 );
	pert_pep_random->add_mover( pert_pep_shear, 1 );

	// create repeat mover
	moves::RepeatMoverOP pert_pep_repeat( new moves::RepeatMover( pert_pep_random, option[ dddm::pert_pep_num_rep ].value() ) );

	/*********************************************************
	  _____      	       _      _	  ___  	   _
	 |_  	_|__ _ _ _ __ (_)_ _ (_) / __| ___| |_ _  _ _ __
	   | |/ -_) '_| '  \|	| ' \| | \__ \/	-_)  _|	|| | '_	\
	   |_|\___|_|	|_|_|_|_|_||_|_| |___/\___|\__|\_,_| .__/
	     	       	       	       	       	       	   |_|
	**********************************************************/

	// get termini start and end positions
	Size ter_start( pose.conformation().chain_begin( 1 ) ); Size ter_end( pose.conformation().chain_begin( 1 ) + 5 );

	// create movemap for termini
	kinematics::MoveMapOP pert_ter_mm( new kinematics::MoveMap() );
	pert_ter_mm->set_bb_true_range(ter_start, ter_end);

	// create small and shear movers
	moves::SmallMoverOP pert_ter_small( new moves::SmallMover( pert_ter_mm, option[ dddm::pert_ter_small_temp ].value(), 1 ) );
	pert_ter_small->angle_max( 'H', option[ dddm::pert_ter_small_H ].value() );
	pert_ter_small->angle_max( 'L', option[ dddm::pert_ter_small_L ].value() );
	pert_ter_small->angle_max( 'E', option[ dddm::pert_ter_small_E ].value() );

 	moves::ShearMoverOP pert_ter_shear( new moves::ShearMover( pert_ter_mm,  option[ dddm::pert_ter_shear_temp ].value(), 1 ) );
	pert_ter_shear->angle_max( 'H', option[ dddm::pert_ter_shear_H ].value() );
	pert_ter_shear->angle_max( 'L', option[ dddm::pert_ter_shear_L ].value() );
	pert_ter_shear->angle_max( 'E', option[ dddm::pert_ter_shear_E ].value() );

	// create sequence mover
	moves::RandomMoverOP pert_ter_random( new moves::RandomMover() );
	pert_ter_random->add_mover( pert_ter_small, 1 );
	pert_ter_random->add_mover( pert_ter_shear, 1 );

	// create repeat mover
	moves::RepeatMoverOP pert_ter_repeat( new moves::RepeatMover( pert_ter_random, option[ dddm::pert_ter_num_rep ].value() ) );

	/******************************************************************************
	  ___	    _  	       	       	  _____	   _   	  _    	 ___   	  _
	 | _ \___| |_	__ _ _ __  ___ _ |_   _| _(_)__	_| |___	/ __| ___| |_ _	 _ _ __
	 |   / _ \  _/ _` | '	 \/ -_)	'_|| ||	'_| / _` | (_-<	\__ \/ -_)  _| || | '_ \
	 |_|_\___/\__\__,_|_|_|_\___|_|  |_||_| |_\__,_|_/__/	|___/\___|\__|\_,_| .__/
	     	       	       	       	       	       	       	       	       	  |_|
	*******************************************************************************/

	// create a task factory and tack operations
	TaskFactoryOP pert_tf(new TaskFactory());
	pert_tf->push_back( new InitializeFromCommandlineOperation() );

	ReadResfileOperationOP pert_rrop( new ReadResfileOperation() );
	pert_rrop->default_filename();
	pert_tf->push_back( pert_rrop );

	RestrictToRepackingOperationOP pert_rtrp( new RestrictToRepackingOperation() );
	pert_tf->push_back( pert_rtrp );

	TaskOperations::RestrictToInterfaceOperationOP pert_rtio( new TaskOperations::RestrictToInterfaceOperation(1, 2) ); //magic numbers: assume chains 1 and 2
	pert_tf->push_back( pert_rtio );

	// create a rotamer trials mover
	moves::RotamerTrialsMoverOP pert_rt(new moves::EnergyCutRotamerTrialsMover( score_fxn, pert_tf, pert_mc, 0.1 /*energycut*/ ) );

	/*********************************************************
	   ___       	       	       	    ___	     _
	  / __|___ _ __  _ __	 ___ _ _   / __| ___| |_ _  _ _	__
	 | (__/ _ \ '	 \| '  \/ _ \ '	\  \__ \/ -_)  _| || | '_ \
	  \___\___/_|_|_|_|_|_\___/_||_| |___/\___|\__|\_,_| .__/
	     	       	       	       	       	       	     |_|
	**********************************************************/

	// create a random mover to hold the docking, loop, termini, and peptide pertubation movers
	moves::RandomMoverOP pert_random( new moves::RandomMover() );
	pert_random->add_mover( pert_dock_rbpm, 1 );
	pert_random->add_mover( pert_pep_repeat, 1 );
	pert_random->add_mover( pert_ter_repeat, 1 );
	pert_random->add_mover( pert_loop1_ccd_sequence, 1 );
	pert_random->add_mover( pert_loop2_ccd_sequence, 1 );

	// create a sequence move to hold random and rotamer trials movers
	moves::SequenceMoverOP pert_sequence( new moves::SequenceMover() );
	pert_sequence->add_mover( pert_random );
	pert_sequence->add_mover( pert_rt );

	// create a TrialMover for the pertubation
	moves::TrialMoverOP pert_trial( new moves::TrialMover( pert_sequence, pert_mc ) );

	/*********************************************************************************************************************
	 ____  	       	 _		             __  __ _	         ____  _
	|  _ \ 	___  ___(_) __ _ _ __	    |  \/	 (_)_ __    |  _ \| |__	  __ _ ___  ___
	| | | |/ _ \/ __| |/ _`	| '_ \	  | |\/| | | '_	\   | |_) | '_ \ / _` /	__|/ _ \
	| |_| |	 __/\__	\ | (_|	| | | |	  | |  | | | | | |  |  __/| | |	| (_| \__ \  __/
	|____/ \___||___/_|\__,	|_| |_|	  |_|  |_|_|_| |_|  |_|	  |_| |_|\__,_|___/\___|
	       	       	   |___/
	**********************************************************************************************************************/
	/*********************************************************
	  ___	       	_      	      ___      _
	 |   \ ___ __(_)__ _ _ _   / __| ___|	|_ _  _	_ __
	 | |)	/ -_|_-< / _` |	' \  \__ \/ -_)	 _| || | '_ \
	 |___/\___/__/_\__, |_||_| |___/\___|\__|\_,_| .__/
	     	       	 |___/ 	       	       	       |_|
	**********************************************************/

	// create a tack factory and task operations
	TaskFactoryOP desn_tf( new TaskFactory() );
	desn_tf->push_back( new InitializeFromCommandlineOperation() );

	ReadResfileOperationOP desn_rrop = new ReadResfileOperation();
	desn_rrop->default_filename();
	desn_tf->push_back( desn_rrop );

	desn_tf->push_back( pert_rtio ); //not bothering to construct it a second time

	// create a pack rotamers mover
	moves::PackRotamersMoverOP desn_pr( new moves::PackRotamersMover() );
	desn_pr->task_factory( desn_tf );
	desn_pr->score_function( score_fxn );

	/*********************************************************
	  __ 	__ _   	  _    	  _    	     ___      _
	 |  \/  (_)_ _ (_)_ __ (_)______  / __| ___| |_ _  _ _ __
	 | |\/| | | '	\| | ' 	\| |_ /	-_) \__	\/ -_) 	_| || |	'_ \
	 |_| 	|_|_|_||_|_|_|_|_|_/__\___| |___/\___|\__|\_,_|	.__/
	     	       	       	       	       	       	      |_|
	**********************************************************/

	// create move map for minimization
	kinematics::MoveMapOP desn_mm( new kinematics::MoveMap() );
	desn_mm->set_bb( true );
	desn_mm->set_chi( true );
	desn_mm->set_jump( 1, true );

	// create minimization mover
	moves::MinMoverOP desn_min( new moves::MinMover( desn_mm, score_fxn, option[ OptionKeys::run::min_type ].value(), 0.01,	true ) );

	//definitely want sidechain minimization here
	using protocols::moves::TaskAwareMinMoverOP;
	using protocols::moves::TaskAwareMinMover;
	TaskAwareMinMoverOP desn_ta_min = new TaskAwareMinMover( desn_min, desn_tf );

	/*********************************************************
	   ___       	       	       	    ___	     _
	  / __|___ _ __  _ __	 ___ _ _   / __| ___| |_ _  _ _	__
	 | (__/ _ \ '	 \| '  \/ _ \ '	\  \__ \/ -_)  _| || | '_ \
	  \___\___/_|_|_|_|_|_\___/_||_| |___/\___|\__|\_,_| .__/
	     	       	       	       	       	       	     |_|
	**********************************************************/

	// create a sequence mover to hold pack rotamers and minimization movers
	moves::SequenceMoverOP desn_sequence( new moves::SequenceMover() );
	desn_sequence->add_mover( desn_pr );
	desn_sequence->add_mover( desn_ta_min );

	/*********************************************************************************************************************
	 __  __	      _	       	_
	|  \/  | __ _(_)_ __   | |    ___   ___	 _ __
	| |\/| |/ _` | | '_ \  | |   / _ \ / _ \| '_ \
	| |  | | (_| | | | | | | |__| (_) | (_)	| |_) |
	|_|  |_|\__,_|_|_| |_| |_____\___/ \___/| .__/
	       	       	       	       	       	|_|
	**********************************************************************************************************************/

	//Note you probably do not want a main loop like this in your protocol - it would be better to have most of the code preceding this as a mover's apply() function, so here you would have the mover/monte carlo statements but not the outer for loop and pose IO
	core::pose::Pose const copy_pose(pose);
	for ( Size i = 1; i <= Size( option[ out::nstruct ].value() ) ; ++i ) {
		pose = copy_pose;
		pert_mc->reset(pose);
		std::stringstream filename_pdb;
		filename_pdb << "design_" << i << ".pdb";

		// pert loop
		for( Size j = 1; j <= Size( option[ dddm::pert_num ].value() ); ++j ) {
			std::cout << "PERTURB: " << i << " / "  << j << std::endl;
			pert_trial->apply( pose );
		}

		pert_mc->recover_low( pose );

		// design
		std::cout << "DESIGN: " << i << std::endl;
		desn_sequence->apply( pose );
		pose.dump_scored_pdb( filename_pdb.str(), *score_fxn );
	}

	return 0;
}

// this only works for two chains and assumes the protein is first and the peptide is second
// also is assumes that the protein peptide jump is between the two loop jumps, and that there are two loops!
// inspired by protocols/docking/DockingProtocol.cc
void
setup_pert_foldtree( core::pose::Pose & pose, protocols::loops::Loops & myloops )
{
	using namespace kinematics;
	using namespace protocols::loops;

	// get current fold tree
	FoldTree f( pose.fold_tree() );
	f.clear();

	/*********************************************************
	  ___	       	 _     	  _
	 |   \ ___  __| |__  _ | |_  _ _ __  _ __ ___
	 | |)	/ _ \/ _| / / |	|| | ||	| '  \|	'_ (_-<
	 |___/\___/\__|_\_\  \__/ \_,_|_|_|_|	.__/__/
	     	       	       	       	      |_|
	**********************************************************/

	// get the start and end for both chains
	Size pro_start( pose.conformation().chain_begin( 1 ) );
	Size pro_end( pose.conformation().chain_end( 1 ) );
	Size pep_start( pose.conformation().chain_begin( 2 ) );
	Size pep_end( pose.conformation().chain_end( 2 ) );

	// get jump positions based on the center of mass of the chains
	Size dock_jump_pos_pro( geometry::residue_center_of_mass( pose, pro_start, pro_end ) );
	Size dock_jump_pos_pep( geometry::residue_center_of_mass( pose, pep_start, pep_end ) );

	// build fold tree
	Size jump_index( f.num_jump() + 1 );
	//-1 is a magic number for PEPTIDE EDGE.  There is a constant defined with the fold tree that should have been used here.
	f.add_edge( pro_start, dock_jump_pos_pro, -1 );
	f.add_edge( dock_jump_pos_pro, pro_end, -1 );
	f.add_edge( pep_start, dock_jump_pos_pep, -1);
	f.add_edge( dock_jump_pos_pep, pep_end, -1 );
	f.add_edge( dock_jump_pos_pro, dock_jump_pos_pep, jump_index );

	/**********************************************************
	  _  	       	       	    _
	 | | 	 ___  ___ _ __ 	 _ | |_	 _ _ __	 _ __ ___
	 | |__/ _ \/ _ \ '_ \	| || | || | '  \| '_ (_-<
	 |____\___/\___/ .__/	 \__/ \_,_|_|_|_| .__/__/
	     	       	 |_|   	       	       	|_|
	************************************************************/

	// add loop jumps and cutpoints to working foldtree
	protocols::loops::Loop loop1( myloops[1] );
	protocols::loops::Loop loop2( myloops[2] );

	f.new_jump( loop1.start() - 1, loop1.stop() + 1, loop1.cut() );
	f.new_jump( loop2.start() - 1, loop2.stop() + 1, loop2.cut() );


	// set pose foldtree to foldtree we just created
	f.reorder(1);
	f.check_fold_tree();
	assert( f.check_fold_tree() );

	std::cout << "AFTER: " << f << std::endl;

	pose.fold_tree( f );
}
